package App::Manoc::Controller::APIv1::DHCPReservation;
#ABSTRACT: Catalyst Controller for DHCP APIs

use Moose;

our $VERSION = '2.99.2'; ##TRIAL VERSION

use namespace::autoclean;

BEGIN { extends 'App::Manoc::Controller::APIv1' }


sub reservation_base : Chained('deserialize') PathPart('dhcp/reservation') CaptureArgs(0) {
    my ( $self, $c ) = @_;
    $c->stash( resultset => $c->model('ManocDB::DHCPReservation') );
}


sub reservation_post : Chained('reservation_base') PathPart('') POST {
    my ( $self, $c ) = @_;

    $c->stash(
        api_validate => {
            type  => 'hash',
            items => {
                server => {
                    type     => 'scalar',
                    required => 1,
                },
                reservations => {
                    type     => 'array',
                    required => 1,
                },
            }
        }
    );
    $c->forward('validate') or return;

    my $req_data = $c->stash->{api_request_data};

    my $server_name = $req_data->{server};
    my $server      = $c->model('ManocDB::DHCPServer')->find('name');
    if ( !$server ) {
        push @{ $c->stash->{api_field_errors} }, 'Unknown server';
        return;
    }
    my $rs        = $c->stash->{resultset};
    my $records   = $req_data->{reservations};
    my $n_created = 0;

    $c->schema->txn_do(
        sub {
            $server->reservations->update( on_server => 0 );

            foreach my $r (@$records) {
                my $macaddr = $r->{macaddr}                                    or next;
                my $ipaddr  = App::Manoc::IPAddress::IPv4->new( $r->{ipaddr} ) or next;
                my $status  = $r->{server};
                my $hostname = $r->{hostname};
                my $name     = $r->{name};

                $rs->update_or_create(
                    {
                        macaddr   => $macaddr,
                        ipaddr    => $ipaddr,
                        hostname  => $hostname,
                        name      => $name,
                        server    => $server,
                        on_server => 1,
                    }
                );
                $n_created++;
            }
        }
    );
    my $data = { message => "created $n_created entries", };

    $c->stash( api_response_data => $data );
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::APIv1::DHCPReservation - Catalyst Controller for DHCP APIs

=head1 VERSION

version 2.99.2

=head1 ACTIONS

=head2 reservation_base

Base action for reservation.

=head2 reservation_post

POST api/v1/dhcp/reservation

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
