package App::Manoc::Controller::Building;
#ABSTRACT: Building Controller

use Moose;

our $VERSION = '2.99.2'; ##TRIAL VERSION

use namespace::autoclean;

use App::Manoc::Form::Building;

BEGIN { extends 'Catalyst::Controller'; }
with
    "App::Manoc::ControllerRole::CommonCRUD" => { -excludes => 'delete_object', },
    "App::Manoc::ControllerRole::JSONView"   => { -excludes => 'get_json_object', };

__PACKAGE__->config(
    # define PathPart
    action => {
        setup => {
            PathPart => 'building',
        }
    },
    class                   => 'ManocDB::Building',
    form_class              => 'App::Manoc::Form::Building',
    enable_permission_check => 1,
    view_object_perm        => undef,
    json_columns            => [ 'id', 'name', 'description', 'label' ],
    object_list_options => { prefetch => 'racks' },
);


sub delete_object {
    my ( $self, $c ) = @_;
    my $building = $c->stash->{'object'};

    if ( $building->warehouses->count ) {
        $c->flash( error_msg => 'Building has associated warehouses and cannot be deleted.' );
        return;
    }

    if ( $building->racks->count ) {
        $c->flash( error_msg => 'Building has associated racks and cannot be deleted.' );
        return;
    }

    return $building->delete;
}


sub get_json_object {
    my ( $self, $c, $building ) = @_;

    my $r = $self->prepare_json_object( $c, $building );
    $r->{racks} = [ map +{ id => $_->id, name => $_->name }, $building->racks ];
    return $r;
}

__PACKAGE__->meta->make_immutable;

1;
# Local Variables:
# mode: cperl
# indent-tabs-mode: nil
# cperl-indent-level: 4
# cperl-indent-parens-as-block: t
# End:

__END__

=pod

=head1 NAME

App::Manoc::Controller::Building - Building Controller

=head1 VERSION

version 2.99.2

=head1 METHODS

=head2 delete_object

Override default implementation to warn when building has associated racks or
warehouses.

=head2 get_json_object

Override to add rack information

=head1 AUTHORS

=over 4

=item *

Gabriele Mambrini <gmambro@cpan.org>

=item *

Enrico Liguori

=back

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2017 by Gabriele Mambrini.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
