package RedHat::Kickstart;

our $VERSION = '0.3';

#####################################################################
# RedHat::Kickstart.pm
# by Patrick Devine
#
# This software is covered under the same terms as Perl itself.
#
# WARNING:
#
# This software is no where near finished and lots of stuff will
# probably change before it is officially released (as 1.0).
#


use RedHat::SysConfig;
@ISA = qw/ RedHat::SysConfig /;


#####################################################################
# method:	kickstart
# function:	creates a scalar containing all of the data necessary
#		for to create a kickstart file

sub kickstart {
  my $self	= shift;
  my $version	= shift;

  my $buf;
  my $settings	= $self->{settings};

  $buf .= "# This file was autogenerated by RedHat::Kickstart.pm\n";
  $buf .= "# Use at your own risk.\n";

  $buf .= "\n# Installation Method\n";
  if( $settings->{inst_type}->{inst_type} eq 'nfs' ) {
    my( $host, $dir );
    ( $host, $dir ) = split ':', $settings->{inst_type}->{nfsserver}
      if $settings->{inst_type}->{nfsserver};
    $buf .= "nfs --server $host --dir $dir\n";
  } elsif( $settings->{inst_type}->{inst_type} eq 'cdrom' ) {
    $buf .= "cdrom\n";
  }

  if( exists $settings->{rootpw} ) {
    $buf .= "\n# root password\nrootpw ";
    $buf .= '--iscrypted '
      if $settings->{rootpw}->{iscrypted};
    $buf .= "$settings->{rootpw}->{rootpw}\n";
  }


  if( exists $settings->{auth}->{useshadow} or
      exists $settings->{auth}->{enablemd5} or
      exists $settings->{auth}->{enablenis} ) {
 
    $buf .= "\n# Autconfig\nauth ";
    if( exists $settings->{auth}->{enablenis} ) {
      $buf .= "--enablenis --nisdomain $settings->{auth}->{nisdomain} ";
      $buf .= "--nisserver $settings->{auth}->{nisserver} ";
    }
    $buf .= '--useshadow ' if exists $settings->{auth}->{useshadow};
    $buf .= '--enablemd5 ' if exists $settings->{auth}->{enablemd5};
    if( exists $settings->{auth}->{enableldap} and $version >= 700 ) {
      $buf .= '--enableldap ';
      $buf .= '--enableldapauth '
        if $settings->{auth}->{enableldapauth};
      $buf .= "--ldapserver=$settings->{auth}->{ldapserver} "
        if $settings->{auth}->{ldapserver};
      $buf .= "--ldapbasedn=$settings->{auth}->{ldapbasedn} "
        if $settings->{auth}->{ldapbasedn};
    }
    if( $settings->{auth}->{enablekrb5} and $version >= 700 ) {
      $buf .= '--enablekrb5 ';
      $buf .= "--krb5realm $settings->{auth}->{krb5realm} "
        if $settings->{auth}->{krb5realm};
      $buf .= '--krb5kdc ';
      $buf .= join ',', map { $settings->{auth}->{"krb5kdc$_"} } 0 .. 3;
      $buf .= ' ';
      $buf .= "--krb5adminserver $settings->{auth}->{krb5kdc0} "
       if $settings->{auth}->{krb5kdc0};
    }
    if( $settings->{auth}->{enablehesiod} and $version >= 700 ) {
      $buf .= '--enablehesiod ';
      $buf .= "--hesiodlhs $settings->{auth}->{hesiodlhs} "
        if $settings->{auth}->{hesiodlhs};
      $buf .= "--hesiodrhs $settings->{auth}->{hesiodrhs} "
        if $settings->{auth}->{hesiodrhs};
    }
    $buf .= "\n";
 
  }

  if( exists $self->{settings}->{device} ) {
    $buf .= "\n# Extra Devices\n";
    for( @{ $self->{settings}->{device} } ) {
      next unless $_->{type} eq 'scsi' or $_->{type} eq 'eth';
      $buf .= "device $_->{type} $_->{module} ";
      $buf .= "--opts \"$_->{opts}\""
        if exists $_->{opts};
      $buf .= "\n";
    }
  }

  if( exists $settings->{driverdisk} and $version >= 623 ) {
    $buf .= "\n# Driver Disk\n driverdisk ";
    $buf .= $settings->{driverdisk}->{partition};
    $buf .= " --type $settings->{driverdisk}->{fs}\n";
  }

  if( exists $settings->{lilo} ) {
    $buf .= "\n# LILO\nlilo ";
    $buf .= "--append \"$settings->{lilo}->{append}\" "
      if exists $settings->{lilo}->{append};
    $buf .= "--location $settings->{lilo}->{location}\n"
      if $settings->{lilo}->{location};
    $buf .= "\n";
  }
 
  $buf .= "\n# LILO Check\nlilocheck\n"
    if exists $settings->{lilocheck} and $version >= 623;

  if( exists $settings->{timezone}->{timezone} ) {
    $buf .= "\n# Timezone\ntimezone ";
    $buf .= "--utc "
      if exists $settings->{timezone}->{utc};
    $buf .= "$settings->{timezone}->{timezone}\n";
  }

  $buf .= "\n# X Windowing System\n";
 
  if( exists $settings->{xconfig} ) {
    $buf .= "xconfig ";
    $buf .= "--noprobe \"$settings->{xconfig}->{noprobe}\" "
      if exists $settings->{xconfig}->{noprobe} and $version >= 623;
    $buf .= "--server \"$settings->{xconfig}->{server}\" "
      if exists $settings->{xconfig}->{server};
    $buf .= "--card \"$settings->{xconfig}->{card}\" "
      if exists $settings->{xconfig}->{card};
    $buf .= "--monitor \"$settings->{xconfig}->{monitor}\" "
      if exists $settings->{xconfig}->{monitor};
    $buf .= "--hsync \"$settings->{xconfig}->{hsync}\" "
      if exists $settings->{xconfig}->{hsync};
    $buf .= "--vsync \"$settings->{xconfig}->{vsync}\" "
      if exists $settings->{xconfig}->{vsync};
    $buf .= "--startxonboot \"$settings->{xconfig}->{startxonboot}\" "
      if exists $settings->{xconfig}->{startxonboot} and $version >= 623;
    $buf .= "\n";
  } else {
    $buf .= "skipx\n";
  }

  $buf .= "\n# Skip X\nskipx\n"
    if exists $settings->{skipx}->{skipx} and $version >= 623;
 
  $buf .= "\n# Install or Upgrade";
  $buf .= "\ninstall\n"
    unless exists $settings->{install}->{install} and
      $settings->{install}->{install} eq 'upgrade';

  if( exists $settings->{network} ) {

    $settings->{network}->{bootproto} = $settings->{network}->{network}
      if $settings->{network}->{network};

    $buf .= "\n# Network install type\nnetwork --bootproto ";
    if( $settings->{network}->{bootproto} ne 'static' ) {
      $buf .= "$settings->{network}->{bootproto}\n";
    } else {
      $buf .= "static ";
      $buf .= "--ip $settings->{network}->{ip} "
        if exists $settings->{network}->{ip};
      $buf .= "--netmask $settings->{network}->{netmask} "
        if exists $settings->{network}->{netmask};
      $buf .= "--gateway $settings->{network}->{gateway} "
        if exists $settings->{network}->{gateway};
      $buf .= "--nameserver $settings->{network}->{nameserver} "
        if exists $settings->{network}->{nameserver};
      $buf .= "--hostname $settings->{network}->{hostname} "
        if exists $settings->{network}->{hostname};
      $buf .= "\n";
    }
  }
 
  $buf .= "\n# Language\nlang $settings->{lang}->{lang}\n"
    if exists $settings->{lang}->{lang};
 
  $buf .= "\n# Keyboard\nkeyboard $settings->{keyboard}->{keyboard}\n"
    if exists $settings->{keyboard}->{keyboard};
 
  $buf .= "\n# Zero mbr\nzerombr yes\n"
    if exists $settings->{zerombr}->{zerombr};

  if( $settings->{mouse} ) {
    $buf .= "\n# Mouse\nmouse ";
    if( $version >= 623 ) {
      $buf .= "--device $settings->{mouse}->{device} "
        if exists $settings->{mouse}->{device};
      $buf .= "--emulthree "
        if exists $settings->{mouse}->{emulthree};
    } else {
      $buf .= "--kickstart ";
    }
    $buf .= "$settings->{mouse}->{mouse}\n";
  }
 
  $buf .= "\n# Reboot after install\nreboot\n"
    if exists $settings->{reboot}->{reboot} and $version > 623;
 
  if( $settings->{install} ne 'upgrade' ) {
 
    $buf .= "\n# Clear Partitions\nclearpart " .
	    "--$settings->{clearpart}->{clearpart}\n"
      if $settings->{clearpart}->{clearpart};
 
    $buf .= "\n# Partitioning\n";
    for( @{ $self->{settings}->{partition} } ) {
      next unless $_->{size};

      $_->{dir} = 'swap'
	if $_->{type} eq 'swap';

      $buf .= "part $_->{dir} --size $_->{size} ";
      $buf .= "--grow "
	if exists $_->{grow};
      $buf .= "--ondisk $_->{ondisk}"
	if exists $_->{ondisk} and $_->{ondisk} ne 'logical';
      $buf .= "\n";
    }

    if( exists $self->{settings}->{raid} ) {
      $buf .= "\n# Software RAID\n";
      for( @{ $self->{settings}->{raid} } ) {
	$buf .= "raid $_->{dir} ";
	$buf .= "--level $_->{level} "
	  if exists $_->{level};
	$buf .= "--device $_->{device} "
	  if exists $_->{device};

	$buf .= join ' ', @{ $_->{partitions} };
	$buf .= "\n";
      }
    }

    $buf .= "\n# Packages\n%packages\n";

    for( keys %{ $self->{settings}->{package} } ) {
      $buf .= "$_\n"
        if $self->{settings}->{package}->{$_} eq 'on';
    }
 
    $buf .= "\n# Post install\n%post\n";
 
    for( keys %{ $self->{settings}->{service} } ) {
      $buf .= ( $self->{settings}->{service}->{$_} eq 'on' ?
		"/sbin/chkconfig --add $_\n" :
		"/sbin/chkconfig --del $_\n" );
    }
  }

  if( exists $self->{settings}->{pre} ) {
    $buf .= "\n\%pre\n";
    if( ref( $self->{settings}->{pre} ) eq 'ARRAY' ) {
      $buf .= join "\n", @{ $self->{settings}->{pre}->{pre} };
    } else {
      $buf .= $self->{settings}->{pre}->{pre};
    }
  }

  if( exists $self->{settings}->{post} ) {
    $buf .= "\n\%post\n";
    if( ref( $self->{settings}->{post} ) eq 'ARRAY' ) {
      $buf .= join "\n", @{ $self->{settings}->{post}->{post} };
    } else {
      $buf .= $self->{settings}->{post}->{post};
    }
  }

  return \$buf;

}


1;

__END__

=head1 NAME

Kickstart - generate RedHat Kickstart files.

=head1 DESCRIPTION

Kickstart uses the B<RedHat::SysConfig.pm> module to allow a perl script to make
method calls which generate a RedHat Kickstart file.

=item auth { KEY => VALUE, ... }

=item auth

The auth method allows for the configuration of various authentication
parameters such as MD5 and Shadow passwords, and NIS, LDAP, Kerberos and
Hesiod configuration.

    enablemd5		use md5 password encryption
    enablenis		enable NIS support
    nisdomain		specify a domain name for NIS
    nisserver		specify a server to use with NIS
    useshadow		use shadow passwords
    enableldap		enable LDAP user authentication
    ldapserver		specify an LDAP server
    ldapbasedn		specify the base LDAP domain name
    enablekrb5		enable Kerberos 5 authentication
    krb5realm		specify the Kerberos realm
    krb5kdc		specify a list of KDC values
    krb5adminserver	specify the master KDC
    enablehesiod	enable Hesiod support for directory lookup
    hesiodlhs		specify heriod lhs (left-hand side)
    hesiodrhs		specify heriod rhs (right-hand side)

=item clearpart all | linux

=item clearpart

Specify which partitions on a disk to remove in order to set up the correct
partitions.

    all		removes all partitions
    linux	remove ext2, swap raid partitions

=item device

=item device

Not currently working.

=item driverdisk

=item driverdisk { KEY => VALUE, ... }

Specify a disk which will be copied to the root directory of the installed
system.

    partition	specify the partition containing the driver disk
    type	specify the file system type of the partition

=item install

=item install

[text to come]

=item inst_type

=item inst_type

[text to come]

=item keyboard VALUE

=item keyboard

The keyboard method is used to specify the type of keyboard which is
attached to a system.

Valid values for x86 architecture include:

C<azerty>, C<be-latin1>, C<be2-latin1>, C<fr-latin0>, C<fr-latin1>, C<fr-pc>,
C<fr>, C<wangbe>, C<ANSI-dvorak>, C<dvorak-l>, C<dvorak-r>, C<dvorak>,
C<pc-dvorak-latin1>, C<tr_f-latin5>, C<trf>, C<bg>, C<cf>, C<cz-lat2-prog>,
C<cz-lat2>, C<defkeymap>, C<defkeymap_V1.0>, C<dk-latin1>, C<dk. emacs>,
C<emacs2>, C<es>, C<fi-latin1>, C<fi>, C<gr-pc>, C<gr>, C<hebrew>, C<hu101>,
C<is-latin1>, C<it-ibm>, C<it>, C<it2>, C<jp106>, C<la-latin1>, C<lt>,
C<lt.l4>, C<nl>, C<no-latin1>, C<no>, C<pc110>, C<pl>, C<pt-latin1>,
C<pt-old>, C<ro>, C<ru-cp1251>, C<ru-ms>, C<ru-yawerty>, C<ru>, C<ru1>, C<ru2>,
C<ru_win>, C<se-latin1>, C<sk-prog-qwerty>, C<sk-prog>, C<sk-qwerty>,
C<tr_q-latin5>, C<tralt>, C<trf>, C<trq>, C<ua>, C<uk>, C<us>, C<croat>,
C<cz-us-qwertz>, C<de-latin1-nodeadkeys>, C<de-latin1>, C<de>, C<fr_CH-latin1>,
C<fr_CH>, C<hu>, C<sg-latin1-lk450>, C<sg-latin1>, C<sg>, C<sk-prog-qwertz>,
C<sk-qwertz>, C<slovene>

and for Sparc:

C<sun-pl-altgraph>, C<sun-pl>, C<sundvorak>, C<sunkeymap>, C<sunt4-es>,
C<sunt4-no-latin1>, C<sunt5-cz-us>, C<sunt5-de-latin1>, C<sunt5-es>,
C<sunt5-fi-latin1>, C<sunt5-fr-latin1>, C<sunt5-ru>, C<sunt5-uk>, C<sunt5-us-cz>

=item lang

=item lang

The lang method is used to specify the type of language which will be used
during the installation.

Valid languages include:

C<cs_CZ>, C<en_US>, C<fr_FR>, C<de_DE>, C<hu_HU>, C<is_IS>, C<id_ID>, C<it_IT>,
C<ja_JP.ujis>, C<no_NO>, C<pl_PL>, C<ro_RO>, C<sk_SK>, C<sl_SI>, C<es_ES>,
C<ru_RU.KOI8-R>, C<uk_UA>

Use C<en_US> to specify US english.

=item lilo { KEY => VALUE, ... }

=item lilo

  append	specify arguments to be passed to the kernel
  linear	specify that LILO to work in linear mode
  location	specify the locatation where LILO will be written

=item lilocheck 1

=item lilocheck

If set, LILO will not perform an installation onto a system which has LILO
written in the master boot record of the first hard drive.

=item mouse VALUE

=item mouse

The mouse method specifies which mouse type should be configured for the
system.

Valid mouse types include:

C<alpsps/2>, C<ascii>, C<asciips/2>, C<atibm>, C<generic>, C<generic3>,
C<genericps/2>, C<generic3ps/2>, C<geniusnm>, C<geniusnmps/2>,
C<geniusnsps/2>, C<thinking>, C<thinkingps/2>, C<logitech>,
C<logitechcc>, C<logibm>, C<logimman>, C<logimmanps/2>, C<logimman+>,
C<logimman+ps/2>, C<microsoft>, C<msnew>, C<msintelli>, C<msintellips/2>,
C<msbm>, C<mousesystems>, C<mmseries>, C<mmhittab>, C<sun>, C<none>

=item network

=item network

[text to come]

=item packages

=item packages

[text to come]

=item part

=item part

    asprimary
    bytes-per-inode
    dir
    grow
    maxsize
    noformat
    ondisk
    onpart
    onprimary
    size
    type

[text to come]

=item post

=item post

[text to come]

=item pre

=item pre

[text to come]

=item raid

=item raid

    device	specify the name of the device (eg. md0, ...)
    dir		specify the mount point of the raid device
    level	specify the raid level to use (0, 1 or 5)
    partitions	an array of partitions to use for the device

=item reboot 1

=item reboot

Use the reboot method to specify that the system should be rebooted after
completing the installation.

=item rootpw { iscrypted => 1, rootpw => '..' } | VALUE

=item rootpw

The rootpw method can either be called by passing a hash reference to it, or
with a scalar value.  If the root password being specified is already
encrypted, you should call rootpw with a hash reference and set C<iscrypted>
to be on.

  iscrypted	specify that the password is encrypted
  rootpw	specify the root password

=item skipx 1

=item skipx

It is possible to skip X11 configuration entirely by using the skipx method.

=item timezone { utc => 1, timezone => '..' } | VALUE

=item timezone

The timezone method can be called by passing a hash reference to it, or with
a scalar value.  If the hardware clock is set to Greenwich Mean Time, you
should call the method with a hash reference and set C<utc> to be on.

  timezone	specify the timezone the system will be located in
  utc		specify that the hardware clock is set to UTC

=item upgrade

=item upgrade

[text to come]

=item xconfig

=item xconfig

The xconfig method can be used to configure the X Windowing System.

    card		specify which type of card to use
    defaultdesktop	specify to use kde or gnome
    hsync		specify the horizontal sync freq.
    monitor		specify what type of monitor to use
    noprobe		specify to not probe the monitor
    startxonboot	specify using run level 5 (instead of 3)
    vsync		specify the vertical sync freq.

=item zerombr 1

=item zerombr

Specify that the master boot record of the primary drive should be initialized.


=head1 AUTHOR INFORMATION

Written by Patrick Devine, 2001.

=cut

