package CPAN::Mini::Webserver::Templates;
use strict;
use warnings;
use MIME::Base64;
use Template::Declare::Tags;
use base 'Template::Declare';

private template 'header' => sub {
    my ( $self, $title ) = @_;

    head {
        title {$title};
        link {
            attr {
                rel   => 'stylesheet',
                href  => '/static/css/screen.css',
                type  => 'text/css',
                media => 'screen, projection'
            }
        };
        link {
            attr {
                rel   => 'stylesheet',
                href  => '/static/css/print.css',
                type  => 'text/css',
                media => 'print'
            }
        };
        outs_raw
            '<!--[if IE]><link rel="stylesheet" href="/static/css/ie.css" type="text/css" media="screen, projection"><![endif]-->';
        link {
            attr {
                rel  => 'icon',
                href => '/static/images/favicon.png',
                type => 'image/png',
            }
        };
        link {
            attr {
                rel   => 'search',
                href  => '/static/xml/opensearch.xml',
                type  => 'application/opensearchdescription+xml',
                title => 'minicpan search',
            }
        };

        meta { attr { generator => 'CPAN::Mini::Webserver' } };
    }
};

private template 'footer' => sub {
    my $self    = shift;
    my $version = $CPAN::Mini::Webserver::VERSION;

    div {
        attr { id => "footer" };
        small {
            "Generated by CPAN::Mini::Webserver $version";
        };
    }
};

private template 'author_link' => sub {
    my ( $self, $author ) = @_;
    a {
        attr { href => '/~' . lc( $author->pauseid ) . '/' };
        $author->name;
    };
};

private template 'distribution_link' => sub {
    my ( $self, $distribution ) = @_;
    a {
        attr {    href => '/~'
                . lc( $distribution->cpanid ) . '/'
                . $distribution->distvname
                . '/' };
        $distribution->distvname;
    };
};

private template 'package_link' => sub {
    my ( $self, $package ) = @_;
    my $distribution = $package->distribution;
    a {
        attr {    href => '/package/'
                . lc( $distribution->cpanid ) . '/'
                . $distribution->distvname . '/'
                . $package->package
                . '/' };
        $package->package;
    };
};

private template distribution_file => sub {
    my ( $self, $pauseid, $distvname, $filename ) = (@_);

    my $display_filename
        = ( $filename =~ /^$distvname\/(.*)$/ )
        ? $1
        : $filename;
    my $href
        = ( $filename =~ /\.(pm|PL|pod)$/ )
        ? "/~$pauseid/$distvname/$filename"
        : "/raw/~$pauseid/$distvname/$filename";
    row {
        cell {
            a {
                attr { href => $href };
                span {
                    $display_filename;
                };
            };
        };
    };
};

private template 'searchbar' => sub {
    my $self = shift;
    my $q    = shift;

    table {
        row {
            form {
                attr { method => 'get', action => '/search/' };
                cell {
                    attr { class => 'searchbar' };
                    outs_raw
                        q|<a href="/"><img src="/static/images/logo.png"></a>|;
                };
                cell {
                    attr { class => 'searchbar' };
                    input {
                        { attr { type => 'text', name => 'q', value => $q } };
                    };
                    input {
                        {
                            attr {
                                type  => 'submit',
                                value => 'Search Mini CPAN'
                                }
                        };
                    };
                };
            };
        };
    };
};

private template 'search_results' => sub {
    my ( $self, $arguments ) = @_;
    my $parse_cpan_authors = $arguments->{parse_cpan_authors};
    my $q                  = $arguments->{q};
    my @authors            = @{ $arguments->{authors} };
    my @distributions      = @{ $arguments->{distributions} };
    my @packages           = @{ $arguments->{packages} };
    if ( @authors + @distributions + @packages ) {
        outs_raw '<table>';
        foreach my $author (@authors) {

            row {
                cell {
                    show( 'author_link', $author );

                };
            };
        }

        foreach my $distribution (@distributions) {
            row {
                cell {
                    show( 'distribution_link', $distribution );
                    outs ' by ';
                    show( 'author_link',
                        $parse_cpan_authors->author( $distribution->cpanid )
                    );

                };
            };
        }
        foreach my $package (@packages) {
            row {
                cell {
                    show( 'package_link', $package );
                    outs ' by ';
                    show(
                        'author_link',
                        $parse_cpan_authors->author(
                            $package->distribution->cpanid
                        )
                    );
                };
            };
        }
        outs_raw '</table>';
    } else {
        p {'No results found.'};
    }
};

template 'index' => sub {
    my $self = shift;

    html {
        attr { xmlns => 'http://www.w3.org/1999/xhtml' };
        div {
            attr { class => 'container' };
            div {
                attr { class => 'span-24' };
                show( 'header', 'Index' );
                body {
                    show('searchbar');
                    h1 {'Index'};
                    p {'Welcome to CPAN::Mini::Webserver. Start searching!'};
                };
                show('footer');
            };
        };
    };
};

template '404' => sub {
    my ( $self, $arguments ) = @_;
    my $q = $arguments->{q};
    html {
        attr { xmlns => 'http://www.w3.org/1999/xhtml' };
        div {
            attr { class => 'container' };
            div {
                attr { class => 'span-24' };
                show( 'header', 'File not found' );
                body {
                    show( 'searchbar', $q );
                    h1 {'Sorry. I couldn\'t find the page you wanted.'};
                    p {
                        "Unfortunately, the page you were looking for doesn't exist. Perhaps a quick search for $q will turn up what you were looking for:";
                    };
                };
                h2 {
                    outs "Search for ";
                    outs_raw '&#147;';
                    outs $q;
                    outs_raw '&#148;';
                };
                show( 'search_results', $arguments );
                show('footer');
            };
        };
    };
};

template 'search' => sub {
    my ( $self, $arguments ) = @_;
    my $q = $arguments->{q};
    html {
        show( 'header', "Search for `$q'" );
        body {
            div {
                attr { class => 'container' };
                div {
                    attr { class => 'span-24' };
                    show( 'searchbar', $q );
                    h1 {
                        outs "Search for ";
                        outs_raw '&#147;';
                        outs $q;
                        outs_raw '&#148;';
                    };
                    show( 'search_results', $arguments );
                    show('footer');
                }
            };
        }
    };
};

private template 'authorinfo' => sub {
    my ( $self, $author ) = @_;

    my $pauseid = $author->pauseid;
    my $email   = $author->email;
    my $url     = $author->can('homepage') ? $author->homepage : undef;
    my $prefix
        = 'id' . '/'
        . substr( $pauseid, 0, 1 ) . '/'
        . substr( $pauseid, 0, 2 ) . '/'
        . $pauseid;

    h2 {"Links"};
    ul {
        li {
            a {
                attr { href => "http://backpan.perl.org/authors/$prefix" };
                'BackPAN';
            };
        }
        li {
            a {
                attr { href => "mailto:$email" };
                $email;
            };
        }
        li {
            a {
                attr { href => $url };
                $url;
            };
        }
        li {
            a {
                attr {
                    href => "http://cpantesters.perl.org/author/$pauseid.html" };
                'CPANTesters';
            };
        }
        li {
            a {
                attr { href =>
                        "http://bbbike.radzeit.de/~slaven/cpantestersmatrix.cgi?author=$pauseid"
                };
                'Test Matrix';
            };
        }
    }
};

template 'author' => sub {
    my ( $self, $arguments ) = @_;
    my $author        = $arguments->{author};
    my $pauseid       = $arguments->{pauseid};
    my $distvname     = $arguments->{distvname};
    my @distributions = @{ $arguments->{distributions} };
    my $dates         = $arguments->{dates};

    html {
        show( 'header', $author->name );
        body {
            div {
                attr { class => 'container' };
                div {
                    attr { class => 'span-24 last' };
                    show('searchbar');
                    h1 { show( 'author_link', $author ) };
                }
                div {
                    attr { class => 'span-18 last' };
                    outs_raw '<table>';
                    foreach my $distribution (@distributions) {
                        row {
                            cell {
                                show( 'distribution_link', $distribution );

                            };
                            cell {
                                outs $dates->{ $distribution->distvname };
                            };
                        };
                    }
                    outs_raw '</table>';
                }
                div {
                    attr { class => 'span-6 last' };
                    show( 'authorinfo', $author );
                };

                div {
                    attr { class => 'span-24 last' };
                    show('footer');
                };

            };
        };

    }
};

private template 'dependencies' => sub {
    my ( $self, $meta, $pcp ) = @_;

    div {
        attr { class => 'dependencies' };
        h2 {'Dependencies'};
        ul {
            foreach
                my $deptype (qw(requires build_requires configure_requires))
            {
                if ( defined $meta->{$deptype} ) {
                    foreach my $package ( sort keys %{ $meta->{$deptype} } ) {
                        next if $package eq 'perl';
                        my $d = $pcp->package($package)->distribution;
                        next unless $d;
                        my $distvname = $d->distvname;
                        my $author    = $d->cpanid;
                        li {
                            a {
                                attr { href => "/~$author/$distvname/" };
                                $package;
                            };
                            if ( $deptype =~ /(.*?)_/ ) {
                                outs " ($1 requirement)";
                            }
                        }
                    }
                }
            }
        }
    }
};

private template 'metadata' => sub {
    my ( $self, $meta ) = @_;

    h2 {'Metadata'};
    div {
        attr { class => 'metadata' };
        dl {
            foreach my $key ( qw(abstract license), 'release date' ) {
                if ( defined $meta->{$key} ) {
                    dt { ucfirst $key; };
                    if ( defined $meta->{resources}->{$key} ) {
                        a {
                            attr {
                                href => delete $meta->{resources}->{$key} };
                            $meta->{$key};
                        };
                    } else {
                        dd { $meta->{$key} };
                    }
                }
            }
            foreach my $datum ( keys %{ $meta->{resources} } ) {
                dt { ucfirst $datum; }
                dd {
                    a {
                        attr { href => $meta->{resources}->{$datum}; };
                        $meta->{resources}->{$datum};
                    }

                }
            }
        }
    };
};

private template 'download' => sub {
    my ( $self, $author, $distribution ) = @_;
    my $distvname = $distribution->distvname;
    h2 {'Download'};
    div {
        a {
            attr { href => '/download/~' . $author->pauseid . "/$distvname" };
            $distribution->filename;
        }
    };
};

private template 'install' => sub {
    my ( $self, $author, $distribution, $filenames ) = @_;
    my $distvname = $distribution->distvname;

    # Check whether we have the module/distribution installed
    # And display the status
    # Just fudge:
    # * If we have lib/*.pm, that's a contained module
    my @modules = map {
        m![^/]*/lib/(.*?)\.pm!;
        $_ = $1;
        s!/!::!g;
        $_
        } grep {
        m![^/]*/lib/.*?\.pm$!
        } @{$filenames};

    my $installed_version = Module::InstalledVersion->new( $modules[0] );

    my $msg    = "Not installed on this Perl";
    my $action = 'Install';
    if ( $installed_version->{version} ) {
        $msg = sprintf 'You have version %s installed.',
            $installed_version->{version};
        if ( $installed_version->{version} lt $distribution->version ) {
            $action = 'Update';
        } elsif ( $installed_version->{version} eq $distribution->version ) {
            $action = 'Reinstall';
        } else {
            $action = 'Downgrade';
        }
    }

    h2 {'Install'};
    div {
        attr { class => 'install' };
        div { attr { 'class' => "install-message" }; $msg };
        form {
            attr { class => 'install-link' } attr { method => 'PUT' };
            attr {
                      action => '/install/~'
                    . lc( $distribution->cpanid ) . '/'
                    . $distribution->distvname . '/'
                    . $distribution->filename;
            };
            button {$action} $action;
        };
    };
};

private template 'dist_links' => sub {
    my ( $self, $distribution ) = @_;
    my $distname = $distribution->dist;

    h2 {'Links'};
    ul {
        li {
            outs "Test ";
            a {
                attr { href =>
                        "http://bbbike.radzeit.de/~slaven/cpantestersmatrix.cgi?dist=$distname"
                };
                "matrix";
            };
            outs " and ";
            a {
                attr { href =>
                        "http://cpantesters.perl.org/show/$distname.html" };
                "reports";
            };
        }
        li {
            a {
                attr { href =>
                        "http://rt.cpan.org/NoAuth/Bugs.html?Dist=$distname"
                };
                "RT";
            };
            outs " (or via ";
            a {
                attr { href => "mailto:bug-$distname\@rt.cpan.org" } "email";
            };
            outs ")";
        }
        li {
            a {
                attr { href => "http://annocpan.org/dist/$distname" };
                "AnnoCPAN";
            }
        }
        li {
            a {
                attr { href => "http://cpanratings.perl.org/d/$distname" };
                "CPAN Ratings";
            }
        }
    }
};

template 'filelist' => sub {
    my ( $self, $pauseid, $distvname, $label, $filenames ) = @_;
    h2 {$label};
    outs_raw '<table>';
    foreach my $filename (@$filenames) {
        show(
            distribution_file => $pauseid,
            $distvname, $filename
        );
    }
    outs_raw '</table>';
};

template 'distribution' => sub {
    my ( $self, $arguments ) = @_;
    my $author       = $arguments->{author};
    my $pauseid      = $arguments->{pauseid};
    my $distvname    = $arguments->{distvname};
    my $distribution = $arguments->{distribution};
    my @filenames    = @{ $arguments->{filenames} };
    my $meta         = $arguments->{meta};
    my $pcp          = $arguments->{pcp};
    html {
        show( 'header', $author->name . ' > ' . $distvname );
        body {
            div {
                attr { class => 'container' };
                div {
                    attr { class => 'span-24 last' };
                    show('searchbar');
                    h1 {
                        show( 'author_link', $author );
                        outs ' > ';
                        show( 'distribution_link', $distribution );
                    };
                }
                div {
                    attr { class => 'span-18 last' };

                    #                    outs_raw '<table>';
                    my ( @code, @test, @other );
                    foreach (@filenames) {
                        if ( m{(?:/bin/|\.p(?:m|l)$)} and not m{/inc/} ) {
                            push @code, $_;
                        } elsif (/\.t$/) {
                            push @test, $_;
                        } else {
                            push @other, $_;
                        }
                    }
                    show( 'filelist', $pauseid, $distvname, 'Code',  \@code );
                    show( 'filelist', $pauseid, $distvname, 'Tests', \@test );
                    show( 'filelist', $pauseid, $distvname, 'Other',
                        \@other );

                   #                    foreach my $filename (@filenames) {
                   #                        show(
                   #                            distribution_file => $pauseid,
                   #                            $distvname, $filename
                   #                        );
                   #                    }
                   #                    outs_raw '</table>';
                };
                div {
                    attr { class => 'span-6 last' };
                    show( 'metadata',     $meta );
                    show( 'dependencies', $meta, $pcp );
                    show( 'download',     $author, $distribution );
                    show( 'install',    $author, $distribution, \@filenames );
                    show( 'dist_links', $distribution );
                };
                div {
                    attr { class => 'span-24 last' };
                    show('footer');
                };

            }

        };

    }
};

template 'file' => sub {
    my ( $self, $arguments ) = @_;
    my $author       = $arguments->{author};
    my $distribution = $arguments->{distribution};
    my $filename     = $arguments->{filename};
    my $pauseid      = $arguments->{pauseid};
    my $distvname    = $arguments->{distvname};

    my $file     = $arguments->{filename};
    my $contents = $arguments->{contents};
    my $html     = $arguments->{html};
    html {
        show( 'header',
            $author->name . ' > ' . $distvname . ' > ' . $filename );
        body {
            div {
                attr { class => 'container' };
                div {
                    attr { class => 'span-24' };
                    show('searchbar');
                    h1 {
                        show( 'author_link', $author );
                        outs ' > ';
                        show( 'distribution_link', $distribution );
                        outs ' > ';
                        outs $filename;
                    };

                    a {
                        attr {
                            href => "/raw/~$pauseid/$distvname/$filename" };
                        "See raw file";
                    };
                    if ($html) {
                        div {
                            attr { id => "pod" };
                            outs_raw $html;
                        };
                    } else {
                        pre {$contents};
                    }
                    show('footer');
                };

            };
        };

    }
};

template 'raw' => sub {
    my ( $self, $arguments ) = @_;
    my $author       = $arguments->{author};
    my $distribution = $arguments->{distribution};
    my $filename     = $arguments->{filename};
    my $pauseid      = $arguments->{pauseid};
    my $distvname    = $arguments->{distvname};
    my $contents     = $arguments->{contents};
    my $html         = $arguments->{html};
    html {
        show( 'header',
            $author->name . ' > ' . $distvname . ' > ' . $filename );
        body {
            div {
                attr { class => 'container' };
                div {
                    attr { class => 'span-24' };
                    show('searchbar');
                    h1 {
                        show( 'author_link', $author );
                        outs ' > ';
                        show( 'distribution_link', $distribution );
                        outs ' > ';
                        outs $filename;
                    };
                    if ($html) {
                        div {
                            attr { id => "code" };
                            code {
                                outs_raw $html;
                            };
                        };
                    } else {
                        pre {$contents};
                    }
                    div {
                        attr { class => 'download-link' };
                        a {
                            attr {    href => '/download/~'
                                    . $author->pauseid
                                    . "/$distvname/$filename" };
                            "Download as plain text";
                        };
                    };
                    show('footer');
                };

            };
        };

    }
};

template 'opensearch' => sub {
    my $self = shift;
    outs_raw q|<?xml version="1.0" encoding="UTF-8"?>
<OpenSearchDescription xmlns="http://a9.com/-/spec/opensearch/1.1/">
<ShortName>minicpan_webserver</ShortName>
<Description>Search minicpan</Description>
<InputEncoding>UTF-8</InputEncoding>
<Image width="16" height="16">data:image/png,%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%10%00%00%00%10%08%03%00%00%00(-%0FS%00%00%00%01sRGB%00%AE%CE%1C%E9%00%00%003PLTE8%00%00%05%08%04%16%18%15%1E%1F%1D!%22%20%26(%26%2C-%2B130%3B%3D%3AFHELMKXZWegdxyw%84%86%83%9E%A0%9D%CC%CE%CBjq%F6r%00%00%00%01tRNS%00%40%E6%D8f%00%00%00lIDAT%18%D3u%8FY%0E%C20%0C%05%BD%AF)%ED%FDO%0B%85%10%15%04%EF%C7%1A%7B%2C%D9%00%7Fr%C4W%A3u%EB%2B%EFn%E3sAnr1%8E%E11%D4rq%1Bn%9E%CC%8B%15%C5%01%14u%B2%A0%3EmA9K1Z%BD%5C%C6%87%18%B4%18%8A0%A0Q%2B%C3%CC%232%9D%CE%19%E1%3B%3C%E6%E6%CA%BC%C4%A5%BB%C2%84%FC%D7%DBw%7BS%02%E3Ki%23G%00%00%00%00IEND%AEB%60%82</Image>
<Url type="text/html" method="get" template="http://localhost:2963/search/?q={searchTerms}"/>
</OpenSearchDescription>
|;
};

template 'css_screen' => sub {
    my $self = shift;
    outs_raw
        q|/* -----------------------------------------------------------------------

   Blueprint CSS Framework 0.7.1
   http://blueprintcss.googlecode.com

   * Copyright (c) 2007-2008. See LICENSE for more info.
   * See README for instructions on how to use Blueprint.
   * For credits and origins, see AUTHORS.
   * This is a compressed file. See the sources in the 'src' directory.

----------------------------------------------------------------------- */

/* reset.css */
html, body, div, span, object, iframe, h1, h2, h3, h4, h5, h6, p, blockquote, pre, a, abbr, acronym, address, code, del, dfn, em, img, q, dl, dt, dd, ol, ul, li, fieldset, form, label, legend, table, caption, tbody, tfoot, thead, tr, th, td {margin:0;padding:0;border:0;font-weight:inherit;font-style:inherit;font-size:100%;font-family:inherit;vertical-align:baseline;}
body {line-height:1.5;}
table {border-collapse:separate;border-spacing:0;}
caption, th, td {text-align:left;font-weight:normal;}
table, td, th {vertical-align:middle;}
blockquote:before, blockquote:after, q:before, q:after {content:"";}
blockquote, q {quotes:"" "";}
a img {border:none;}

/* typography.css */
body {font-size:75%;color:#222;background:#fff;font-family:"Helvetica Neue", Helvetica, Arial, sans-serif;}
h1, h2, h3, h4, h5, h6 {font-weight:normal;color:#111;}
h1 {font-size:3em;line-height:1;margin-bottom:0.5em;}
h2 {font-size:2em;margin-bottom:0.75em;}
h3 {font-size:1.5em;line-height:1;margin-bottom:1em;}
h4 {font-size:1.2em;line-height:1.25;margin-bottom:1.25em;height:1.25em;}
h5 {font-size:1em;font-weight:bold;margin-bottom:1.5em;}
h6 {font-size:1em;font-weight:bold;}
h1 img, h2 img, h3 img, h4 img, h5 img, h6 img {margin:0;}
p {margin:0 0 1.5em;}
p img {float:left;margin:1.5em 1.5em 1.5em 0;padding:0;}
p img.right {float:right;margin:1.5em 0 1.5em 1.5em;}
a:focus, a:hover {color:#000;}
a {color:#009;text-decoration:underline;}
blockquote {margin:1.5em;color:#666;font-style:italic;}
strong {font-weight:bold;}
em, dfn {font-style:italic;}
dfn {font-weight:bold;}
sup, sub {line-height:0;}
abbr, acronym {border-bottom:1px dotted #666;}
address {margin:0 0 1.5em;font-style:italic;}
del {color:#666;}
pre, code {margin:1.5em 0;white-space:pre;}
pre, code, tt {font:1em 'andale mono', 'lucida console', monospace;line-height:1.5;}
li ul, li ol {margin:0 1.5em;}
ul, ol {margin:0 1.5em 1.5em 1.5em;}
ul {list-style-type:disc;}
ol {list-style-type:decimal;}
dl {margin:0 0 1.5em 0;}
dl dt {font-weight:bold;}
dd {margin-left:1.5em;}
table {margin-bottom:1.4em;width:100%;}
th {font-weight:bold;background:#C3D9FF;}
th, td {padding:4px 10px 4px 5px;}
tr.even td {background:#E5ECF9;}
tfoot {font-style:italic;}
caption {background:#eee;}
.small {font-size:.8em;margin-bottom:1.875em;line-height:1.875em;}
.large {font-size:1.2em;line-height:2.5em;margin-bottom:1.25em;}
.hide {display:none;}
.quiet {color:#666;}
.loud {color:#000;}
.highlight {background:#ff0;}
.added {background:#060;color:#fff;}
.removed {background:#900;color:#fff;}
.first {margin-left:0;padding-left:0;}
.last {margin-right:0;padding-right:0;}
.top {margin-top:0;padding-top:0;}
.bottom {margin-bottom:0;padding-bottom:0;}

/* grid.css */
.container {width:950px;margin:0 auto;}
.showgrid {background:url(src/grid.png);}
body {margin:1.5em 0;}
div.span-1, div.span-2, div.span-3, div.span-4, div.span-5, div.span-6, div.span-7, div.span-8, div.span-9, div.span-10, div.span-11, div.span-12, div.span-13, div.span-14, div.span-15, div.span-16, div.span-17, div.span-18, div.span-19, div.span-20, div.span-21, div.span-22, div.span-23, div.span-24 {float:left;margin-right:10px;}
div.last {margin-right:0;}
.span-1 {width:30px;}
.span-2 {width:70px;}
.span-3 {width:110px;}
.span-4 {width:150px;}
.span-5 {width:190px;}
.span-6 {width:230px;}
.span-7 {width:270px;}
.span-8 {width:310px;}
.span-9 {width:350px;}
.span-10 {width:390px;}
.span-11 {width:430px;}
.span-12 {width:470px;}
.span-13 {width:510px;}
.span-14 {width:550px;}
.span-15 {width:590px;}
.span-16 {width:630px;}
.span-17 {width:670px;}
.span-18 {width:710px;}
.span-19 {width:750px;}
.span-20 {width:790px;}
.span-21 {width:830px;}
.span-22 {width:870px;}
.span-23 {width:910px;}
.span-24, div.span-24 {width:950px;margin:0;}
.append-1 {padding-right:40px;}
.append-2 {padding-right:80px;}
.append-3 {padding-right:120px;}
.append-4 {padding-right:160px;}
.append-5 {padding-right:200px;}
.append-6 {padding-right:240px;}
.append-7 {padding-right:280px;}
.append-8 {padding-right:320px;}
.append-9 {padding-right:360px;}
.append-10 {padding-right:400px;}
.append-11 {padding-right:440px;}
.append-12 {padding-right:480px;}
.append-13 {padding-right:520px;}
.append-14 {padding-right:560px;}
.append-15 {padding-right:600px;}
.append-16 {padding-right:640px;}
.append-17 {padding-right:680px;}
.append-18 {padding-right:720px;}
.append-19 {padding-right:760px;}
.append-20 {padding-right:800px;}
.append-21 {padding-right:840px;}
.append-22 {padding-right:880px;}
.append-23 {padding-right:920px;}
.prepend-1 {padding-left:40px;}
.prepend-2 {padding-left:80px;}
.prepend-3 {padding-left:120px;}
.prepend-4 {padding-left:160px;}
.prepend-5 {padding-left:200px;}
.prepend-6 {padding-left:240px;}
.prepend-7 {padding-left:280px;}
.prepend-8 {padding-left:320px;}
.prepend-9 {padding-left:360px;}
.prepend-10 {padding-left:400px;}
.prepend-11 {padding-left:440px;}
.prepend-12 {padding-left:480px;}
.prepend-13 {padding-left:520px;}
.prepend-14 {padding-left:560px;}
.prepend-15 {padding-left:600px;}
.prepend-16 {padding-left:640px;}
.prepend-17 {padding-left:680px;}
.prepend-18 {padding-left:720px;}
.prepend-19 {padding-left:760px;}
.prepend-20 {padding-left:800px;}
.prepend-21 {padding-left:840px;}
.prepend-22 {padding-left:880px;}
.prepend-23 {padding-left:920px;}
div.border {padding-right:4px;margin-right:5px;border-right:1px solid #eee;}
div.colborder {padding-right:24px;margin-right:25px;border-right:1px solid #eee;}
.pull-1 {margin-left:-40px;}
.pull-2 {margin-left:-80px;}
.pull-3 {margin-left:-120px;}
.pull-4 {margin-left:-160px;}
.pull-5 {margin-left:-200px;}
.pull-6 {margin-left:-240px;}
.pull-7 {margin-left:-280px;}
.pull-8 {margin-left:-320px;}
.pull-9 {margin-left:-360px;}
.pull-10 {margin-left:-400px;}
.pull-11 {margin-left:-440px;}
.pull-12 {margin-left:-480px;}
.pull-13 {margin-left:-520px;}
.pull-14 {margin-left:-560px;}
.pull-15 {margin-left:-600px;}
.pull-16 {margin-left:-640px;}
.pull-17 {margin-left:-680px;}
.pull-18 {margin-left:-720px;}
.pull-19 {margin-left:-760px;}
.pull-20 {margin-left:-800px;}
.pull-21 {margin-left:-840px;}
.pull-22 {margin-left:-880px;}
.pull-23 {margin-left:-920px;}
.pull-24 {margin-left:-960px;}
.pull-1, .pull-2, .pull-3, .pull-4, .pull-5, .pull-6, .pull-7, .pull-8, .pull-9, .pull-10, .pull-11, .pull-12, .pull-13, .pull-14, .pull-15, .pull-16, .pull-17, .pull-18, .pull-19, .pull-20, .pull-21, .pull-22, .pull-23, .pull-24 {float:left;position:relative;}
.push-1 {margin:0 -40px 1.5em 40px;}
.push-2 {margin:0 -80px 1.5em 80px;}
.push-3 {margin:0 -120px 1.5em 120px;}
.push-4 {margin:0 -160px 1.5em 160px;}
.push-5 {margin:0 -200px 1.5em 200px;}
.push-6 {margin:0 -240px 1.5em 240px;}
.push-7 {margin:0 -280px 1.5em 280px;}
.push-8 {margin:0 -320px 1.5em 320px;}
.push-9 {margin:0 -360px 1.5em 360px;}
.push-10 {margin:0 -400px 1.5em 400px;}
.push-11 {margin:0 -440px 1.5em 440px;}
.push-12 {margin:0 -480px 1.5em 480px;}
.push-13 {margin:0 -520px 1.5em 520px;}
.push-14 {margin:0 -560px 1.5em 560px;}
.push-15 {margin:0 -600px 1.5em 600px;}
.push-16 {margin:0 -640px 1.5em 640px;}
.push-17 {margin:0 -680px 1.5em 680px;}
.push-18 {margin:0 -720px 1.5em 720px;}
.push-19 {margin:0 -760px 1.5em 760px;}
.push-20 {margin:0 -800px 1.5em 800px;}
.push-21 {margin:0 -840px 1.5em 840px;}
.push-22 {margin:0 -880px 1.5em 880px;}
.push-23 {margin:0 -920px 1.5em 920px;}
.push-24 {margin:0 -960px 1.5em 960px;}
.push-1, .push-2, .push-3, .push-4, .push-5, .push-6, .push-7, .push-8, .push-9, .push-10, .push-11, .push-12, .push-13, .push-14, .push-15, .push-16, .push-17, .push-18, .push-19, .push-20, .push-21, .push-22, .push-23, .push-24 {float:right;position:relative;}
.box {padding:1.5em;margin-bottom:1.5em;background:#E5ECF9;}
hr {background:#ddd;color:#ddd;clear:both;float:none;width:100%;height:.1em;margin:0 0 1.45em;border:none;}
hr.space {background:#fff;color:#fff;}
.clearfix:after, .container:after {content:".";display:block;height:0;clear:both;visibility:hidden;}
.clearfix, .container {display:inline-block;}
* html .clearfix, * html .container {height:1%;}
.clearfix, .container {display:block;}
.clear {clear:both;}

/* forms.css */
label {font-weight:bold;}
fieldset {padding:1.4em;margin:0 0 1.5em 0;border:1px solid #ccc;}
legend {font-weight:bold;font-size:1.2em;}
input.text, input.title, textarea, select {margin:0.5em 0;border:1px solid #bbb;}
input.text:focus, input.title:focus, textarea:focus, select:focus {border:1px solid #666;}
input.text, input.title {width:300px;padding:5px;}
input.title {font-size:1.5em;}
textarea {width:390px;height:250px;padding:5px;}
.error, .notice, .success {padding:.8em;margin-bottom:1em;border:2px solid #ddd;}
.error {background:#FBE3E4;color:#8a1f11;border-color:#FBC2C4;}
.notice {background:#FFF6BF;color:#514721;border-color:#FFD324;}
.success {background:#E6EFC2;color:#264409;border-color:#C6D880;}
.error a {color:#8a1f11;}
.notice a {color:#514721;}
.success a {color:#264409;}

/* /home/acme/hg/CPAN-Mini-Webserver/root/static/css/my-screen.css */
h1 {font-size:2em;clear:both;margin-top:10px;}
h2 {font-size:1.7em;clear:both;}
h3 {font-size:1.4em;clear:both;}
body {background:#ffffff;font-size:100%;font-family:Georgia, "Times New Roman", Times, serif;}
pre, code, tt {font-size:80%;}
td.searchbar {vertical-align:middle;}
div#searchbar {min-height:10em;display:table-cell;vertical-align:middle;}
#code {font-size:120%;font-family:monospace;padding:10px 10px 10px 10px;}
#eval {font-family:monospace;border-width:1px;border-style:solid solid solid solid;border-color:#ccc;padding:5px 5px 5px 5px;}
.line_number {color:#aaaaaa;}
.comment {color:#228B22;}
.symbol {color:#00688B;}
.word {color:#8B008B;font-weight:bold;}
.structure {color:#000000;}
.number {color:#B452CD;}
.single {color:#CD5555;}
.double {color:#CD5555;}

/* buttons */
a.button, button {display:block;float:left;margin:0 0.583em 0.667em 0;padding:5px 10px 5px 7px;border:1px solid #dedede;border-top:1px solid #eee;border-left:1px solid #eee;background-color:#f5f5f5;font-family:"Lucida Grande", Tahoma, Arial, Verdana, sans-serif;font-size:100%;line-height:130%;text-decoration:none;font-weight:bold;color:#565656;cursor:pointer;}
button {width:auto;overflow:visible;padding:4px 10px 3px 7px;}
button[type] {padding:4px 10px 4px 7px;line-height:17px;}
*:first-child+html button[type] {padding:4px 10px 3px 7px;}
button img, a.button img {margin:0 3px -3px 0 !important;padding:0;border:none;width:16px;height:16px;float:none;}
button:hover, a.button:hover {background-color:#dff4ff;border:1px solid #c2e1ef;color:#336699;}
a.button:active {background-color:#6299c5;border:1px solid #6299c5;color:#fff;}
body .positive {color:#529214;}
a.positive:hover, button.positive:hover {background-color:#E6EFC2;border:1px solid #C6D880;color:#529214;}
a.positive:active {background-color:#529214;border:1px solid #529214;color:#fff;}
body .negative {color:#d12f19;}
a.negative:hover, button.negative:hover {background:#fbe3e4;border:1px solid #fbc2c4;color:#d12f19;}
a.negative:active {background-color:#d12f19;border:1px solid #d12f19;color:#fff;}
|;
};

template 'css_print' => sub {
    my $self = shift;
    outs_raw
        q|/* -----------------------------------------------------------------------

   Blueprint CSS Framework 0.7.1
   http://blueprintcss.googlecode.com

   * Copyright (c) 2007-2008. See LICENSE for more info.
   * See README for instructions on how to use Blueprint.
   * For credits and origins, see AUTHORS.
   * This is a compressed file. See the sources in the 'src' directory.

----------------------------------------------------------------------- */

/* print.css */
body {line-height:1.5;font-family:"Helvetica Neue", Helvetica, Arial, sans-serif;color:#000;background:none;font-size:10pt;}
.container {background:none;}
hr {background:#ccc;color:#ccc;width:100%;height:2px;margin:2em 0;padding:0;border:none;}
hr.space {background:#fff;color:#fff;}
h1, h2, h3, h4, h5, h6 {font-family:"Helvetica Neue", Arial, "Lucida Grande", sans-serif;}
code {font:.9em "Courier New", Monaco, Courier, monospace;}
img {float:left;margin:1.5em 1.5em 1.5em 0;}
a img {border:none;}
p img.top {margin-top:0;}
blockquote {margin:1.5em;padding:1em;font-style:italic;font-size:.9em;}
.small {font-size:.9em;}
.large {font-size:1.1em;}
.quiet {color:#999;}
.hide {display:none;}
a:link, a:visited {background:transparent;font-weight:700;text-decoration:underline;}
a:link:after, a:visited:after {content:" (" attr(href) ") ";font-size:90%;}
|;
};

template 'css_ie' => sub {
    my $self = shift;
    outs_raw
        q|/* -----------------------------------------------------------------------

   Blueprint CSS Framework 0.7.1
   http://blueprintcss.googlecode.com

   * Copyright (c) 2007-2008. See LICENSE for more info.
   * See README for instructions on how to use Blueprint.
   * For credits and origins, see AUTHORS.
   * This is a compressed file. See the sources in the 'src' directory.

----------------------------------------------------------------------- */

/* ie.css */
body {text-align:center;}
.container {text-align:left;}
* html .column {overflow-x:hidden;}
* html legend {margin:-18px -8px 16px 0;padding:0;}
ol {margin-left:2em;}
sup {vertical-align:text-top;}
sub {vertical-align:text-bottom;}
html>body p code {*white-space:normal;}
hr {margin:-8px auto 11px;}
|;
};

template 'images_logo' => sub {
    my $self = shift;
    my $encoded
        = q|iVBORw0KGgoAAAANSUhEUgAAAFIAAAAYCAMAAABeMVqjAAAAAXNSR0IArs4c6QAAApFQTFRFAgIC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|;
    outs_raw decode_base64($encoded);
};

template 'images_favicon' => sub {
    my $self = shift;
    my $encoded
        = q|iVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAMAAAAoLQ9TAAAAAXNSR0IArs4c6QAAADNQTFRFOAAA
BQgEFhgVHh8dISIgJigmLC0rMTMwOz06RkhFTE1LWFpXZWdkeHl3hIaDnqCdzM7LanH2cgAAAAF0
Uk5TAEDm2GYAAABsSURBVBjTdY9ZDsIwDAW9rynt/U8LhRAVBO/HGnss2QB/csRXo3XrK+9u43NB
bnIxjuEx1HJxG26ezIsVxQEUdbKgPm1BOUsxWr1cxocYtBiKMKBRK8PMIzKdzhnhOzzm5sq8xKW7
woT819t3e1MC40tpI0cAAAAASUVORK5CYII=
|;
    outs_raw decode_base64($encoded);
};

__END__

=head1 NAME

CPAN::Mini::Webserver::Templates - Templates for CPAN::Mini::Webserver

=head1 DESCRIPTION

This module holds the templates, CSS and images for 
CPAN::Mini::Webserver.

=head1 AUTHOR

Leon Brocard <acme@astray.com>

=head1 COPYRIGHT

Copyright (C) 2008, Leon Brocard.

This module is free software; you can redistribute it or 
modify it under the same terms as Perl itself.
