/*
  xhzview.c
  
  Copyright (C) 1989, 1990	Ricky Yeung (yeung@june.cs.washington.edu)
  
  A simple program to display mixed 8-bit or 16-bit characters
  on X window.  It is intended for viewing Chinese code files
  (e.g. GB or BIG5).

  It requires Mark Lillibridge's wsimple.c (bundled with xfd source)
  to compile.
     cc -o xhzview wsimple.c xhzview.c -lX11

  History:
     09/26/89	RY First working version.
     10/05/89	RY No longer requires cclwrap filter.
                Can read directly from stdin.
		Some minor bugs fixed.
     08/26/90	Version 2.0
     		Eliminate the -f option
		Eliminate tempfile if not read from stdin
		More browsing control
		Use line ends instead of sequential reading
		Allow resize of window
		Help facility
		Did some cleanup

  This program is free for general use.
*/

#include	<stdio.h>
#include	<X11/Xlib.h>
#include	<X11/Xutil.h>
#include	<X11/Xatom.h>

#define	min(a,b)	(a<b) ? a : b
#define	max(a,b)	(a>b) ? a : b

#define TITLE_DEFAULT "xhzview 2.0"
#define DEFX_DEFAULT 5  /* default window pop-up location */
#define DEFY_DEFAULT 5
#define RESIZE_X_INC 1
#define RESIZE_Y_INC 1
#define MIN_X_SIZE 1
#define MIN_Y_SIZE 1

#include        "wsimple.h"

#define true 1
#define false 0

#define LINELEN 3000
#define NAMELEN 80

#define	NLINES	100	/* amount to realloc for line ends */

char cFONT[NAMELEN]="beijing24";   /* default hanzi font */
char eFONT[NAMELEN]="vg-20";       /* default ascii font */
char filename[NAMELEN]="";	   /* input file name, empty for stdin */
char tempfile[]="/usr/tmp/xhzview0_XXXXXX";	/* tmp file */
int lnSpacing = 4;   /* default line spacing */
int chSpacing = 0;   /* space between ascii & hanzi */
int lines = 24;      /* default lines / page */
int chars = 40;      /* default chars / line */
int curLn = 0;       /* current line number */
int cLnHt;           /* line height */
int noLnWrap=false;  /* default run through line wrap filter */

GC gc;		     		/* default GC */
XFontStruct *c_font, *e_font;	/* Two fonts */

char line[LINELEN];		/* line buffer */

FILE *fp = NULL;

static	int	lp;	/* # line-ends */
static	int	nLn;	/* number of total line-ends allocated */
static	struct lnType {int st; long n} *lns = NULL;	/* line-end structure */

myexit(i)
    int i;
{
    /* clean up before exit */
    if (strcmp(tempfile, filename)) unlink(tempfile);
    exit(i);
}


/* open a file with error check */
FILE *myfopen (file, mode)
    char *file, *mode;
{
    FILE *fp;
    
    if (!(fp=fopen(file, mode)))
    {
	fprintf(stderr, "Can't open file %s.\n", file);
	myexit(1);
    }
    return (fp);
}

usage()
{
    fprintf(stderr, "usage: %s [-cfn <HanziFont>] [-afn <AsciiFont>] [-nowrap] \\ \n", program_name);
    fprintf(stderr, "        [-ls <ls>] [-cs <cs>] [-l <l>] [-n <n>] [filename]\n");
    fprintf(stderr, "  <HanziFont>  - Default is '%s'.\n", cFONT);
    fprintf(stderr, "  <AsciiFont>  - Default is '%s'.\n", eFONT);    
    fprintf(stderr, "  <ls>  - space between lines, 1/<ls> of the max. line height,  \n");
    fprintf(stderr, "          0 for none.  Default is %d.\n", lnSpacing);
    fprintf(stderr, "  <cs>  - space between Ascii/hanzi, 1/<cs> of hanzi width,\n");
    fprintf(stderr, "          0 for none.  Default is 0.\n");
    fprintf(stderr, "  <l>   - # lines per page. Default is %d.\n",lines);
    fprintf(stderr, "  <n>   - # characters per line. Default is %d.\n",chars);
    fprintf(stderr, " The -nowrap option tells not to wrap long lines.\n");
    exit(1);
}

main(argc, argv)
     int	argc;
     char	**argv;
{
    register int i;
    XGCValues gc_init;
    XEvent event;
    char buffer[10];


    INIT_NAME;

    /* parse command line */
    Get_X_Options(&argc, argv);
    for (i=1; i<argc; i++)
    {
	if (!strcmp("-afn",argv[i]))
	{
	    if (++i >= argc) usage();
	    strcpy(eFONT,argv[i]);
	}
	else if (!strcmp("-cfn",argv[i]))
	{
	    if (++i >= argc) usage();
	    strcpy(cFONT,argv[i]);
	}
	else if (!strcmp("-ls",argv[i]))
	{
	    if (++i >= argc) usage();
	    lnSpacing = atoi(argv[i]);
	}
	else if (!strcmp("-cs",argv[i]))
	{
	    if (++i >= argc) usage();
	    chSpacing = atoi(argv[i]);
	}
	else if (!strcmp("-l",argv[i]))
	{
	    if (++i >= argc) usage();
	    lines = atoi(argv[i]);
	}
	else if (!strcmp("-n",argv[i]))
	{
	    if (++i >= argc) usage();
	    chars = atoi(argv[i]);
	}
	else if (!strcmp("-nowrap",argv[i])) noLnWrap = true;
	else if (argv[i][0]=='-') usage();
	else if (argv[i][0]!='-') strcpy(filename,argv[i]);
    }

    c_font = Open_Font(cFONT);
    e_font = Open_Font(eFONT);

    Resolve_X_Options();

    /* calculate line height, spacing, etc. */
    cLnHt = max(c_font->ascent + c_font->descent,
		e_font->ascent + e_font->descent);
    if (lnSpacing) lnSpacing = cLnHt / lnSpacing;
    if (chSpacing) chSpacing = c_font->max_bounds.width / chSpacing;
    cLnHt += lnSpacing;

    /* Get minimun perferred size */
    Calc_Default_Size();

    Create_Default_Window();

    gc = Get_Default_GC();

    gc_init.font = c_font->fid;
    XChangeGC(dpy, gc, GCFont, &gc_init);

    XSelectInput (dpy, wind, ButtonPressMask | KeyPressMask | ExposureMask);
    XMapWindow(dpy, wind);

    mktemp(tempfile);
    Cal_LnEnds(filename, tempfile, !noLnWrap);
    fp = myfopen(filename[0] ? filename : tempfile, "r");

    for (;;) 
    {
	XNextEvent(dpy, &event);
	switch (event.type) {
	  case Expose:
	    if (event.xexpose.count==0)
		Display_Page();
	    break;
	  case MappingNotify:
	    XRefreshKeyboardMapping (&event);
	    break;
	  case ButtonPress:
	    if (event.xbutton.button==1) Do_Key('b');
	    if (event.xbutton.button==2) myexit(0);
	    if (event.xbutton.button==3) Do_Key('f');
	    break;
	  case KeyPress:
	    i = XLookupString(&event, buffer, 10, 0, 0);
	    if (i==1) Do_Key(buffer[0]);
	    break;
	}
    }
}

help()
{
    printf("\n<n> <Left Mouse Button>  - go backward n pages (default 1)\n");
    printf("<n> <Right Mouse Button> - go forward n pages (default 1)\n");
    printf("h - this help message\n");
    printf("q - quit\n");
    printf("< - go to the first page of file\n");
    printf("> - go to the last page of file\n");
    printf("<n> f - go forward n pages (default 1)\n");
    printf("<n> b - go backward n pages (default 1)\n");
    printf("<n> <RET> - go forward n lines (default 1)\n");
    printf("<n> p - go backward n lines (default 1)\n");
    printf("<n> g - go to the n-th line (default line 1)\n");
}

/* calculate the window's minimum size & increment */
Calc_Default_Size()
{
    if (!geometry) 
    {
	/* user didn't override, use ideal size */
	/* leave one extra space at the right edge */
	size_hints.width = c_font->max_bounds.width * (chars + 1);  
	size_hints.height =  cLnHt * lines;
    }
    size_hints.width_inc = c_font->max_bounds.width;
    size_hints.height_inc = cLnHt;
    size_hints.min_width = c_font->max_bounds.width;
    size_hints.min_height = cLnHt;
}

/*
  display a page starting from curLn
  handles resized window as well
  */
Display_Page()
{
    int ip=0, i=0, y, end;
    XWindowAttributes	wind_info;
    static int lastLn = -1, lastWid = -1, lastHt = -1;

    /* Get the size of the window */
    if (!XGetWindowAttributes(dpy, wind, &wind_info))
	Fatal_Error("Can't get window attributes!");

    /* size pre-calculated first time unless user specifed geometry */
    if ((lastWid==-1) && !geometry)
    {
	lastWid = wind_info.width;
	lastHt = wind_info.height;
    }

    /* window resized */
    if ((lastWid != wind_info.width) || (lastHt != wind_info.height))
    {
	int	olp = lp;
	
	size_hints.width = wind_info.width;
	size_hints.height = wind_info.height;
	lines = wind_info.height / cLnHt;
	chars = wind_info.width / c_font->max_bounds.width;
	Cal_LnEnds(filename[0] ? filename : tempfile, "", !noLnWrap);
	lastLn = -1;	/* sort-of force window to clear */
	curLn = curLn * lp / olp;
    }
    
    if (lastLn!=curLn) XClearWindow(dpy, wind);
    y = c_font->ascent;
    end = curLn + (min(lines + curLn, lp - curLn));
    for (i=curLn; i<end; i++)
    {
	readLn(fp, lns, line, i);
	MyDrawLine(y, line);
	y += cLnHt;
    }
    lastLn = curLn;
    lastWid = wind_info.width;
    lastHt = wind_info.height;
}

MyDrawString(s, startp, end, ch16p, xp, y)
    char *s;
    int *startp, end, *ch16p, *xp, y;
{
    XGCValues tmp_gc;
    int start = *startp, ch16 = *ch16p, n;
        
    if (ch16)
    {
	n = (end - start) / 2;	/* two bytes counted as 1 char */
	tmp_gc.font = c_font->fid;
	XChangeGC(dpy, gc, GCFont, &tmp_gc);
	XDrawImageString16(dpy, wind, gc, *xp, y, s + start, n);
	*xp += n * c_font->max_bounds.width + chSpacing;
	/* XTextWidth doesn't count the undefined chars
	   *xp += XTextWidth16(c_font, s + start, end - start);
	   */
    }
    else
    {	
	tmp_gc.font = e_font->fid;
	XChangeGC(dpy, gc, GCFont, &tmp_gc);
	XDrawImageString(dpy, wind, gc, *xp, y-e_font->descent,
			 s + start, end - start);
	*xp += XTextWidth(e_font, s + start, end - start) + chSpacing;
    }
    *startp = end;
    *ch16p = ! ch16;
}

/* draws a line in the file at vertical position y */
MyDrawLine(y, line)
    int y;
    char *line;
{
    register int i = 0;
    int s = 0, x = 1, ch16, len = strlen(line), is16;

    if (len==0) return;

    /* check if high bit set to determine initial mode */
    ch16 = line[0] & 0x80;
    
    for (i=0; i<len; i++)
    {
	is16 = line[i] & 0x80;	/* check if high bit set */

	/* if there is a mode switch, dump the line */
	if ((is16 && !ch16) || (ch16 && !is16))
	    MyDrawString(line, &s, i, &ch16, &x, y);

	/* skip 2nd byte of two-byte char */
	if (is16) i++;		
    }
    
    /* take care the left-over */
    MyDrawString(line, &s, i, &ch16, &x, y);
}

/* go backward n pages */
Go_Back(n)
    int	n;
{
    if (n==0) n=1;
    if (curLn==0)
    {
	Beep();
	return;
    }
    else if ((curLn - n * lines)<0) curLn=0;
    else curLn -= n * lines;
    Display_Page();
}

/* go forward n pages */
Go_Forward(n)
    int n;
{
    if (n==0) n=1;
    if ((curLn + n * lines)>=lp) 
    {
	Beep();
	return;
    }
    else
    {
	curLn += n * lines;
	Display_Page();
    }
}

/* go to the last page */
Go_End()
{
    curLn = lp - lines;
    if (curLn<0) curLn = 0;
    Display_Page();
}

Do_Key(c)
    char c;
{
    static 	int	n = 0;
    
    switch (c)
    {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9':
	n = n * 10 + (c - '0');
	break;
      case 'q':
	myexit(0);
	break;
      case ' ':
      case 'f':
	Go_Forward(n);
	n = 0;
	break;
      case 'b':
	Go_Back(n);
	n = 0;
	break;
      case '<':
	curLn=0;
	Display_Page();
	n = 0;
	break;
      case '>':
	Go_End();
	n = 0;
	break;
      case 'h':
	help();
	n = 0;
	break;
      case '\015':	/* RET */
	if (n==0) n = 1;
	if (curLn+n>=lp)
	{
	    Beep();
	    break;
	}
	curLn += n;
	Display_Page();
	n = 0;
	break;
      case 'p':	
	if (n==0) n = 1;
	if (curLn-n<0) 
	{
	    Beep();
	    break;
	}
	curLn -= n;
	Display_Page();
	n = 0;
	break;
      case 'g':
	if (n==0) n = 1;
	else if (n>=lp) 
	{
	    Beep();
	    break;
	}
	curLn = n - 1;
	Display_Page();
	n = 0;
	break;
      default:
	Beep();
	n = 0;
	break;
    }
}    

/* calculate line-ends */
Cal_LnEnds(infile, outfile, wrap)
    char *infile, *outfile;
    int wrap;
{
    int write_tmp = !infile[0];
    int i = 0;		/* character count */
    int n = 0;		/* char count for current line */
    int lnChrs;		/* chars per line */
    int c, ishz;
    FILE	*ifp, *ofp;

    /* allocate line-end structure */
    if (lns) free(lns);
    lns = (struct lnType *) malloc(NLINES * sizeof(struct lnType));
    nLn = NLINES;
    lp = 0;
    if (!lns) 
	Fatal_Error("Not enough memory.\n");
    lns[0].st = 0;
    
    if (write_tmp)
    {
	/* no input file, dump stdin to a temp file */
	ifp = stdin;
	ofp = myfopen(outfile, "w");
    }
    else ifp = myfopen(infile, "r");

    lnChrs = chars * 2;

    while ((c=fgetc(ifp))!=EOF)
    {
	if (write_tmp) fputc(c, ofp);
	n++;
	if (ishz = (c & 0x80))
	{
	    /* skip 2nd byte if hz byte found */
	    if ((c=fgetc(ifp))==EOF) break;
	    if (write_tmp) fputc(c, ofp);
	    i++;
	    n++;
	}
	
	if ((c=='\n')||((n>=lnChrs)&&wrap))
	{
	    /* always put the 2nd byte at the same line */
	    lns[lp++].n = (long) ((ishz) ? n : ((n>0) ? n - 1 : 0));

	    if (lp>=nLn)
	    {
		/* if no more line-ends, allocate more */
		nLn += NLINES;
		lns = (struct lnType *)
		    realloc(lns, nLn * sizeof(struct lnType));
		if (!lns) 
		    Fatal_Error("Not enough memory.\n");
	    }

	    /* not two-byte or newline character left for next line */
	    lns[lp].st = ((c!='\n') && !ishz) ? i : i + 1;
	    n = (c!='\n') && !ishz;
	}
	i++;
    }
    fclose(ifp);
    if (write_tmp) fclose(ofp);
}

/*
  Read the i-th line from file fp, using line-ends info from lns.
  The line is stored in the character array line.
  */
readLn(fp, lns, line, i)
    FILE *fp;
    struct lnType *lns;
    char *line;
    int i;
{
    register	int j;
    
#ifndef SEEK_SET
#define SEEK_SET 0
#endif

    fseek(fp, lns[i].st, SEEK_SET);
    for (j=0; j<lns[i].n; j++)
	line[j] = fgetc(fp);
    line[lns[i].n] = '\0';
}




