/*
 * LaTeX to HTML translator.
 * Copyright (C) 1996 David Mosberger-Tang.
 * This file is part of the LaTeX to HTML translator package.
 *
 * The LaTeX to HTML translator is free software; you can redistribute
 * it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * The LaTeX to HTML translator is distributed in the hope that it
 * will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the LaTeX to HTML translator; see the file COPYING.  If
 * not, write to the Free Software Foundation, Inc., 675 Mass Ave,
 * Cambridge, MA 02139, USA.
 */
/*
 * String hash-tables.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

struct entry {
    struct entry *	next;
    const char *	key;
    void *		value;
};

struct sh {
    int			num_buckets;
    struct entry *	tab[0];
};


static unsigned
nhash (const unsigned char * str, size_t len)
{
    unsigned val = 0;
    size_t i;

    for (i = 0; i < len; ++i) {
	val = ((val << 8) | str[i]) ^ (val >> 24);
    }
    return val;
}


struct sh *
sh_new (int num_buckets)
{
    struct sh * sh;
    size_t size = sizeof(sh) + num_buckets * sizeof(sh->tab[0]);

    if (num_buckets & (num_buckets - 1)) {
	fprintf(stderr,
		"sh_new: num_buckets must be a power of two (not %d)\n",
		num_buckets);
	exit(1);
    }

    sh = malloc(size);
    if (!sh) {
	fprintf(stderr, "sh_new: out of memory\n");
	return 0;
    }
    memset(sh, 0, size);
    sh->num_buckets = num_buckets;

    return sh;
}


void
sh_enter (struct sh * sh, const char * name, void * value)
{
    struct entry * e;
    unsigned i;

    i = nhash(name, strlen(name)) & (sh->num_buckets - 1);
    e = malloc(sizeof(struct entry));
    e->next = sh->tab[i];
    sh->tab[i] = e;
    e->key = name;
    e->value = value;
}


void *
sh_lookup (struct sh * sh, const char * name)
{
    struct entry * e;
    unsigned i;

    i = nhash(name, strlen(name)) & (sh->num_buckets - 1);

    for (e = sh->tab[i]; e; e = e->next) {
	if (strcmp(e->key, name) == 0) {
	    return e->value;
	}
    }
    return 0;
}


void *
sh_nlookup (struct sh * sh, const char * name, size_t len)
{
    struct entry * e;
    unsigned i;

    i = nhash(name, len) & (sh->num_buckets - 1);

    for (e = sh->tab[i]; e; e = e->next) {
	if (strncmp(e->key, name, len) == 0 && e->key[len] == '\0') {
	    return e->value;
	}
    }
    return 0;
}
