﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/client/AWSClient.h>
#include <aws/core/client/AWSClientAsyncCRTP.h>
#include <aws/core/client/ClientConfiguration.h>
#include <aws/core/utils/json/JsonSerializer.h>
#include <aws/ivschat/IvschatServiceClientModel.h>
#include <aws/ivschat/Ivschat_EXPORTS.h>

namespace Aws {
namespace ivschat {
/**
 * <p> <b>Introduction</b> </p> <p>The Amazon IVS Chat control-plane API enables
 * you to create and manage Amazon IVS Chat resources. You also need to integrate
 * with the <a
 * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html">
 * Amazon IVS Chat Messaging API</a>, to enable users to interact with chat rooms
 * in real time.</p> <p>The API is an AWS regional service. For a list of supported
 * regions and Amazon IVS Chat HTTPS service endpoints, see the Amazon IVS Chat
 * information on the <a
 * href="https://docs.aws.amazon.com/general/latest/gr/ivs.html">Amazon IVS
 * page</a> in the <i>AWS General Reference</i>. </p> <p>This document describes
 * HTTP operations. There is a separate <i>messaging</i> API for managing Chat
 * resources; see the <a
 * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/chat-messaging-api.html">
 * Amazon IVS Chat Messaging API Reference</a>.</p> <p> <b>Notes on
 * terminology:</b> </p> <ul> <li> <p>You create service applications using the
 * Amazon IVS Chat API. We refer to these as <i>applications</i>.</p> </li> <li>
 * <p>You create front-end client applications (browser and Android/iOS apps) using
 * the Amazon IVS Chat Messaging API. We refer to these as <i>clients</i>. </p>
 * </li> </ul> <p> <b>Resources</b> </p> <p>The following resources are part of
 * Amazon IVS Chat:</p> <ul> <li> <p> <b>LoggingConfiguration</b> — A configuration
 * that allows customers to store and record sent messages in a chat room. See the
 * Logging Configuration endpoints for more information.</p> </li> <li> <p>
 * <b>Room</b> — The central Amazon IVS Chat resource through which clients connect
 * to and exchange chat messages. See the Room endpoints for more information.</p>
 * </li> </ul> <p> <b>Tagging</b> </p> <p>A <i>tag</i> is a metadata label that you
 * assign to an AWS resource. A tag comprises a <i>key</i> and a <i>value</i>, both
 * set by you. For example, you might set a tag as <code>topic:nature</code> to
 * label a particular video category. See <a
 * href="https://docs.aws.amazon.com/tag-editor/latest/userguide/best-practices-and-strats.html">Best
 * practices and strategies</a> in <i>Tagging Amazon Web Services Resources and Tag
 * Editor</i> for details, including restrictions that apply to tags and "Tag
 * naming limits and requirements"; Amazon IVS Chat has no service-specific
 * constraints beyond what is documented there.</p> <p>Tags can help you identify
 * and organize your AWS resources. For example, you can use the same tag for
 * different resources to indicate that they are related. You can also use tags to
 * manage access (see <a
 * href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access_tags.html">Access
 * Tags</a>).</p> <p>The Amazon IVS Chat API has these tag-related operations:
 * <a>TagResource</a>, <a>UntagResource</a>, and <a>ListTagsForResource</a>. The
 * following resource supports tagging: Room.</p> <p>At most 50 tags can be applied
 * to a resource.</p> <p> <b>API Access Security</b> </p> <p>Your Amazon IVS Chat
 * applications (service applications and clients) must be authenticated and
 * authorized to access Amazon IVS Chat resources. Note the differences between
 * these concepts:</p> <ul> <li> <p> <i>Authentication</i> is about verifying
 * identity. Requests to the Amazon IVS Chat API must be signed to verify your
 * identity.</p> </li> <li> <p> <i>Authorization</i> is about granting permissions.
 * Your IAM roles need to have permissions for Amazon IVS Chat API requests.</p>
 * </li> </ul> <p>Users (viewers) connect to a room using secure access tokens that
 * you create using the <a>CreateChatToken</a> operation through the AWS SDK. You
 * call CreateChatToken for every user’s chat session, passing identity and
 * authorization information about the user.</p> <p> <b>Signing API Requests</b>
 * </p> <p>HTTP API requests must be signed with an AWS SigV4 signature using your
 * AWS security credentials. The AWS Command Line Interface (CLI) and the AWS SDKs
 * take care of signing the underlying API calls for you. However, if your
 * application calls the Amazon IVS Chat HTTP API directly, it’s your
 * responsibility to sign the requests.</p> <p>You generate a signature using valid
 * AWS credentials for an IAM role that has permission to perform the requested
 * action. For example, DeleteMessage requests must be made using an IAM role that
 * has the <code>ivschat:DeleteMessage</code> permission.</p> <p>For more
 * information:</p> <ul> <li> <p>Authentication and generating signatures — See <a
 * href="https://docs.aws.amazon.com/AmazonS3/latest/API/sig-v4-authenticating-requests.html">Authenticating
 * Requests (Amazon Web Services Signature Version 4)</a> in the <i>Amazon Web
 * Services General Reference</i>.</p> </li> <li> <p>Managing Amazon IVS
 * permissions — See <a
 * href="https://docs.aws.amazon.com/ivs/latest/userguide/security-iam.html">Identity
 * and Access Management</a> on the Security page of the <i>Amazon IVS User
 * Guide</i>.</p> </li> </ul> <p> <b>Amazon Resource Names (ARNs)</b> </p> <p>ARNs
 * uniquely identify AWS resources. An ARN is required when you need to specify a
 * resource unambiguously across all of AWS, such as in IAM policies and API calls.
 * For more information, see <a
 * href="https://docs.aws.amazon.com/general/latest/gr/aws-arns-and-namespaces.html">Amazon
 * Resource Names</a> in the <i>AWS General Reference</i>.</p>
 */
class AWS_IVSCHAT_API IvschatClient : public Aws::Client::AWSJsonClient, public Aws::Client::ClientWithAsyncTemplateMethods<IvschatClient> {
 public:
  typedef Aws::Client::AWSJsonClient BASECLASS;
  static const char* GetServiceName();
  static const char* GetAllocationTag();

  typedef IvschatClientConfiguration ClientConfigurationType;
  typedef IvschatEndpointProvider EndpointProviderType;

  /**
   * Initializes client to use DefaultCredentialProviderChain, with default http client factory, and optional client config. If client
   * config is not specified, it will be initialized to default values.
   */
  IvschatClient(const Aws::ivschat::IvschatClientConfiguration& clientConfiguration = Aws::ivschat::IvschatClientConfiguration(),
                std::shared_ptr<IvschatEndpointProviderBase> endpointProvider = nullptr);

  /**
   * Initializes client to use SimpleAWSCredentialsProvider, with default http client factory, and optional client config. If client config
   * is not specified, it will be initialized to default values.
   */
  IvschatClient(const Aws::Auth::AWSCredentials& credentials, std::shared_ptr<IvschatEndpointProviderBase> endpointProvider = nullptr,
                const Aws::ivschat::IvschatClientConfiguration& clientConfiguration = Aws::ivschat::IvschatClientConfiguration());

  /**
   * Initializes client to use specified credentials provider with specified client config. If http client factory is not supplied,
   * the default http client factory will be used
   */
  IvschatClient(const std::shared_ptr<Aws::Auth::AWSCredentialsProvider>& credentialsProvider,
                std::shared_ptr<IvschatEndpointProviderBase> endpointProvider = nullptr,
                const Aws::ivschat::IvschatClientConfiguration& clientConfiguration = Aws::ivschat::IvschatClientConfiguration());

  /* Legacy constructors due deprecation */
  /**
   * Initializes client to use DefaultCredentialProviderChain, with default http client factory, and optional client config. If client
   * config is not specified, it will be initialized to default values.
   */
  IvschatClient(const Aws::Client::ClientConfiguration& clientConfiguration);

  /**
   * Initializes client to use SimpleAWSCredentialsProvider, with default http client factory, and optional client config. If client config
   * is not specified, it will be initialized to default values.
   */
  IvschatClient(const Aws::Auth::AWSCredentials& credentials, const Aws::Client::ClientConfiguration& clientConfiguration);

  /**
   * Initializes client to use specified credentials provider with specified client config. If http client factory is not supplied,
   * the default http client factory will be used
   */
  IvschatClient(const std::shared_ptr<Aws::Auth::AWSCredentialsProvider>& credentialsProvider,
                const Aws::Client::ClientConfiguration& clientConfiguration);

  /* End of legacy constructors due deprecation */
  virtual ~IvschatClient();

  /**
   * <p>Creates an encrypted token that is used by a chat participant to establish an
   * individual WebSocket chat connection to a room. When the token is used to
   * connect to chat, the connection is valid for the session duration specified in
   * the request. The token becomes invalid at the token-expiration timestamp
   * included in the response.</p> <p>Use the <code>capabilities</code> field to
   * permit an end user to send messages or moderate a room.</p> <p>The
   * <code>attributes</code> field securely attaches structured data to the chat
   * session; the data is included within each message sent by the end user and
   * received by other participants in the room. Common use cases for attributes
   * include passing end-user profile data like an icon, display name, colors,
   * badges, and other display features.</p> <p>Encryption keys are owned by Amazon
   * IVS Chat and never used directly by your application.</p><p><h3>See Also:</h3>
   * <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateChatToken">AWS
   * API Reference</a></p>
   */
  virtual Model::CreateChatTokenOutcome CreateChatToken(const Model::CreateChatTokenRequest& request) const;

  /**
   * A Callable wrapper for CreateChatToken that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename CreateChatTokenRequestT = Model::CreateChatTokenRequest>
  Model::CreateChatTokenOutcomeCallable CreateChatTokenCallable(const CreateChatTokenRequestT& request) const {
    return SubmitCallable(&IvschatClient::CreateChatToken, request);
  }

  /**
   * An Async wrapper for CreateChatToken that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename CreateChatTokenRequestT = Model::CreateChatTokenRequest>
  void CreateChatTokenAsync(const CreateChatTokenRequestT& request, const CreateChatTokenResponseReceivedHandler& handler,
                            const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::CreateChatToken, request, handler, context);
  }

  /**
   * <p>Creates a logging configuration that allows clients to store and record sent
   * messages.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateLoggingConfiguration">AWS
   * API Reference</a></p>
   */
  virtual Model::CreateLoggingConfigurationOutcome CreateLoggingConfiguration(
      const Model::CreateLoggingConfigurationRequest& request) const;

  /**
   * A Callable wrapper for CreateLoggingConfiguration that returns a future to the operation so that it can be executed in parallel to
   * other requests.
   */
  template <typename CreateLoggingConfigurationRequestT = Model::CreateLoggingConfigurationRequest>
  Model::CreateLoggingConfigurationOutcomeCallable CreateLoggingConfigurationCallable(
      const CreateLoggingConfigurationRequestT& request) const {
    return SubmitCallable(&IvschatClient::CreateLoggingConfiguration, request);
  }

  /**
   * An Async wrapper for CreateLoggingConfiguration that queues the request into a thread executor and triggers associated callback when
   * operation has finished.
   */
  template <typename CreateLoggingConfigurationRequestT = Model::CreateLoggingConfigurationRequest>
  void CreateLoggingConfigurationAsync(const CreateLoggingConfigurationRequestT& request,
                                       const CreateLoggingConfigurationResponseReceivedHandler& handler,
                                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::CreateLoggingConfiguration, request, handler, context);
  }

  /**
   * <p>Creates a room that allows clients to connect and pass
   * messages.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/CreateRoom">AWS
   * API Reference</a></p>
   */
  virtual Model::CreateRoomOutcome CreateRoom(const Model::CreateRoomRequest& request = {}) const;

  /**
   * A Callable wrapper for CreateRoom that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename CreateRoomRequestT = Model::CreateRoomRequest>
  Model::CreateRoomOutcomeCallable CreateRoomCallable(const CreateRoomRequestT& request = {}) const {
    return SubmitCallable(&IvschatClient::CreateRoom, request);
  }

  /**
   * An Async wrapper for CreateRoom that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename CreateRoomRequestT = Model::CreateRoomRequest>
  void CreateRoomAsync(const CreateRoomResponseReceivedHandler& handler,
                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr,
                       const CreateRoomRequestT& request = {}) const {
    return SubmitAsync(&IvschatClient::CreateRoom, request, handler, context);
  }

  /**
   * <p>Deletes the specified logging configuration.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteLoggingConfiguration">AWS
   * API Reference</a></p>
   */
  virtual Model::DeleteLoggingConfigurationOutcome DeleteLoggingConfiguration(
      const Model::DeleteLoggingConfigurationRequest& request) const;

  /**
   * A Callable wrapper for DeleteLoggingConfiguration that returns a future to the operation so that it can be executed in parallel to
   * other requests.
   */
  template <typename DeleteLoggingConfigurationRequestT = Model::DeleteLoggingConfigurationRequest>
  Model::DeleteLoggingConfigurationOutcomeCallable DeleteLoggingConfigurationCallable(
      const DeleteLoggingConfigurationRequestT& request) const {
    return SubmitCallable(&IvschatClient::DeleteLoggingConfiguration, request);
  }

  /**
   * An Async wrapper for DeleteLoggingConfiguration that queues the request into a thread executor and triggers associated callback when
   * operation has finished.
   */
  template <typename DeleteLoggingConfigurationRequestT = Model::DeleteLoggingConfigurationRequest>
  void DeleteLoggingConfigurationAsync(const DeleteLoggingConfigurationRequestT& request,
                                       const DeleteLoggingConfigurationResponseReceivedHandler& handler,
                                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::DeleteLoggingConfiguration, request, handler, context);
  }

  /**
   * <p>Sends an event to a specific room which directs clients to delete a specific
   * message; that is, unrender it from view and delete it from the client’s chat
   * history. This event’s <code>EventName</code> is <code>aws:DELETE_MESSAGE</code>.
   * This replicates the <a
   * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-deletemessage-publish.html">
   * DeleteMessage</a> WebSocket operation in the Amazon IVS Chat Messaging
   * API.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteMessage">AWS
   * API Reference</a></p>
   */
  virtual Model::DeleteMessageOutcome DeleteMessage(const Model::DeleteMessageRequest& request) const;

  /**
   * A Callable wrapper for DeleteMessage that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename DeleteMessageRequestT = Model::DeleteMessageRequest>
  Model::DeleteMessageOutcomeCallable DeleteMessageCallable(const DeleteMessageRequestT& request) const {
    return SubmitCallable(&IvschatClient::DeleteMessage, request);
  }

  /**
   * An Async wrapper for DeleteMessage that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename DeleteMessageRequestT = Model::DeleteMessageRequest>
  void DeleteMessageAsync(const DeleteMessageRequestT& request, const DeleteMessageResponseReceivedHandler& handler,
                          const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::DeleteMessage, request, handler, context);
  }

  /**
   * <p>Deletes the specified room.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DeleteRoom">AWS
   * API Reference</a></p>
   */
  virtual Model::DeleteRoomOutcome DeleteRoom(const Model::DeleteRoomRequest& request) const;

  /**
   * A Callable wrapper for DeleteRoom that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename DeleteRoomRequestT = Model::DeleteRoomRequest>
  Model::DeleteRoomOutcomeCallable DeleteRoomCallable(const DeleteRoomRequestT& request) const {
    return SubmitCallable(&IvschatClient::DeleteRoom, request);
  }

  /**
   * An Async wrapper for DeleteRoom that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename DeleteRoomRequestT = Model::DeleteRoomRequest>
  void DeleteRoomAsync(const DeleteRoomRequestT& request, const DeleteRoomResponseReceivedHandler& handler,
                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::DeleteRoom, request, handler, context);
  }

  /**
   * <p>Disconnects all connections using a specified user ID from a room. This
   * replicates the <a
   * href="https://docs.aws.amazon.com/ivs/latest/chatmsgapireference/actions-disconnectuser-publish.html">
   * DisconnectUser</a> WebSocket operation in the Amazon IVS Chat Messaging
   * API.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/DisconnectUser">AWS
   * API Reference</a></p>
   */
  virtual Model::DisconnectUserOutcome DisconnectUser(const Model::DisconnectUserRequest& request) const;

  /**
   * A Callable wrapper for DisconnectUser that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename DisconnectUserRequestT = Model::DisconnectUserRequest>
  Model::DisconnectUserOutcomeCallable DisconnectUserCallable(const DisconnectUserRequestT& request) const {
    return SubmitCallable(&IvschatClient::DisconnectUser, request);
  }

  /**
   * An Async wrapper for DisconnectUser that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename DisconnectUserRequestT = Model::DisconnectUserRequest>
  void DisconnectUserAsync(const DisconnectUserRequestT& request, const DisconnectUserResponseReceivedHandler& handler,
                           const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::DisconnectUser, request, handler, context);
  }

  /**
   * <p>Gets the specified logging configuration.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetLoggingConfiguration">AWS
   * API Reference</a></p>
   */
  virtual Model::GetLoggingConfigurationOutcome GetLoggingConfiguration(const Model::GetLoggingConfigurationRequest& request) const;

  /**
   * A Callable wrapper for GetLoggingConfiguration that returns a future to the operation so that it can be executed in parallel to other
   * requests.
   */
  template <typename GetLoggingConfigurationRequestT = Model::GetLoggingConfigurationRequest>
  Model::GetLoggingConfigurationOutcomeCallable GetLoggingConfigurationCallable(const GetLoggingConfigurationRequestT& request) const {
    return SubmitCallable(&IvschatClient::GetLoggingConfiguration, request);
  }

  /**
   * An Async wrapper for GetLoggingConfiguration that queues the request into a thread executor and triggers associated callback when
   * operation has finished.
   */
  template <typename GetLoggingConfigurationRequestT = Model::GetLoggingConfigurationRequest>
  void GetLoggingConfigurationAsync(const GetLoggingConfigurationRequestT& request,
                                    const GetLoggingConfigurationResponseReceivedHandler& handler,
                                    const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::GetLoggingConfiguration, request, handler, context);
  }

  /**
   * <p>Gets the specified room.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/GetRoom">AWS API
   * Reference</a></p>
   */
  virtual Model::GetRoomOutcome GetRoom(const Model::GetRoomRequest& request) const;

  /**
   * A Callable wrapper for GetRoom that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename GetRoomRequestT = Model::GetRoomRequest>
  Model::GetRoomOutcomeCallable GetRoomCallable(const GetRoomRequestT& request) const {
    return SubmitCallable(&IvschatClient::GetRoom, request);
  }

  /**
   * An Async wrapper for GetRoom that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename GetRoomRequestT = Model::GetRoomRequest>
  void GetRoomAsync(const GetRoomRequestT& request, const GetRoomResponseReceivedHandler& handler,
                    const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::GetRoom, request, handler, context);
  }

  /**
   * <p>Gets summary information about all your logging configurations in the AWS
   * region where the API request is processed.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListLoggingConfigurations">AWS
   * API Reference</a></p>
   */
  virtual Model::ListLoggingConfigurationsOutcome ListLoggingConfigurations(
      const Model::ListLoggingConfigurationsRequest& request = {}) const;

  /**
   * A Callable wrapper for ListLoggingConfigurations that returns a future to the operation so that it can be executed in parallel to other
   * requests.
   */
  template <typename ListLoggingConfigurationsRequestT = Model::ListLoggingConfigurationsRequest>
  Model::ListLoggingConfigurationsOutcomeCallable ListLoggingConfigurationsCallable(
      const ListLoggingConfigurationsRequestT& request = {}) const {
    return SubmitCallable(&IvschatClient::ListLoggingConfigurations, request);
  }

  /**
   * An Async wrapper for ListLoggingConfigurations that queues the request into a thread executor and triggers associated callback when
   * operation has finished.
   */
  template <typename ListLoggingConfigurationsRequestT = Model::ListLoggingConfigurationsRequest>
  void ListLoggingConfigurationsAsync(const ListLoggingConfigurationsResponseReceivedHandler& handler,
                                      const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr,
                                      const ListLoggingConfigurationsRequestT& request = {}) const {
    return SubmitAsync(&IvschatClient::ListLoggingConfigurations, request, handler, context);
  }

  /**
   * <p>Gets summary information about all your rooms in the AWS region where the API
   * request is processed. Results are sorted in descending order of
   * <code>updateTime</code>.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListRooms">AWS
   * API Reference</a></p>
   */
  virtual Model::ListRoomsOutcome ListRooms(const Model::ListRoomsRequest& request = {}) const;

  /**
   * A Callable wrapper for ListRooms that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename ListRoomsRequestT = Model::ListRoomsRequest>
  Model::ListRoomsOutcomeCallable ListRoomsCallable(const ListRoomsRequestT& request = {}) const {
    return SubmitCallable(&IvschatClient::ListRooms, request);
  }

  /**
   * An Async wrapper for ListRooms that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename ListRoomsRequestT = Model::ListRoomsRequest>
  void ListRoomsAsync(const ListRoomsResponseReceivedHandler& handler,
                      const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr,
                      const ListRoomsRequestT& request = {}) const {
    return SubmitAsync(&IvschatClient::ListRooms, request, handler, context);
  }

  /**
   * <p>Gets information about AWS tags for the specified ARN.</p><p><h3>See
   * Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/ListTagsForResource">AWS
   * API Reference</a></p>
   */
  virtual Model::ListTagsForResourceOutcome ListTagsForResource(const Model::ListTagsForResourceRequest& request) const;

  /**
   * A Callable wrapper for ListTagsForResource that returns a future to the operation so that it can be executed in parallel to other
   * requests.
   */
  template <typename ListTagsForResourceRequestT = Model::ListTagsForResourceRequest>
  Model::ListTagsForResourceOutcomeCallable ListTagsForResourceCallable(const ListTagsForResourceRequestT& request) const {
    return SubmitCallable(&IvschatClient::ListTagsForResource, request);
  }

  /**
   * An Async wrapper for ListTagsForResource that queues the request into a thread executor and triggers associated callback when operation
   * has finished.
   */
  template <typename ListTagsForResourceRequestT = Model::ListTagsForResourceRequest>
  void ListTagsForResourceAsync(const ListTagsForResourceRequestT& request, const ListTagsForResourceResponseReceivedHandler& handler,
                                const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::ListTagsForResource, request, handler, context);
  }

  /**
   * <p>Sends an event to a room. Use this within your application’s business logic
   * to send events to clients of a room; e.g., to notify clients to change the way
   * the chat UI is rendered.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/SendEvent">AWS
   * API Reference</a></p>
   */
  virtual Model::SendEventOutcome SendEvent(const Model::SendEventRequest& request) const;

  /**
   * A Callable wrapper for SendEvent that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename SendEventRequestT = Model::SendEventRequest>
  Model::SendEventOutcomeCallable SendEventCallable(const SendEventRequestT& request) const {
    return SubmitCallable(&IvschatClient::SendEvent, request);
  }

  /**
   * An Async wrapper for SendEvent that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename SendEventRequestT = Model::SendEventRequest>
  void SendEventAsync(const SendEventRequestT& request, const SendEventResponseReceivedHandler& handler,
                      const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::SendEvent, request, handler, context);
  }

  /**
   * <p>Adds or updates tags for the AWS resource with the specified
   * ARN.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/TagResource">AWS
   * API Reference</a></p>
   */
  virtual Model::TagResourceOutcome TagResource(const Model::TagResourceRequest& request) const;

  /**
   * A Callable wrapper for TagResource that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename TagResourceRequestT = Model::TagResourceRequest>
  Model::TagResourceOutcomeCallable TagResourceCallable(const TagResourceRequestT& request) const {
    return SubmitCallable(&IvschatClient::TagResource, request);
  }

  /**
   * An Async wrapper for TagResource that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename TagResourceRequestT = Model::TagResourceRequest>
  void TagResourceAsync(const TagResourceRequestT& request, const TagResourceResponseReceivedHandler& handler,
                        const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::TagResource, request, handler, context);
  }

  /**
   * <p>Removes tags from the resource with the specified ARN.</p><p><h3>See
   * Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UntagResource">AWS
   * API Reference</a></p>
   */
  virtual Model::UntagResourceOutcome UntagResource(const Model::UntagResourceRequest& request) const;

  /**
   * A Callable wrapper for UntagResource that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename UntagResourceRequestT = Model::UntagResourceRequest>
  Model::UntagResourceOutcomeCallable UntagResourceCallable(const UntagResourceRequestT& request) const {
    return SubmitCallable(&IvschatClient::UntagResource, request);
  }

  /**
   * An Async wrapper for UntagResource that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename UntagResourceRequestT = Model::UntagResourceRequest>
  void UntagResourceAsync(const UntagResourceRequestT& request, const UntagResourceResponseReceivedHandler& handler,
                          const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::UntagResource, request, handler, context);
  }

  /**
   * <p>Updates a specified logging configuration.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateLoggingConfiguration">AWS
   * API Reference</a></p>
   */
  virtual Model::UpdateLoggingConfigurationOutcome UpdateLoggingConfiguration(
      const Model::UpdateLoggingConfigurationRequest& request) const;

  /**
   * A Callable wrapper for UpdateLoggingConfiguration that returns a future to the operation so that it can be executed in parallel to
   * other requests.
   */
  template <typename UpdateLoggingConfigurationRequestT = Model::UpdateLoggingConfigurationRequest>
  Model::UpdateLoggingConfigurationOutcomeCallable UpdateLoggingConfigurationCallable(
      const UpdateLoggingConfigurationRequestT& request) const {
    return SubmitCallable(&IvschatClient::UpdateLoggingConfiguration, request);
  }

  /**
   * An Async wrapper for UpdateLoggingConfiguration that queues the request into a thread executor and triggers associated callback when
   * operation has finished.
   */
  template <typename UpdateLoggingConfigurationRequestT = Model::UpdateLoggingConfigurationRequest>
  void UpdateLoggingConfigurationAsync(const UpdateLoggingConfigurationRequestT& request,
                                       const UpdateLoggingConfigurationResponseReceivedHandler& handler,
                                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::UpdateLoggingConfiguration, request, handler, context);
  }

  /**
   * <p>Updates a room’s configuration.</p><p><h3>See Also:</h3>   <a
   * href="http://docs.aws.amazon.com/goto/WebAPI/ivschat-2020-07-14/UpdateRoom">AWS
   * API Reference</a></p>
   */
  virtual Model::UpdateRoomOutcome UpdateRoom(const Model::UpdateRoomRequest& request) const;

  /**
   * A Callable wrapper for UpdateRoom that returns a future to the operation so that it can be executed in parallel to other requests.
   */
  template <typename UpdateRoomRequestT = Model::UpdateRoomRequest>
  Model::UpdateRoomOutcomeCallable UpdateRoomCallable(const UpdateRoomRequestT& request) const {
    return SubmitCallable(&IvschatClient::UpdateRoom, request);
  }

  /**
   * An Async wrapper for UpdateRoom that queues the request into a thread executor and triggers associated callback when operation has
   * finished.
   */
  template <typename UpdateRoomRequestT = Model::UpdateRoomRequest>
  void UpdateRoomAsync(const UpdateRoomRequestT& request, const UpdateRoomResponseReceivedHandler& handler,
                       const std::shared_ptr<const Aws::Client::AsyncCallerContext>& context = nullptr) const {
    return SubmitAsync(&IvschatClient::UpdateRoom, request, handler, context);
  }

  void OverrideEndpoint(const Aws::String& endpoint);
  std::shared_ptr<IvschatEndpointProviderBase>& accessEndpointProvider();

 private:
  friend class Aws::Client::ClientWithAsyncTemplateMethods<IvschatClient>;
  void init(const IvschatClientConfiguration& clientConfiguration);

  IvschatClientConfiguration m_clientConfiguration;
  std::shared_ptr<IvschatEndpointProviderBase> m_endpointProvider;
};

}  // namespace ivschat
}  // namespace Aws
