//=================================================================================================
/*!
//  \file blaze/math/typetraits/RequiresEvaluation.h
//  \brief Header file for the RequiresEvaluation type trait
//
//  Copyright (C) 2012-2020 Klaus Iglberger - All Rights Reserved
//
//  This file is part of the Blaze library. You can redistribute it and/or modify it under
//  the terms of the New (Revised) BSD License. Redistribution and use in source and binary
//  forms, with or without modification, are permitted provided that the following conditions
//  are met:
//
//  1. Redistributions of source code must retain the above copyright notice, this list of
//     conditions and the following disclaimer.
//  2. Redistributions in binary form must reproduce the above copyright notice, this list
//     of conditions and the following disclaimer in the documentation and/or other materials
//     provided with the distribution.
//  3. Neither the names of the Blaze development group nor the names of its contributors
//     may be used to endorse or promote products derived from this software without specific
//     prior written permission.
//
//  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY
//  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
//  OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT
//  SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
//  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
//  TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
//  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
//  CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN
//  ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
//  DAMAGE.
*/
//=================================================================================================

#ifndef _BLAZE_MATH_TYPETRAITS_REQUIRESEVALUATION_H_
#define _BLAZE_MATH_TYPETRAITS_REQUIRESEVALUATION_H_


//*************************************************************************************************
// Includes
//*************************************************************************************************

#include <blaze/util/IntegralConstant.h>
#include <blaze/util/typetraits/IsReference.h>


namespace blaze {

//=================================================================================================
//
//  CLASS DEFINITION
//
//=================================================================================================

//*************************************************************************************************
/*!\brief Compile time check to query the requirement to evaluate an expression.
// \ingroup math_type_traits
//
// Via this type trait it is possible to determine whether a given vector or matrix expression
// type requires an intermediate evaluation in the context of a compound expression. In case
// the given type requires an evaluation, the \a value member constant is set to \a true, the
// nested type definition \a Type is \a TrueType, and the class derives from \a TrueType.
// Otherwise \a value is set to \a false, \a Type is \a FalseType, and the class derives
// from \a FalseType.
//
// \note that this type trait can only be applied to Blaze vector or matrix expressions
// or any other type providing the nested type \a CompositeType. In case this nested type
// is not available, applying the type trait results in a compile time error!
*/
template< typename T >
struct RequiresEvaluation
   : public BoolConstant< !IsReference_v<typename T::CompositeType> >
{};
//*************************************************************************************************


//*************************************************************************************************
/*!\brief Auxiliary variable template for the RequiresEvaluation type trait.
// \ingroup math_type_traits
//
// The RequiresEvaluation_v variable template provides a convenient shortcut to access the nested
// \a value of the RequiresEvaluation class template. For instance, given the type \a T the
// following two statements are identical:

   \code
   constexpr bool value1 = blaze::RequiresEvaluation<T>::value;
   constexpr bool value2 = blaze::RequiresEvaluation_v<T>;
   \endcode
*/
template< typename T >
constexpr bool RequiresEvaluation_v = RequiresEvaluation<T>::value;
//*************************************************************************************************

} // namespace blaze

#endif
