/*******************************************************************************
* Copyright 2018-2023 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

/*
 *
 *  Content:
 *            Touch test for all available oneMKL VM math functions with ILP64 API:
 *
 *            HA (High Accuracy), LA (Low Accuracy), EP (Enhanced Performance)
 *            single, double, complex precision function diffferent variants:
 *            basic, explicit mode, strided and strided with explicit mode
 *            are being called.
 *
 *******************************************************************************/

#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <float.h>
#include <complex.h>
#if (defined __INTEL_COMPILER) || (defined __INTEL_LLVM_COMPILER)
  #include <mathimf.h>
#else
  #include <math.h>
#endif

#include "mkl.h"

/**
 * Common constants:
 */
/* oneMKL VM vector length - number of inputs to be evaluated */
static const int64_t length = 1000;

/* Stride values for strided API tests */
static const int64_t stride = 3;

/* Maximum printed number of errors for each function - to reduce log size */
static const int max_printed = 1;

/* Maximum allowed ulp's (+1.0 due to rounded reference double precision function) */
static const float  s_allowed_ulp[] = { 1.0f, 4.0f, 5000.0f };
static const float  c_allowed_ulp[] = { 1.0f, 4.0f, 9000.0f };
static const double d_allowed_ulp[] = { 1.0 + 1.0,   4.0 + 1.0,  7.0e7 };
static const double z_allowed_ulp[] = { 1.0 + 1.0,   4.0 + 1.0,  1.0e8 };

/* Fixed argument value */
static const double fixed = 3.14;

/* Enable errors printout */
static const int print_err = 1;

/* Enable warnings printout */
static const int print_warn = 0;

/**
 * Mapping to oneMKL VM accuracy mode constants:
 */
static const unsigned int vm_mode[] = {VML_HA, VML_LA, VML_EP};

/* Maximum function full name length */
#define NAME_LEN 64

/**
 * Available function API variants:
 */
enum VmApi
{
    kFunc = 0,   /* Basic */
    kMFunc,      /* Explicit mode */
    kFuncI,      /* Strided */
    kMFuncI,     /* Strided with explicit mode */
    kApiNum       /* Number of API variants */
};

/**
 * Available function accuracies:
 */
enum VmAccuracy
{
    kHA = 0,  /* HA */
    kLA,      /* LA */
    kEP,      /* EP */
    kAccNum    /* Number of accuracies */
};

/**
 * Available function precisions:
 */
enum VmPrecision
{
    kSP = 0,  /* Single precision */
    kDP,      /* Double precision */
    kCP,      /* Complex single precision */
    kZP,      /* Complex double precision */
};

/**
 * Available function argument layouts:
 */
enum VmLayout
{
    kVtoV = 0,     /* 1 vector argument  -> 1 vector result */
    kVVtoV,        /* 2 vector arguments -> 1 vector result */
    kVtoVV,        /* 1 vector argument  -> 2 vector results */
    kVXtoV,        /* 1 vector and 1 fixed arguments -> 1 vector result */
    kVVXtoV,       /* 2 vector and 4 fixed arguments -> 1 vector result */
    kVCtoVR,       /* 1 complex argument  -> 1 real result */
    kVRtoVC,       /* 1 real argument  -> 1 complex result */
};

/**
 * Input arguments:
 */
typedef struct
{
    float          *sarg1, *sarg2;
    double         *darg1, *darg2;
    float complex  *carg1, *carg2;
    double complex *zarg1, *zarg2;

    double *sref1, *sref2;
    double *dref1, *dref2;
    double complex *cref1, *cref2;
    double complex *zref1, *zref2;
} VmInputData;

/**
 * Output results:
 */
typedef struct
{
    float          *sres1[kApiNum], *sres2[kApiNum];
    double         *dres1[kApiNum], *dres2[kApiNum];
    float complex  *cres1[kApiNum], *cres2[kApiNum];
    double complex *zres1[kApiNum], *zres2[kApiNum];
} VmOutputData;

/**
 * Types for pointers to real reference functions :
 */
typedef double (*RefVtoV)(double);
typedef double (*RefVVtoV)(double, double);
typedef void   (*RefVtoVV)(double, double*, double* );
typedef double (*RefVVXtoV)(double, double, double, double, double, double );

/**
 * Types for pointers to complex reference functions :
 */
typedef double complex (*CRefVtoV)(double complex);
typedef double         (*CRefCtoR)(double complex);
typedef double complex (*CRefRtoC)(double);
typedef double complex (*CRefVVtoV)(double complex, double complex);

/**
 * Type for pointer to VM functions launcher:
 */
typedef void (*VmFunc)(int, VmInputData*, VmOutputData*);

/**
 * @brief VM functions family launchers
 *
 * Run all variants of VM function family
 *
 * @param[in] acc          Accuracy
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 *
 */
 /**
 * Abs
 */
static void own_vm_abs (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAbs_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAbsI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAbs_64   (length, in->darg1, out->dres1[kFunc]);
    vdAbsI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAbs_64   (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kFunc]);
    vcAbsI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kFuncI], stride);
    vzAbs_64   (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kFunc]);
    vzAbsI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kFuncI], stride);

    vmsAbs_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAbsI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAbs_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAbsI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAbs_64  (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kMFunc], mode);
    vmcAbsI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kMFuncI], stride, mode);
    vmzAbs_64  (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kMFunc], mode);
    vmzAbsI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_abs */

 /**
 * Arg
 */
static void own_vm_arg (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcArg_64   (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kFunc]);
    vcArgI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kFuncI], stride);
    vzArg_64   (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kFunc]);
    vzArgI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kFuncI], stride);

    vmcArg_64  (length, (MKL_Complex8*)in->carg1, (float*)out->cres1[kMFunc], mode);
    vmcArgI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (float*)out->cres1[kMFuncI], stride, mode);
    vmzArg_64  (length, (MKL_Complex16*)in->zarg1, (double*)out->zres1[kMFunc], mode);
    vmzArgI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (double*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_arg */

 /**
 * CIS
 */
static void own_vm_cis (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcCIS_64   (length, in->sarg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCISI_64  (length/stride, in->sarg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCIS_64   (length, in->darg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCISI_64  (length/stride, in->darg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcCIS_64  (length, in->sarg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCISI_64 (length/stride, in->sarg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCIS_64  (length, in->darg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCISI_64 (length/stride, in->darg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cis */

/**
 * Acos
 */
static void own_vm_acos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcos_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAcosI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcos_64   (length, in->darg1, out->dres1[kFunc]);
    vdAcosI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAcos_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAcosI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAcos_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAcosI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAcos_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcosI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcos_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcosI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAcos_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAcosI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAcos_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAcosI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acos */

/**
 * Atan
 */
static void own_vm_atan (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtan_64   (length, in->darg1, out->dres1[kFunc]);
    vdAtanI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAtan_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAtanI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAtan_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAtanI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAtan_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAtan_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAtanI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAtan_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAtanI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan */

/**
 * Asin
 */
static void own_vm_asin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsin_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsin_64   (length, in->darg1, out->dres1[kFunc]);
    vdAsinI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAsin_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAsinI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAsin_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAsinI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAsin_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsin_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAsin_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAsinI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAsin_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAsinI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asin */

/**
 * Acosh
 */
static void own_vm_acosh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcosh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAcoshI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcosh_64   (length, in->darg1, out->dres1[kFunc]);
    vdAcoshI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAcosh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAcoshI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAcosh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAcoshI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAcosh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcoshI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcosh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcoshI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAcosh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAcoshI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAcosh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAcoshI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acosh */

/**
 * Asinh
 */
static void own_vm_asinh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsinh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinhI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsinh_64   (length, in->darg1, out->dres1[kFunc]);
    vdAsinhI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAsinh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAsinhI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAsinh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAsinhI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAsinh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinhI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsinh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinhI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAsinh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAsinhI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAsinh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAsinhI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asinh */

/**
 * Atanh
 */
static void own_vm_atanh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtanh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanhI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtanh_64   (length, in->darg1, out->dres1[kFunc]);
    vdAtanhI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcAtanh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcAtanhI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAtanh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzAtanhI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAtanh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanhI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtanh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanhI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcAtanh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAtanhI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAtanh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAtanhI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atanh */

/**
 * Add
 */
static void own_vm_add (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAdd_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAddI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAdd_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAddI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcAdd_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcAddI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzAdd_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzAddI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsAdd_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAddI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAdd_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAddI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcAdd_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcAddI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzAdd_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzAddI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_add */

/**
 * Sub
 */
static void own_vm_sub (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSub_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsSubI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdSub_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdSubI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcSub_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcSubI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSub_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzSubI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSub_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsSubI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdSub_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdSubI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcSub_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSubI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSub_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSubI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sub */

/**
 * Mul
 */
static void own_vm_mul (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMul_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMulI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMul_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMulI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcMul_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcMulI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzMul_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzMulI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsMul_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMulI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMul_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMulI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcMul_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcMulI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzMul_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzMulI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_mul */

/**
 * Atan2
 */
static void own_vm_atan2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan2_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAtan2I_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAtan2_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAtan2I_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsAtan2_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAtan2I_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan2_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAtan2I_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan2 */

/**
 * Atan2pi
 */
static void own_vm_atan2pi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtan2pi_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsAtan2piI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdAtan2pi_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdAtan2piI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsAtan2pi_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsAtan2piI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtan2pi_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdAtan2piI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atan2pi */

/**
 * CopySign
 */
static void own_vm_copysign (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCopySign_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsCopySignI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdCopySign_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdCopySignI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsCopySign_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsCopySignI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdCopySign_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdCopySignI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_copysign */

/**
 * Fdim
 */
static void own_vm_fdim (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFdim_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFdimI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFdim_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFdimI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFdim_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFdimI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFdim_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFdimI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fdim */

/**
 * Fmax
 */
static void own_vm_fmax (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmax_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFmaxI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmax_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFmaxI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmax_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFmaxI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmax_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFmaxI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmax */

/**
 * Fmin
 */
static void own_vm_fmin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmin_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFminI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmin_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFminI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmin_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFminI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmin_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFminI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmin */

/**
 * MaxMag
 */
static void own_vm_maxmag (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMaxMag_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMaxMagI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMaxMag_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMaxMagI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsMaxMag_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMaxMagI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMaxMag_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMaxMagI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_maxmag */

/**
 * MinMag
 */
static void own_vm_minmag (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsMinMag_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsMinMagI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdMinMag_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdMinMagI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsMinMag_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsMinMagI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdMinMag_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdMinMagI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_minmag */

/**
 * Fmod
 */
static void own_vm_fmod (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFmod_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsFmodI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdFmod_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdFmodI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsFmod_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsFmodI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdFmod_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdFmodI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_fmod */

/**
 * Hypot
 */
static void own_vm_hypot (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsHypot_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsHypotI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdHypot_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdHypotI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsHypot_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsHypotI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdHypot_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdHypotI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_hypot */

/**
 * NextAfter
 */
static void own_vm_nextafter (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsNextAfter_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsNextAfterI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdNextAfter_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdNextAfterI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsNextAfter_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsNextAfterI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdNextAfter_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdNextAfterI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_nextafter */

/**
 * Powr
 */
static void own_vm_powr (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPowr_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsPowrI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdPowr_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdPowrI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsPowr_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsPowrI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdPowr_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdPowrI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_powr */

/**
 * Pow2o3
 */
static void own_vm_pow2o3 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow2o3_64   (length, in->sarg1, out->sres1[kFunc]);
    vsPow2o3I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdPow2o3_64   (length, in->darg1, out->dres1[kFunc]);
    vdPow2o3I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsPow2o3_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsPow2o3I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow2o3_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdPow2o3I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_pow2o3 */

/**
 * Pow3o2
 */
static void own_vm_pow3o2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow3o2_64   (length, in->sarg1, out->sres1[kFunc]);
    vsPow3o2I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdPow3o2_64   (length, in->darg1, out->dres1[kFunc]);
    vdPow3o2I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsPow3o2_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsPow3o2I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow3o2_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdPow3o2I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_pow3o2 */

/**
 * Cbrt
 */
static void own_vm_cbrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCbrt_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCbrtI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCbrt_64   (length, in->darg1, out->dres1[kFunc]);
    vdCbrtI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCbrt_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCbrtI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCbrt_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCbrtI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cbrt */

/**
 * InvCbrt
 */
static void own_vm_invcbrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInvCbrt_64   (length, in->sarg1, out->sres1[kFunc]);
    vsInvCbrtI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInvCbrt_64   (length, in->darg1, out->dres1[kFunc]);
    vdInvCbrtI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInvCbrt_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvCbrtI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInvCbrt_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvCbrtI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_invcbrt */

/**
 * InvSqrt
 */
static void own_vm_invsqrt (int acc, VmInputData* in, VmOutputData* out)
{

    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInvSqrt_64   (length, in->sarg1, out->sres1[kFunc]);
    vsInvSqrtI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInvSqrt_64   (length, in->darg1, out->dres1[kFunc]);
    vdInvSqrtI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInvSqrt_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvSqrtI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInvSqrt_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvSqrtI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_invsqrt */

/**
 * Remainder
 */
static void own_vm_remainder (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRemainder_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsRemainderI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdRemainder_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdRemainderI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);

    vmsRemainder_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsRemainderI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdRemainder_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdRemainderI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_remainder */

/**
 * Div
 */
static void own_vm_div (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsDiv_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsDivI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdDiv_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdDivI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcDiv_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcDivI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzDiv_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzDivI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsDiv_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsDivI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdDiv_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdDivI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcDiv_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcDivI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzDiv_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzDivI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_div */

/**
 * Pow
 */
static void own_vm_pow (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsPow_64   (length, in->sarg1, in->sarg2, out->sres1[kFunc]);
    vsPowI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kFuncI], stride);
    vdPow_64   (length, in->darg1, in->darg2, out->dres1[kFunc]);
    vdPowI_64  (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kFuncI], stride);
    vcPow_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcPowI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzPow_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzPowI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsPow_64  (length, in->sarg1, in->sarg2, out->sres1[kMFunc], mode);
    vmsPowI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, out->sres1[kMFuncI], stride, mode);
    vmdPow_64  (length, in->darg1, in->darg2, out->dres1[kMFunc], mode);
    vmdPowI_64 (length/stride, in->darg1, stride, in->darg2, stride, out->dres1[kMFuncI], stride, mode);
    vmcPow_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcPowI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzPow_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzPowI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_div */

/**
 * Powx
 */
static void own_vm_powx (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];
    float  sarg2 = fixed;
    double darg2 = fixed;
    MKL_Complex8  carg2 = {fixed, fixed};
    MKL_Complex16 zarg2 = {fixed, fixed};


    vmlSetMode(mode);

    vsPowx_64   (length, in->sarg1, sarg2, out->sres1[kFunc]);
    vsPowxI_64  (length/stride, in->sarg1, stride, sarg2, out->sres1[kFuncI], stride);
    vdPowx_64   (length, in->darg1, darg2, out->dres1[kFunc]);
    vdPowxI_64  (length/stride, in->darg1, stride, darg2, out->dres1[kFuncI], stride);
    vcPowx_64   (length, (MKL_Complex8*)in->carg1, carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcPowxI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, carg2, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzPowx_64   (length, (MKL_Complex16*)in->zarg1, zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzPowxI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, zarg2, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsPowx_64  (length, in->sarg1, sarg2, out->sres1[kMFunc], mode);
    vmsPowxI_64 (length/stride, in->sarg1, stride, sarg2, out->sres1[kMFuncI], stride, mode);
    vmdPowx_64  (length, in->darg1, darg2, out->dres1[kMFunc], mode);
    vmdPowxI_64 (length/stride, in->darg1, stride, darg2, out->dres1[kMFuncI], stride, mode);
    vmcPowx_64  (length, (MKL_Complex8*)in->carg1, carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcPowxI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, carg2, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzPowx_64  (length, (MKL_Complex16*)in->zarg1, zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzPowxI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, zarg2, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_powx */

/**
 * LinearFrac
 */
static void own_vm_linearfrac (int acc, VmInputData* in, VmOutputData* out)
{
    float  sfixed = (float)fixed;
    double dfixed = (double)fixed;
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLinearFrac_64   (length, in->sarg1, in->sarg2, sfixed, sfixed, sfixed, sfixed, out->sres1[kFunc]);
    vsLinearFracI_64  (length/stride, in->sarg1, stride, in->sarg2, stride, sfixed, sfixed, sfixed, sfixed, out->sres1[kFuncI], stride);
    vdLinearFrac_64   (length, in->darg1, in->darg2, dfixed, dfixed, dfixed, dfixed, out->dres1[kFunc]);
    vdLinearFracI_64  (length/stride, in->darg1, stride, in->darg2, stride, dfixed, dfixed, dfixed, dfixed, out->dres1[kFuncI], stride);

    vmsLinearFrac_64  (length, in->sarg1, in->sarg2, sfixed, sfixed, sfixed, sfixed, out->sres1[kMFunc], mode);
    vmsLinearFracI_64 (length/stride, in->sarg1, stride, in->sarg2, stride, sfixed, sfixed, sfixed, sfixed, out->sres1[kMFuncI], stride, mode);
    vmdLinearFrac_64  (length, in->darg1, in->darg2, dfixed, dfixed, dfixed, dfixed, out->dres1[kMFunc], mode);
    vmdLinearFracI_64 (length/stride, in->darg1, stride, in->darg2, stride, dfixed, dfixed, dfixed, dfixed, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_linearfrac */

/**
 * Sqrt
 */
static void own_vm_sqrt (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSqrt_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSqrtI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSqrt_64   (length, in->darg1, out->dres1[kFunc]);
    vdSqrtI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSqrt_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSqrtI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSqrt_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSqrtI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSqrt_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSqrtI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSqrt_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSqrtI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSqrt_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSqrtI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSqrt_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSqrtI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sqrt */

/**
 * Ln
 */
static void own_vm_ln (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLn_64   (length, in->sarg1, out->sres1[kFunc]);
    vsLnI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLn_64   (length, in->darg1, out->dres1[kFunc]);
    vdLnI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcLn_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcLnI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzLn_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzLnI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsLn_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLnI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLn_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLnI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcLn_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcLnI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzLn_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzLnI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_ln */

/**
 * Sind
 */
static void own_vm_sind (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSind_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSindI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSind_64   (length, in->darg1, out->dres1[kFunc]);
    vdSindI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSind_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSindI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSind_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSindI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sind */

/**
 * Cosd
 */
static void own_vm_cosd (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCosd_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCosdI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCosd_64   (length, in->darg1, out->dres1[kFunc]);
    vdCosdI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCosd_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCosdI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCosd_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCosdI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cosd */

/**
 * Tand
 */
static void own_vm_tand (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTand_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTandI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTand_64   (length, in->darg1, out->dres1[kFunc]);
    vdTandI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTand_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTandI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTand_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTandI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tand */

/**
 * Sinpi
 */
static void own_vm_sinpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinpi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSinpiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSinpi_64   (length, in->darg1, out->dres1[kFunc]);
    vdSinpiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSinpi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinpiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSinpi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinpiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sinpi */

/**
 * Cospi
 */
static void own_vm_cospi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCospi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCospiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCospi_64   (length, in->darg1, out->dres1[kFunc]);
    vdCospiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCospi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCospiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCospi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCospiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cospi */

/**
 * Tanpi
 */
static void own_vm_tanpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTanpi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTanpiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTanpi_64   (length, in->darg1, out->dres1[kFunc]);
    vdTanpiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTanpi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanpiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTanpi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanpiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tanpi */

/**
 * SinCos
 */
static void own_vm_sincos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinCos_64   (length, in->sarg1, out->sres1[kFunc], out->sres2[kFunc]);
    vsSinCosI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride, out->sres2[kFuncI], stride);
    vdSinCos_64   (length, in->darg1, out->dres1[kFunc], out->dres2[kFunc]);
    vdSinCosI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride, out->dres2[kFuncI], stride);

    vmsSinCos_64  (length, in->sarg1, out->sres1[kMFunc], out->sres2[kMFunc], mode);
    vmsSinCosI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, out->sres2[kMFuncI], stride, mode);
    vmdSinCos_64  (length, in->darg1, out->dres1[kMFunc], out->dres2[kMFunc], mode);
    vmdSinCosI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, out->dres2[kMFuncI], stride, mode);

    return;
} /* own_vm_sincos */

/**
 * Modf
 */
static void own_vm_modf (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsModf_64   (length, in->sarg1, out->sres1[kFunc], out->sres2[kFunc]);
    vsModfI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride, out->sres2[kFuncI], stride);
    vdModf_64   (length, in->darg1, out->dres1[kFunc], out->dres2[kFunc]);
    vdModfI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride, out->dres2[kFuncI], stride);

    vmsModf_64  (length, in->sarg1, out->sres1[kMFunc], out->sres2[kMFunc], mode);
    vmsModfI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, out->sres2[kMFuncI], stride, mode);
    vmdModf_64  (length, in->darg1, out->dres1[kMFunc], out->dres2[kMFunc], mode);
    vmdModfI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, out->dres2[kMFuncI], stride, mode);

    return;
} /* own_vm_modf */

/**
 * Conj
 */
static void own_vm_conj (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcConj_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcConjI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzConj_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzConjI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcConj_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcConjI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzConj_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzConjI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_conj */

/**
 * MulByConj
 */
static void own_vm_mulbyconj (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vcMulByConj_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kFunc]);
    vcMulByConjI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzMulByConj_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kFunc]);
    vzMulByConjI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmcMulByConj_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)in->carg2, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcMulByConjI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)in->carg2, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzMulByConj_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)in->zarg2, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzMulByConjI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)in->zarg2, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_mulbyconj */

/**
 * Sin
 */
static void own_vm_sin (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSin_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSinI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSin_64   (length, in->darg1, out->dres1[kFunc]);
    vdSinI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSin_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSinI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSin_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSinI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSin_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSin_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSin_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSinI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSin_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSinI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sin */

/**
 * Cos
 */
static void own_vm_cos (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCos_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCosI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCos_64   (length, in->darg1, out->dres1[kFunc]);
    vdCosI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcCos_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCosI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCos_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCosI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsCos_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCosI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCos_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCosI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcCos_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCosI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCos_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCosI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cos */

/**
 * Tan
 */
static void own_vm_tan (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTan_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTanI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTan_64   (length, in->darg1, out->dres1[kFunc]);
    vdTanI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcTan_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcTanI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzTan_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzTanI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsTan_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTan_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcTan_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcTanI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzTan_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzTanI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tan */

/**
 * Inv
 */
static void own_vm_inv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsInv_64   (length, in->sarg1, out->sres1[kFunc]);
    vsInvI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdInv_64   (length, in->darg1, out->dres1[kFunc]);
    vdInvI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsInv_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsInvI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdInv_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdInvI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_inv */

/**
 * Sqr
 */
static void own_vm_sqr (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSqr_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSqrI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSqr_64   (length, in->darg1, out->dres1[kFunc]);
    vdSqrI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsSqr_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSqrI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSqr_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSqrI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sqr */

/**
 * Frac
 */
static void own_vm_frac (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFrac_64   (length, in->sarg1, out->sres1[kFunc]);
    vsFracI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdFrac_64   (length, in->darg1, out->dres1[kFunc]);
    vdFracI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsFrac_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsFracI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdFrac_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdFracI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_frac */

/**
 * Ceil
 */
static void own_vm_ceil (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCeil_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCeilI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCeil_64   (length, in->darg1, out->dres1[kFunc]);
    vdCeilI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCeil_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCeilI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCeil_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCeilI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_ceil */

/**
 * Floor
 */
static void own_vm_floor (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsFloor_64   (length, in->sarg1, out->sres1[kFunc]);
    vsFloorI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdFloor_64   (length, in->darg1, out->dres1[kFunc]);
    vdFloorI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsFloor_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsFloorI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdFloor_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdFloorI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_floor */

/**
 * Round
 */
static void own_vm_round (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRound_64   (length, in->sarg1, out->sres1[kFunc]);
    vsRoundI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdRound_64   (length, in->darg1, out->dres1[kFunc]);
    vdRoundI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsRound_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsRoundI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdRound_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdRoundI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_round */

/**
 * Trunc
 */
static void own_vm_trunc (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTrunc_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTruncI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTrunc_64   (length, in->darg1, out->dres1[kFunc]);
    vdTruncI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTrunc_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTruncI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTrunc_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTruncI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_trunc */

/**
 * Rint
 */
static void own_vm_rint (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsRint_64   (length, in->sarg1, out->sres1[kFunc]);
    vsRintI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdRint_64   (length, in->darg1, out->dres1[kFunc]);
    vdRintI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsRint_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsRintI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdRint_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdRintI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_rint */

/**
 * NearbyInt
 */
static void own_vm_nearbyint (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsNearbyInt_64   (length, in->sarg1, out->sres1[kFunc]);
    vsNearbyIntI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdNearbyInt_64   (length, in->darg1, out->dres1[kFunc]);
    vdNearbyIntI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsNearbyInt_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsNearbyIntI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdNearbyInt_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdNearbyIntI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_nearbyint */

/**
 * Acospi
 */
static void own_vm_acospi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAcospi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAcospiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAcospi_64   (length, in->darg1, out->dres1[kFunc]);
    vdAcospiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAcospi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAcospiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAcospi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAcospiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_acospi */

/**
 * Asinpi
 */
static void own_vm_asinpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAsinpi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAsinpiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAsinpi_64   (length, in->darg1, out->dres1[kFunc]);
    vdAsinpiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAsinpi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAsinpiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAsinpi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAsinpiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_asinpi */

/**
 * Atanpi
 */
static void own_vm_atanpi (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsAtanpi_64   (length, in->sarg1, out->sres1[kFunc]);
    vsAtanpiI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdAtanpi_64   (length, in->darg1, out->dres1[kFunc]);
    vdAtanpiI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsAtanpi_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsAtanpiI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdAtanpi_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdAtanpiI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_atanpi */

/**
 * Sinh
 */
static void own_vm_sinh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsSinh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsSinhI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdSinh_64   (length, in->darg1, out->dres1[kFunc]);
    vdSinhI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcSinh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcSinhI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzSinh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzSinhI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsSinh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsSinhI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdSinh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdSinhI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcSinh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcSinhI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzSinh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzSinhI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_sinh */

/**
 * Cosh
 */
static void own_vm_cosh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCosh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCoshI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCosh_64   (length, in->darg1, out->dres1[kFunc]);
    vdCoshI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcCosh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcCoshI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzCosh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzCoshI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsCosh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCoshI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCosh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCoshI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcCosh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcCoshI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzCosh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzCoshI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cosh */

/**
 * Tanh
 */
static void own_vm_tanh (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTanh_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTanhI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTanh_64   (length, in->darg1, out->dres1[kFunc]);
    vdTanhI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcTanh_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcTanhI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzTanh_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzTanhI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsTanh_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTanhI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTanh_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTanhI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcTanh_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcTanhI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzTanh_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzTanhI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tanh */

/**
 * Exp
 */
static void own_vm_exp (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp_64   (length, in->sarg1, out->sres1[kFunc]);
    vsExpI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp_64   (length, in->darg1, out->dres1[kFunc]);
    vdExpI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcExp_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcExpI_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzExp_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzExpI_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsExp_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcExp_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcExpI_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzExp_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzExpI_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp */

/**
 * Exp2
 */
static void own_vm_exp2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp2_64   (length, in->sarg1, out->sres1[kFunc]);
    vsExp2I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp2_64   (length, in->darg1, out->dres1[kFunc]);
    vdExp2I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExp2_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExp2I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp2_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExp2I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp2 */

/**
 * Exp10
 */
static void own_vm_exp10 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExp10_64   (length, in->sarg1, out->sres1[kFunc]);
    vsExp10I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExp10_64   (length, in->darg1, out->dres1[kFunc]);
    vdExp10I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExp10_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExp10I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExp10_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExp10I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_exp10 */

/**
 * Expm1
 */
static void own_vm_expm1 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExpm1_64   (length, in->sarg1, out->sres1[kFunc]);
    vsExpm1I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExpm1_64   (length, in->darg1, out->dres1[kFunc]);
    vdExpm1I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExpm1_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpm1I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExpm1_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpm1I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_expm1 */

/**
 * Log2
 */
static void own_vm_log2 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog2_64   (length, in->sarg1, out->sres1[kFunc]);
    vsLog2I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog2_64   (length, in->darg1, out->dres1[kFunc]);
    vdLog2I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLog2_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog2I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog2_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog2I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log2 */

/**
 * Log10
 */
static void own_vm_log10 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog10_64   (length, in->sarg1, out->sres1[kFunc]);
    vsLog10I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog10_64   (length, in->darg1, out->dres1[kFunc]);
    vdLog10I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);
    vcLog10_64   (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kFunc]);
    vcLog10I_64  (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kFuncI], stride);
    vzLog10_64   (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kFunc]);
    vzLog10I_64  (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kFuncI], stride);

    vmsLog10_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog10I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog10_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog10I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);
    vmcLog10_64  (length, (MKL_Complex8*)in->carg1, (MKL_Complex8*)out->cres1[kMFunc], mode);
    vmcLog10I_64 (length/stride, (MKL_Complex8*)in->carg1, stride, (MKL_Complex8*)out->cres1[kMFuncI], stride, mode);
    vmzLog10_64  (length, (MKL_Complex16*)in->zarg1, (MKL_Complex16*)out->zres1[kMFunc], mode);
    vmzLog10I_64 (length/stride, (MKL_Complex16*)in->zarg1, stride, (MKL_Complex16*)out->zres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log10 */

/**
 * Log1p
 */
static void own_vm_log1p (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLog1p_64   (length, in->sarg1, out->sres1[kFunc]);
    vsLog1pI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLog1p_64   (length, in->darg1, out->dres1[kFunc]);
    vdLog1pI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLog1p_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLog1pI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLog1p_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLog1pI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_log1p */

/**
 * Erf
 */
static void own_vm_erf (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErf_64   (length, in->sarg1, out->sres1[kFunc]);
    vsErfI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErf_64   (length, in->darg1, out->dres1[kFunc]);
    vdErfI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErf_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErf_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erf */

/**
 * Erfc
 */
static void own_vm_erfc (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfc_64   (length, in->sarg1, out->sres1[kFunc]);
    vsErfcI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfc_64   (length, in->darg1, out->dres1[kFunc]);
    vdErfcI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfc_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfcI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfc_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfcI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfc */

/**
 * ErfInv
 */
static void own_vm_erfinv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfInv_64   (length, in->sarg1, out->sres1[kFunc]);
    vsErfInvI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfInv_64   (length, in->darg1, out->dres1[kFunc]);
    vdErfInvI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfInv_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfInvI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfInv_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfInvI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfinv */

/**
 * ErfcInv
 */
static void own_vm_erfcinv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsErfcInv_64   (length, in->sarg1, out->sres1[kFunc]);
    vsErfcInvI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdErfcInv_64   (length, in->darg1, out->dres1[kFunc]);
    vdErfcInvI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsErfcInv_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsErfcInvI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdErfcInv_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdErfcInvI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_erfcinv */

/**
 * CdfNorm
 */
static void own_vm_cdfnorm (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCdfNorm_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCdfNormI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCdfNorm_64   (length, in->darg1, out->dres1[kFunc]);
    vdCdfNormI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCdfNorm_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCdfNormI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCdfNorm_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCdfNormI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cdfnorm */

/**
 * CdfNormInv
 */
static void own_vm_cdfnorminv (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsCdfNormInv_64   (length, in->sarg1, out->sres1[kFunc]);
    vsCdfNormInvI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdCdfNormInv_64   (length, in->darg1, out->dres1[kFunc]);
    vdCdfNormInvI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsCdfNormInv_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsCdfNormInvI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdCdfNormInv_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdCdfNormInvI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_cdfnorminv */

/**
 * LGamma
 */
static void own_vm_lgamma (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsLGamma_64   (length, in->sarg1, out->sres1[kFunc]);
    vsLGammaI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdLGamma_64   (length, in->darg1, out->dres1[kFunc]);
    vdLGammaI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsLGamma_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsLGammaI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdLGamma_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdLGammaI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_lgamma */

/**
 * TGamma
 */
static void own_vm_tgamma (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsTGamma_64   (length, in->sarg1, out->sres1[kFunc]);
    vsTGammaI_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdTGamma_64   (length, in->darg1, out->dres1[kFunc]);
    vdTGammaI_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsTGamma_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsTGammaI_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdTGamma_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdTGammaI_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_tgamma */

/**
 * ExpInt1
 */
static void own_vm_expint1 (int acc, VmInputData* in, VmOutputData* out)
{
    int mode = vm_mode[acc];

    vmlSetMode(mode);

    vsExpInt1_64   (length, in->sarg1, out->sres1[kFunc]);
    vsExpInt1I_64  (length/stride, in->sarg1, stride, out->sres1[kFuncI], stride);
    vdExpInt1_64   (length, in->darg1, out->dres1[kFunc]);
    vdExpInt1I_64  (length/stride, in->darg1, stride, out->dres1[kFuncI], stride);

    vmsExpInt1_64  (length, in->sarg1, out->sres1[kMFunc], mode);
    vmsExpInt1I_64 (length/stride, in->sarg1, stride, out->sres1[kMFuncI], stride, mode);
    vmdExpInt1_64  (length, in->darg1, out->dres1[kMFunc], mode);
    vmdExpInt1I_64 (length/stride, in->darg1, stride, out->dres1[kMFuncI], stride, mode);

    return;
} /* own_vm_expint1 */

/**
 * Custom reference functions absent in LIBM:
 */
static double own_frac (double x) { return (x - trunc (x)); } /* signed fractional part */
static double own_invcbrt (double x) { return pow(x, -(1.0/3.0)); } /* inverse cube root */
static double own_pow2o3 (double x) { return pow(x, 2.0 / 3.0); } /* power x^(2/3) */
static double own_pow3o2 (double x) { return pow(x, 3.0 / 2.0); } /* power x^(3/2) */
static double own_inv (double x) { return 1.0 / x; } /* inverse 1/x */
static double own_sqr (double x) { return x * x; } /* square x*x */
static double own_add (double x, double y) { return x + y; } /* addition x+y */
static double own_sub (double x, double y) { return x - y; } /* subtraction x-y */
static double own_mul (double x, double y) { return x * y; } /* multiplication x-y */
static double own_div (double x, double y) { return x / y; } /* division x/y */
static double own_expint1 (double x) { double r; vmdExpInt1(1,&x, &r, VML_HA); return r; } /* exponential integral - exists in VM only */
static void   own_modf (double x, double* r1, double* r2) { *r2 = modf(x, r1); } /* truncated integer value |x| and the remaining fraction part x-|x| */
static double own_linearfrac (double x, double y, double a, double b, double c, double d) { return (x * a + b)/(y * c + d); } /* linear fraction (x*a + b)/(y*c + d)*/
static double own_minmag (double x, double y) { return (fabs(x) < fabs(y))?x:y; } /* min(|x|,|y|) */
static double own_maxmag (double x, double y) { return (fabs(x) > fabs(y))?x:y; } /* max(|x|,|y|) */
static void   own_sincos (double x, double* r1, double* r2) { *r1 = sin(x); *r2 = cos(x); return; } /* sin & cos */
static double own_invsqrt (double x) { return 1.0 / sqrt(x); } /* 1/sqrt(x) */

static double complex own_cadd (double complex x, double complex y) { return x + y; } /* complex x+y */
static double complex own_csub (double complex x, double complex y) { return x - y; } /* complex x+y */
static double complex own_cmul (double complex x, double complex y) { return x * y; } /* complex x*y */
static double complex own_cdiv (double complex x, double complex y) { return x / y; } /* complex x+y */
static double own_cabs (double complex x) { return hypot(creal(x),cimag(x)); } /* |x| */
static double own_carg (double complex x) { return atan2(cimag(x),creal(x)); } /* complex argument (atan2) */
static double complex own_cis (double x) { double r1, r2; own_sincos(x, &r2, &r1); double complex r = r1 + I * r2; return r; } /* complex CIS (sincos) */
static double complex own_cmulbyconj (double complex x, double complex y) { return x * conj(y); } /* complex x*conj(y) */

/**
 * @brief Safe malloc
 *
 * own_safe_malloc allocates memory and check resulted pointer.
 * Report error and exit application if unsuccessful.
 *
 * @param[in] size          Size in bytes
 * @return                  Pointer to allocated memory
 *
 */
static void* own_safe_malloc(int size)
{
    void* ptr = malloc (size);
    if (ptr == NULL)
    {
       fprintf (stderr, "\t\tERROR: %d bytes allocated unsuccessfully\n", size);
       exit(-1);
    }

    return ptr;
} /* own_safe_malloc */

/**
 * @brief Safe free
 *
 * own_safe_free deallocates memory.
 * Report error if NULL pointer passed.
 *
 * @param[in] ptr          Pointer to memory
 *
 */
static void own_safe_free(void *ptr)
{
    if (ptr != NULL) { free (ptr); }
    else
    {
       fprintf (stderr, "\t\tERROR: NULL pointer cannot be deallocated\n");
       exit(-1);
    }

    return;
} /* own_safe_free */

/**
 * @brief Allocate all input and reference result arrays
 *
 * Safe allocation of input and reference result arrays memory
 *
 * @param[in] len           Arrays length
 * @param[in, out] in       Input and reference resutl arrays
 * @param[in, out] out      Output arrays
 *
 */
static void own_allocate_data (int len, VmInputData* in, VmOutputData* out)
{
    in->sarg1    =  own_safe_malloc (len * sizeof (float));
    in->sarg2    =  own_safe_malloc (len * sizeof (float));
    in->darg1    =  own_safe_malloc (len * sizeof (double));
    in->darg2    =  own_safe_malloc (len * sizeof (double));
    in->carg1    =  own_safe_malloc (len * sizeof (float complex));
    in->carg2    =  own_safe_malloc (len * sizeof (float complex));
    in->zarg1    =  own_safe_malloc (len * sizeof (double complex));
    in->zarg2    =  own_safe_malloc (len * sizeof (double complex));
    in->sref1    =  own_safe_malloc (len * sizeof (double));
    in->sref2    =  own_safe_malloc (len * sizeof (double));
    in->dref1    =  own_safe_malloc (len * sizeof (double));
    in->dref2    =  own_safe_malloc (len * sizeof (double));
    in->cref1    =  own_safe_malloc (len * sizeof (double complex));
    in->cref2    =  own_safe_malloc (len * sizeof (double complex));
    in->zref1    =  own_safe_malloc (len * sizeof (double complex));
    in->zref2    =  own_safe_malloc (len * sizeof (double complex));

    for (int v = kFunc; v < kApiNum; v = v + 1)
    {
        out->sres1[v]   =  own_safe_malloc (len * sizeof (float));
        out->sres2[v]   =  own_safe_malloc (len * sizeof (float));
        out->dres1[v]   =  own_safe_malloc (len * sizeof (double));
        out->dres2[v]   =  own_safe_malloc (len * sizeof (double));
        out->cres1[v]   =  own_safe_malloc (len * sizeof (float complex));
        out->cres2[v]   =  own_safe_malloc (len * sizeof (float complex));
        out->zres1[v]   =  own_safe_malloc (len * sizeof (double complex));
        out->zres2[v]   =  own_safe_malloc (len * sizeof (double complex));
    }
} /* own_allocate_data */

/**
 * @brief Deallocate all input and reference result arrays
 *
 * Safe deallocation of input and reference result arrays memory
 *
 * @param[in, out] in       Input and reference resutl arrays
 * @param[in, out] out      Output arrays
 *
 */
static void own_deallocate_data (VmInputData* in, VmOutputData* out)
{
    own_safe_free (in->sarg1);
    own_safe_free (in->sarg2);
    own_safe_free (in->darg1);
    own_safe_free (in->darg2);
    own_safe_free (in->carg1);
    own_safe_free (in->carg2);
    own_safe_free (in->zarg1);
    own_safe_free (in->zarg2);
    own_safe_free (in->sref1);
    own_safe_free (in->sref2);
    own_safe_free (in->dref1);
    own_safe_free (in->dref2);
    own_safe_free (in->cref1);
    own_safe_free (in->cref2);
    own_safe_free (in->zref1);
    own_safe_free (in->zref2);

    for (int v = kFunc; v <= kMFuncI; v++)
    {
        own_safe_free (out->sres1[v]);
        own_safe_free (out->sres2[v]);
        own_safe_free (out->dres1[v]);
        own_safe_free (out->dres2[v]);
        own_safe_free (out->cres1[v]);
        own_safe_free (out->cres2[v]);
        own_safe_free (out->zres1[v]);
        own_safe_free (out->zres2[v]);
    }

    return;
} /* own_deallocate_data */

/**
 * @brief Single precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static float own_s_compute_ulp (float res, double ref )
{
    int ex       = 0;
    double den   = 1.0;
    float curulp = 0.0;
    if (isfinite (ref))
    {
        frexp (ref, &ex);                                     /* ex: integral power of two of ref */
        den = ldexp (1.0, ex - 24);                           /* den: ulp's denominator 2^(ex-p+1) */
        den = (den == 0.0)? 0x1.p-149 : den;                  /* if den=0 then replace by EPS to avoid divbyzero */
        curulp = (float)fabs ((((double)(res) - ref)) / den); /* |res-ref|/2^(ex-24) */
        curulp = isfinite (curulp)?curulp : FLT_MAX;         /* replace infinite ulp by big finite float number */
    }
    else
    {
        if (fpclassify (res) == (fpclassify (ref))) { curulp = 0; }
        else { curulp = FLT_MAX; }
    }

    return curulp;
} /* own_s_compute_ulp */

/**
 * @brief Complex single precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static float own_c_compute_ulp (float complex res, double complex ref )
{
    return fmaxf (own_s_compute_ulp(creal(res), creal(ref)), own_s_compute_ulp(cimag(res), cimag(ref)));
}

/**
 * @brief Double precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static double own_d_compute_ulp (double res, double ref )
{
    int ex        = 0;
    double den    = 1.0;
    double curulp = 0.0;
    if (isfinite (ref))
    {
        frexp (ref, &ex);                                      /* ex: integral power of two of ref */
        den = ldexp (1.0, ex - 53);                            /* den: ulp's denominator 2^(ex-p+1) */
        den = (den == 0.0)? 0x1.p-1074 : den;                  /* if den=0 then replace by EPS to avoid divbyzero */
        curulp = (double)fabs ((((double)(res) - ref)) / den); /* |res-ref|/2^(ex-53) */
        curulp = isfinite (curulp)?curulp : DBL_MAX;           /* replace infinite ulp by big finite double number */
    }
    else
    {
        if (fpclassify (res) == (fpclassify (ref))) { curulp = 0; }
        else { curulp = DBL_MAX; }
    }

    return curulp;
} /* own_d_compute_ulp */

/**
 * @brief Complex double precision ULP calculation
 *
 * Computes ULP between result and reference value
 *
 * @param[in] res          Computed result
 * @param[in] ref          Reference result
 * @return                 Calculated ULP
 *
 */
static double own_z_compute_ulp (double complex res, double complex ref )
{
    return fmax (own_d_compute_ulp(creal(res), creal(ref)), own_d_compute_ulp(cimag(res), cimag(ref)));
}

/**
 * @brief Fill inputs
 *
 * Fills input vectors by random numbers
 *
 * @param[in] layout       Function arguments layout
 * @param[in] beg          Begin of input ranges
 * @param[in] end          End of input ranges
 * @param[out] in          Input arrays
 *
 */
static void own_fill_input (int layout, double beg, double end, VmInputData* in)
{
    srand(777);
    for (int i = 0; i < length; i = i + 1)
    {
        /**
         * Generate random numbers in [beg, end) range
         */
        double v1 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v2 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v3 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));
        double v4 = (beg + (end - beg) * (double)(rand ()) / (double)(RAND_MAX));

        in->darg1[i] = v1;
        in->darg2[i] = v2;
        in->sarg1[i] = (float)v1;
        in->sarg2[i] = (float)v2;

        in->zarg1[i] = v1 + I * v3;
        in->zarg2[i] = v2 + I * v4;
        in->carg1[i] = (float)v1 + I * (float)v3;
        in->carg2[i] = (float)v2 + I * (float)v4;
    } /* for (int i = 0; i < length; i++) */
} /* own_fill_input */

/**
 * @brief Fill reference vectors
 *
 * Compute reference results
 *
 * @param[in] layout       Function arguments layout
 * @param[in] reffunc      Real reference function
 * @param[in] creffunc     Complex reference function
 * @param[out] in          Input and reference results arrays
 *
 */
static void own_fill_reference (int layout, void* reffunc, void* creffunc, VmInputData* in)
{
    int has_real    = (reffunc != NULL)?1:0, has_complex = (creffunc != NULL)?1:0;
    for (int i = 0; i < length; i = i + 1)
    {
        if (layout == kVtoV)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefVtoV)creffunc) ((double complex) (in->carg1[i]));
                in->zref1[i] = ((CRefVtoV)creffunc) ( in->zarg1[i]);
            }
        }
        else if (layout == kVCtoVR)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                /* Use complex array containers to accept real results */
                ((double*)(in->cref1))[i] = ((CRefCtoR)creffunc) ((double complex) (in->carg1[i]));
                ((double*)(in->zref1))[i] = ((CRefCtoR)creffunc) (in->zarg1[i]);
            }
        }
        else if (layout == kVRtoVC)
        {
            if (has_real)
            {
                /* Use real arrays for kVRtoVC functions */
                in->sref1[i] = ((RefVtoV)reffunc) (in->sarg1[i]);
                in->dref1[i] = ((RefVtoV)reffunc) (in->darg1[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefRtoC)creffunc) (in->sarg1[i]);
                in->zref1[i] = ((CRefRtoC)creffunc) (in->darg1[i]);
            }
        }
        else if (layout == kVVtoV)
        {
            if (has_real)
            {
                in->sref1[i] = ((RefVVtoV)reffunc) (in->sarg1[i], in->sarg2[i]);
                in->dref1[i] = ((RefVVtoV)reffunc) (in->darg1[i], in->darg2[i]);
            }
            if (has_complex)
            {
                in->cref1[i] = ((CRefVVtoV)creffunc) ((double complex) (in->carg1[i]), (double complex) (in->carg2[i]));
                in->zref1[i] = ((CRefVVtoV)creffunc) (in->zarg1[i], in->zarg2[i]);
            }
        }
        else if (layout == kVXtoV)
        {
            float  sfixed = (float)fixed;
            double dfixed = (double)fixed;
            if (has_real)
            {
                in->sref1[i] = ((RefVVtoV)reffunc) (in->sarg1[i], sfixed);
                in->dref1[i] = ((RefVVtoV)reffunc) (in->darg1[i], dfixed);
            }
            if (has_complex)
            {
                double complex cfixed = sfixed + I * sfixed;
                double complex zfixed = dfixed + I * dfixed;
                in->cref1[i] = ((CRefVVtoV)creffunc) ((double complex) (in->carg1[i]), cfixed);
                in->zref1[i] = ((CRefVVtoV)creffunc) (in->zarg1[i], zfixed);
            }
        }
        else if (layout == kVtoVV)
        {
            ((RefVtoVV)reffunc) (in->sarg1[i], &(in->sref1[i]), &(in->sref2[i]));
            ((RefVtoVV)reffunc) (in->darg1[i], &(in->dref1[i]), &(in->dref2[i]));
        }
        else if (layout == kVVXtoV)
        {
            float  sfixed = (float)fixed;
            double dfixed = (double)fixed;
            in->sref1[i] = ((RefVVXtoV)reffunc) (in->sarg1[i], in->sarg2[i], sfixed, sfixed, sfixed, sfixed);
            in->dref1[i] = ((RefVVXtoV)reffunc) (in->darg1[i], in->darg2[i], dfixed, dfixed, dfixed, dfixed);
        }
    } /* for(int i = 0; i < length; i++) */
} /* own_fill_reference */

/**
 * @brief Full VM function name for printout
 *
 * Construct full VM function name with precision, api and accuracy suffices
 *
 * @param[out] buff        Pointer to output string buffer
 * @param[in] maxlen       Maximum string length
 * @param[in] fname        Base function name
 * @param[in] prec         Precision
 * @param[in] api          API variant
 * @param[in] acc          Accuracy
 * @return                 Pointer to constructed name
 *
 */
static char* own_full_name(char* buff, int maxlen, char* fname, int prec, int api, int acc)
{
    const char* api_prefix[]    = {"v", "vm", "v", "vm"};
    const char* api_suffix[]    = {"", "", "I", "I"};
    const char* prec_suffix[]   = {"s", "d", "c", "z"};
    const char* acc_suffix[]    = {"HA", "LA", "EP"};

    snprintf (buff, maxlen, "%s%s%s%s_%s ",
              api_prefix[api], prec_suffix[prec], fname, api_suffix[api], acc_suffix[acc]);

    return buff;
} /* own_full_name */

/**
 * @brief Printout ULP value
 *
 * Print arguments, results and ULP difference.
 *
 * @param[in] fname        Function name
 * @param[in] prec         Function precision
 * @param[in] layout       Function arguments layout
 * @param[in] acc          Function accuracy
 * @param[in] api          Function API variant
 * @param[in] idx          Vector index
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 *
 */
static void own_print_ulp (char* fname, int prec, int layout, int acc, int api, int idx, double ulp, VmInputData* in, VmOutputData* out)
{
    char strbuff[NAME_LEN] = {0};

    if (prec == kSP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name (strbuff, NAME_LEN, fname, prec, api, acc), idx);
        fprintf (stderr, "%.3g {%a}", in->sarg1[idx], in->sarg1[idx]);
        if ((layout == kVVtoV) || (layout == kVVXtoV)) fprintf (stderr, ", %.3g {%a}", in->sarg2[idx], in->sarg2[idx]);
        fprintf (stderr, ") = %.3g {%a}", out->sres1[api][idx], out->sres1[api][idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3g {%a}", out->sres2[api][idx], out->sres2[api][idx]);
        fprintf (stderr, ", expected = %.3lg {%la}", in->sref1[idx], in->sref1[idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", in->sref2[idx], in->sref2[idx]);
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }
    else if (prec == kDP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name (strbuff, NAME_LEN, fname, prec, api, acc), idx);
        fprintf (stderr, "%.3lg {%la}",in->darg1[idx],in->darg1[idx]);
        if ((layout == kVVtoV) || (layout == kVVXtoV)) fprintf (stderr, ", %.3lg {%la}", in->darg2[idx], in->darg2[idx]);
        fprintf (stderr, ") = %.3lg {%la}", out->dres1[api][idx], out->dres1[api][idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", out->dres2[api][idx], out->dres2[api][idx]);
        fprintf (stderr, ", expected = %.3lg {%la}", in->dref1[idx], in->dref1[idx]);
        if (layout == kVtoVV) fprintf (stderr, ", %.3lg {%la}", in->dref2[idx], in->dref2[idx]);
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
   }
    else if (prec == kCP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name(strbuff, NAME_LEN, fname, prec, api, acc), idx);
        if (layout == kVRtoVC) fprintf (stderr, "%.3g {%a}", in->sarg1[idx], in->sarg1[idx]);
        else                  fprintf (stderr, "%.3g+i*%.3g {%a+i*%a}",
                              creal(in->carg1[idx]), cimag(in->carg1[idx]), creal(in->carg1[idx]), cimag(in->carg1[idx]));
        if (layout == kVVtoV)  fprintf (stderr, ", %.3g+i*%.3g {%a+i*%a}",
                              creal(in->carg2[idx]), cimag(in->carg2[idx]), creal(in->carg2[idx]), cimag(in->carg2[idx]));
        if (layout == kVCtoVR) fprintf (stderr, ") = %.3g {%a}", ((float*)(out->cres1[api]))[idx], ((float*)(out->cres1[api]))[idx]);
        else                  fprintf (stderr, ") = %.3g+i*%.3g {%a+i*%a}",
                              creal(out->cres1[api][idx]), cimag(out->cres1[api][idx]), creal(out->cres1[api][idx]), cimag(out->cres1[api][idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3g+i*%.3g {%a+i*%a}",
                              creal(out->cres2[api][idx]), cimag(out->cres2[api][idx]), creal(out->cres2[api][idx]), cimag(out->cres2[api][idx]));
        if (layout == kVCtoVR) fprintf (stderr, ", expected = %.3lg {%la}", ((double*)(in->cref1))[idx], ((double*)(in->cref1))[idx]);
        else                  fprintf (stderr, ", expected = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->cref1[idx]), cimag(in->cref1[idx]), creal(in->cref1[idx]), cimag(in->cref1[idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3g+i*%.3g {%la+i*%la}", creal(in->cref2[idx]),
                              cimag(in->cref2[idx]), creal(in->cref2[idx]), cimag(in->cref2[idx]));
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }
    else if (prec == kZP)
    {
        fprintf (stderr, "\t\tULP_OVER_BOUND: %s[%d](", own_full_name(strbuff, NAME_LEN, fname, prec, api, acc), idx);
        if (layout == kVRtoVC) fprintf (stderr, "%.3lg {%la}", in->darg1[idx], in->darg1[idx]);
        else                  fprintf (stderr, "%.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zarg1[idx]), cimag(in->zarg1[idx]), creal(in->zarg1[idx]), cimag(in->zarg1[idx]));
        if (layout == kVVtoV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zarg2[idx]), cimag(in->zarg2[idx]), creal(in->zarg2[idx]), cimag(in->zarg2[idx]));
        if (layout == kVCtoVR) fprintf (stderr, ") = %.3lg {%la}", ((double*)(out->zres1[api]))[idx], ((double*)(out->zres1[api]))[idx]);
        else                  fprintf (stderr, ") = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(out->zres1[api][idx]), cimag(out->zres1[api][idx]), creal(out->zres1[api][idx]), cimag(out->zres1[api][idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(out->zres2[api][idx]), cimag(out->zres2[api][idx]), creal(out->zres2[api][idx]), cimag(out->zres2[api][idx]));
        if (layout == kVCtoVR) fprintf (stderr, ", expected = %.3lg {%la}", ((double*)(in->zref1))[idx], ((double*)(in->zref1))[idx]);
        else                  fprintf (stderr, ", expected = %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zref1[idx]), cimag(in->zref1[idx]), creal(in->zref1[idx]), cimag(in->zref1[idx]));
        if (layout == kVtoVV)  fprintf (stderr, ", %.3lg+i*%.3lg {%la+i*%la}",
                              creal(in->zref2[idx]), cimag(in->zref2[idx]), creal(in->zref2[idx]), cimag(in->zref2[idx]));
        fprintf (stderr, ", ulp = %.3lg\n", ulp);
    }

    fflush (stderr);
    return;
} /* own_print_ulp */

/**
 * @brief Evaluation of one VM functions family
 *
 * Measure accuracy on VM functions family in comparison to reference scalar implementations.
 *
 * @param[in] fname        Function name
 * @param[in] beg          Begin of input ranges
 * @param[in] end          End of input ranges
 * @param[in] vmfunc       Pointer to VM functions launcher
 * @param[in] reffunc      Real reference function
 * @param[in] creffunc     Complex reference function
 * @param[in] layout       Function arguments layout
 * @param[in] in           Input and reference resutl arrays
 * @param[out] out         Output arrays
 * @return                 Total number of errors
 *
 */
static int own_evaluate_func (char* fname, double beg, double end,
                              VmFunc vmfunc, void* reffunc, void* creffunc,
                              int layout, VmInputData* in, VmOutputData* out)
{
    int printed = 0;
    int err = 0, warn = 0;
    int serr  = 0, derr  = 0, cerr  = 0, zerr  = 0,
        swarn = 0, dwarn = 0, cwarn = 0, zwarn = 0;
    float  sulp[kApiNum][kAccNum]    = {0}, culp[kApiNum][kAccNum]    = {0},
           smaxulp[kApiNum][kAccNum] = {0}, cmaxulp[kApiNum][kAccNum] = {0},
           sresulp[kAccNum]          = {0}, cresulp[kAccNum]          = {0};
    double dulp[kApiNum][kAccNum]    = {0}, zulp[kApiNum][kAccNum]    = {0},
           dmaxulp[kApiNum][kAccNum] = {0}, zmaxulp[kApiNum][kAccNum] = {0},
           dresulp[kAccNum]          = {0}, zresulp[kAccNum]          = {0};
    int has_real = (reffunc != NULL)?1:0, has_complex = (creffunc != NULL)?1:0;

    own_fill_input (layout, beg, end, in);
    own_fill_reference (layout, reffunc, creffunc, in);

    for (int a = kHA; a < kAccNum; a = a + 1)
    {
        /* Launch all API's of function family */
        vmfunc (a, in, out);

        for (int v = kFunc; v < kApiNum; v = v + 1)
        {
            printed = 0;

            if (has_real)
            {
                for (int i = 0; i < length; i = i + 1)
                {
                    warn = 0; err = 0;
                    /* Use stride increment for evaluating strided functions */
                    if (((v == kFuncI) || (v == kMFuncI)) && ((i % stride) || (i >= length/stride))) continue;

                    /* Compute ULP */
                    sulp[v][a] = own_s_compute_ulp (out->sres1[v][i], in->sref1[i]);
                    dulp[v][a] = own_d_compute_ulp (out->dres1[v][i], in->dref1[i]);
                    if (layout == kVtoVV)
                    {
                        sulp[v][a] = fmaxf (sulp[v][a], own_s_compute_ulp (out->sres2[v][i], in->sref2[i]));
                        dulp[v][a] = fmax  (dulp[v][a], own_d_compute_ulp (out->dres1[v][i], in->dref1[i]));
                    }

                    /* Check if ULP is greater than allowed */
                    if (sulp[v][a] > s_allowed_ulp[a])
                    {
                        /* Allows HA/LA linearfrac functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if ((layout == kVVXtoV) && (sulp[v][a] < s_allowed_ulp[kEP])) warn = 1;
                        else err = 1;

                        swarn += warn; serr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kSP, layout, a, v, i, sulp[v][a], in, out);
                            printed++;
                        }
                    }

                    /* Check if ULP is greater than allowed */
                    if (dulp[v][a] > d_allowed_ulp[a])
                    {
                        /* Allows HA/LA linearfrac functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if ((layout == kVVXtoV) && (dulp[v][a] < d_allowed_ulp[kEP])) warn = 1;
                        else err = 1;

                        dwarn += warn; derr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kDP, layout, a, v, i, dulp[v][a], in, out);
                            printed++;
                        }
                    }
                    smaxulp[v][a] = fmaxf (smaxulp[v][a], sulp[v][a]);
                    dmaxulp[v][a] = fmax  (dmaxulp[v][a], dulp[v][a]);
                } /* for (int i = 0; i < length; i++) */
                sresulp[a] = fmaxf (sresulp[a], smaxulp[v][a]);
                dresulp[a] = fmax  (dresulp[a], dmaxulp[v][a]);
            } /* if (has_real) */

            if (has_complex)
            {
                for (int i = 0; i < length; i++)
                {
                    /* Use stride increment for evaluating strided functions */
                    if (((v == kFuncI) || (v == kMFuncI)) && ((i % stride) || (i >= length/stride))) continue;

                    if (layout == kVCtoVR)
                    {
                        float  *sres1 = (float*)(out->cres1[v]);
                        double *sref1 = (double*)(in->cref1);
                        double *dres1 = (double*)(out->zres1[v]), *dref1 = (double*)(in->zref1);
                        culp[v][a] = own_s_compute_ulp (sres1[i], sref1[i]);
                        zulp[v][a] = own_d_compute_ulp (dres1[i], dref1[i]);
                    }
                    else
                    {
                        culp[v][a] = own_c_compute_ulp (out->cres1[v][i], in->cref1[i]);
                        zulp[v][a] = own_z_compute_ulp (out->zres1[v][i], in->zref1[i]);
                    }

                    if (culp[v][a] > c_allowed_ulp[a])
                    {
                        /* Allows HA/LA complex functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if (culp[v][a] < c_allowed_ulp[kEP]) warn = 1;
                        else err = 1;

                        cwarn += warn; cerr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kCP, layout, a, v, i, culp[v][a], in, out);
                            printed++;
                        }
                    }
                    if (zulp[v][a] > z_allowed_ulp[a])
                    {
                        /* Allows HA/LA complex functions ULP within EP threshold */
                        /* and report warning if greater than HA/LA limits */
                        if (zulp[v][a] < z_allowed_ulp[kEP]) warn = 1;
                        else err = 1;

                        zwarn += warn; zerr += err;
                        if ((printed < max_printed) && ((print_err && err) || (print_warn && warn)))
                        {
                            own_print_ulp (fname, kZP, layout, a, v, i, zulp[v][a], in, out);
                            printed++;
                        }
                    }
                    cmaxulp[v][a] = fmaxf (cmaxulp[v][a], culp[v][a]);
                    zmaxulp[v][a] = fmax  (zmaxulp[v][a], zulp[v][a]);
                } /* for (int i = 0; i < length; i++) */
                cresulp[a] = fmaxf (cresulp[a], cmaxulp[v][a]);
                zresulp[a] = fmax  (zresulp[a], zmaxulp[v][a]);
            } /* if (has_complex) */
        } /* for (int v = kFunc; v <= kMFuncI; v++) */
    } /* for (int a = kHA; a <= kEP; a++) */

    if (has_real)
    {
        fprintf (stdout, "\ts%-11s, ha:,%7.2g, la:, %7.2g, ep:, %7.2g, %s\n",
                 fname, sresulp[kHA], sresulp[kLA], sresulp[kEP], (serr)?"OVER":(swarn)?"WARN":"NORM");
        fprintf (stdout, "\td%-11s, ha:,%7.2lg, la:, %7.2lg, ep:, %7.2lg, %s\n",
                 fname, dresulp[kHA], dresulp[kLA], dresulp[kEP], (derr)?"OVER":(dwarn)?"WARN":"NORM");
    }
    if (has_complex)
    {
        fprintf (stdout, "\tc%-11s, ha:,%7.2g, la:, %7.2g, ep:, %7.2g, %s\n",
                 fname, cresulp[kHA], cresulp[kLA], cresulp[kEP], (cerr)?"OVER":(cwarn)?"WARN":"NORM");
        fprintf (stdout, "\tz%-11s, ha:,%7.2lg, la:, %7.2lg, ep:, %7.2lg, %s\n",
                 fname, zresulp[kHA], zresulp[kLA], zresulp[kEP], (zerr)?"OVER":(zwarn)?"WARN":"NORM");
    }

    fflush (stdout);
    // Return total number of errors
    return (serr + derr + cerr + zerr);
} /* own_evaluate_func */

/**
 * @brief Provide string description of VML status code
 *
 * vml_status_string provides string description of VML status code st
 *
 * @param[in] st  VML status code
 * @return         const char* with text of corresponding code
 *
 */
static const char* vml_status_string(int st) {

    switch (st) {
        case VML_STATUS_OK: return "VML_STATUS_OK";
        case VML_STATUS_BADSIZE: return "VML_STATUS_BADSIZE";
        case VML_STATUS_BADMEM: return "VML_STATUS_BADMEM";
        case VML_STATUS_ERRDOM: return "VML_STATUS_ERRDOM";
        case VML_STATUS_SING: return "VML_STATUS_SING";
        case VML_STATUS_OVERFLOW: return "VML_STATUS_OVERFLOW";
        case VML_STATUS_UNDERFLOW: return "VML_STATUS_UNDERFLOW";
        case VML_STATUS_ACCURACYWARNING: return "VML_STATUS_ACCURACYWARNING";
    }
    return "VML_STATUS_UNKNOWN";
}

/**
 * @brief Main function for VM API testing
 *
 * Main performs accuracy testing of all VM OMP offload math functions
 *
 * @param[in] argc         Number of arguments
 * @param[in] argv         Pointer to argument strings
 * @return                 0
 *
 */
int main (int argc, char **argv)
{
    /* Total errors */
    int err = 0;

    /* Error satatus */
    int st = VML_STATUS_OK;

    VmInputData  in;
    VmOutputData out;

    fprintf (stdout, "Classic c vm_all_funcs_64: started...\n"); fflush (stdout);

    own_allocate_data (length, &in, &out);

    vmlClearErrStatus();

    fprintf (stdout, "\t===========================================================\n");
    err += own_evaluate_func ("Asin",       -0.9,   0.9,      own_vm_asin,       asin,           casin,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Acos",       -0.9,   0.9,      own_vm_acos,       acos,           cacos,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan",       -10000, 10000,    own_vm_atan,       atan,           catan,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan2",      -10000, 10000,    own_vm_atan2,      atan2,          NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Asinh",      -10000, 10000,    own_vm_asinh,      asinh,          casinh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Acosh",       1.01,  1000,     own_vm_acosh,      acosh,          cacosh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Atanh",      -0.9,   0.9,      own_vm_atanh,      atanh,          catanh,         kVtoV,      &in, &out);
    err += own_evaluate_func ("Sin",        -10,    10,       own_vm_sin,        sin,            csin,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cos",        -10,    10,       own_vm_cos,        cos,            ccos,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tan",        -10,    10,       own_vm_tan,        tan,            ctan,           kVtoV,      &in, &out);
    err += own_evaluate_func ("SinCos",     -10000, 10000,    own_vm_sincos,     own_sincos,     NULL,           kVtoVV,     &in, &out);
    err += own_evaluate_func ("Sinh",       -50,    50,       own_vm_sinh,       sinh,           csinh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Cosh",       -50,    50,       own_vm_cosh,       cosh,           ccosh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Tanh",       -5,     5,        own_vm_tanh,       tanh,           ctanh,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp",        -75,    75,       own_vm_exp,        exp,            cexp,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp2",       -30,    30,       own_vm_exp2,       exp2,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Expm1",      -30,    30,       own_vm_expm1,      expm1,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Ln",          1.01,  100000,   own_vm_ln,         log,            clog,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log2",        1.01,  100000,   own_vm_log2,       log2,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log1p",       0.01,  100000,   own_vm_log1p,      log1p,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Pow",         0.1,   10,       own_vm_pow,        pow,            cpow,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Powr",        0.1,   10,       own_vm_powr,       pow,            NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Powx",        0.1,   10,       own_vm_powx,       pow,            cpow,           kVXtoV,     &in, &out);
    err += own_evaluate_func ("Pow2o3",      0.1,   10,       own_vm_pow2o3,     own_pow2o3,     NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Pow3o2",      0.1,   10,       own_vm_pow3o2,     own_pow3o2,     NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Sqrt",        0.1,   100,      own_vm_sqrt,       sqrt,           csqrt,          kVtoV,      &in, &out);
    err += own_evaluate_func ("Cbrt",        0.1,   10000,    own_vm_cbrt,       cbrt,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("InvSqrt",     0.1,   10000,    own_vm_invsqrt,    own_invsqrt,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("InvCbrt",     0.1,   10000,    own_vm_invcbrt,    own_invcbrt,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Hypot",      -10000, 10000,    own_vm_hypot,      hypot,          NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Erf",        -5,     5,        own_vm_erf,        erf,            NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Erfc",       -2,     5,        own_vm_erfc,       erfc,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("LGamma",      0,     5,        own_vm_lgamma,     lgamma,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("TGamma",      0,     5,        own_vm_tgamma,     tgamma,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("ExpInt1",     0.1,   5,        own_vm_expint1,    own_expint1,    NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Ceil",       -10000, 10000,    own_vm_ceil,       ceil,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Floor",      -10000, 10000,    own_vm_floor,      floor,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Round",      -10000, 10000,    own_vm_round,      round,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Trunc",      -10000, 10000,    own_vm_trunc,      trunc,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Rint",       -10000, 10000,    own_vm_rint,       rint,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("NearbyInt",  -10000, 10000,    own_vm_nearbyint,  nearbyint,      NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Remainder",  -10000, 10000,    own_vm_remainder,  remainder,      NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("LinearFrac", -1000,  1000,     own_vm_linearfrac, own_linearfrac, NULL,           kVVXtoV,    &in, &out);
    err += own_evaluate_func ("Add",        -10000, 10000,    own_vm_add,        own_add,        own_cadd,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sub",        -10000, 10000,    own_vm_sub,        own_sub,        own_csub,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Mul",        -10000, 10000,    own_vm_mul,        own_mul,        own_cmul,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Div",        -10000, 10000,    own_vm_div,        own_div,        own_cdiv,       kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sqr",        -10000, 10000,    own_vm_sqr,        own_sqr,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Inv",        -10000, 10000,    own_vm_inv,        own_inv,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Modf",       -10000, 10000,    own_vm_modf,       own_modf,       NULL,           kVtoVV,     &in, &out);
    err += own_evaluate_func ("Fmod",       -10000, 10000,    own_vm_fmod,       fmod,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fdim",       -10000, 10000,    own_vm_fdim,       fdim,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fmax",       -10000, 10000,    own_vm_fmax,       fmax,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Fmin",       -10000, 10000,    own_vm_fmin,       fmin,           NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("MaxMag",     -10000, 10000,    own_vm_maxmag,     own_maxmag,     NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("MinMag",     -10000, 10000,    own_vm_minmag,     own_minmag,     NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("NextAfter",  -10000, 10000,    own_vm_nextafter,  nextafter,      NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("CopySign",   -10000, 10000,    own_vm_copysign,   copysign,       NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Frac",       -10000, 10000,    own_vm_frac,       own_frac,       NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Conj",       -10000, 10000,    own_vm_conj,       NULL,           conj,           kVtoV,      &in, &out);
    err += own_evaluate_func ("MulByConj",  -10000, 10000,    own_vm_mulbyconj,  NULL,           own_cmulbyconj, kVVtoV,     &in, &out);
    err += own_evaluate_func ("CIS",        -10000, 10000,    own_vm_cis,        NULL,           own_cis,        kVRtoVC,    &in, &out);
    err += own_evaluate_func ("Arg",        -10000, 10000,    own_vm_arg,        NULL,           own_carg,       kVCtoVR,    &in, &out);
    err += own_evaluate_func ("Abs",        -10000, 10000,    own_vm_abs,        fabs,           own_cabs,       kVCtoVR,    &in, &out);
    /* Functions with Intel-specific reference LIBM implementations */
#if (defined __INTEL_COMPILER) || (defined __INTEL_LLVM_COMPILER)
    err += own_evaluate_func ("Asinpi",     -0.9,   0.9,      own_vm_asinpi,     asinpi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Acospi",     -0.9,   0.9,      own_vm_acospi,     acospi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Atanpi",     -10000, 10000,    own_vm_atanpi,     atanpi,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Atan2pi",    -10000, 10000,    own_vm_atan2pi,    atan2pi,        NULL,           kVVtoV,     &in, &out);
    err += own_evaluate_func ("Sind",       -10000, 10000,    own_vm_sind,       sind,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cosd",       -10000, 10000,    own_vm_cosd,       cosd,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tand",       -10000, 10000,    own_vm_tand,       tand,           NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Sinpi",      -10000, 10000,    own_vm_sinpi,      sinpi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Cospi",      -10000, 10000,    own_vm_cospi,      cospi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Tanpi",      -10000, 10000,    own_vm_tanpi,      tanpi,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Exp10",      -30,    30,       own_vm_exp10,      exp10,          NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("Log10",       1.01,  100,      own_vm_log10,      log10,          clog10,         kVtoV,      &in, &out);
    err += own_evaluate_func ("ErfInv",     -0.9,   0.9,      own_vm_erfinv,     erfinv,         NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("ErfcInv",    -0.1,   1.9,      own_vm_erfcinv,    erfcinv,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("CdfNorm",    -4,     4,        own_vm_cdfnorm,    cdfnorm,        NULL,           kVtoV,      &in, &out);
    err += own_evaluate_func ("CdfNormInv", -0.1,   0.9,      own_vm_cdfnorminv, cdfnorminv,     NULL,           kVtoV,      &in, &out);
#endif

    st   = vmlGetErrStatus();
    fprintf (stdout, "\t===========================================================\n");
    fprintf (stdout, "Classic vm_all_funcs_64: status: %s[%s], accuracy: %s\n\n",
        vml_status_string(st),
        (st >= VML_STATUS_OK) ? "expected" : "unexpected",
        err > 0 ? "over bounds" : "normal"
    );

    own_deallocate_data (&in, &out);

    return 0;
} /* main */

