/**
 * @brief Hardware abstraction layer for Intel based MacBooks
 *
 * This module contains all hardware related functions for the new
 * intel based machines from Apple that are used by the high level
 * logic modules to do their job.
 *
 * This module is needed for all intel based Apple MacBooks.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation
 * (http://www.gnu.org/licenses/gpl.html)
 *
 * @file    src/module_imac.c
 * @author  Stefan Bruda <bruda@cs.ubishops.ca> 
 * @author  Matthias Grimm <matthias.grimm@users.sourceforge.net>
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/io.h>
#ifdef WITH_SMBIOS
#  include <smbios/SystemInfo.h>
#endif

#include <pbb.h>

#include "gettext_macros.h"
#include "input_manager.h"
#include "class_config.h"
#include "module_imac.h"
#include "support.h"
#include "debug.h"

#define SECTION "MODULE IMAC"

#if defined(DEBUG) && SIMU_I386
unsigned char
inb (unsigned long from)
{
	return 0;
}

void
outb (unsigned char byte, unsigned long from)
{
}

int
ioperm (unsigned long from, unsigned long num, int turn_on)
{
	return 0;
}
#endif

struct moddata_imac {
	GString *identity; /* Identity string of this laptop */
	machine_t machine; /* Identifier of machine */
#if defined(DEBUG) && SIMU_AMBIENT
	int ambient;
#endif
} modbase_imac;

/**
 * @brief Initalizes the module imac
 *
 * This function initializes the module imac. The data structure will
 * be initialized and the hardware status is read.
 *
 * Immideately after this initialisation the module is able to and 
 * will answer upon requests.
 *
 * @return  error code, zero if the module is ready to operate
 */
int
imac_init ()
{
	struct moddata_imac *base = &modbase_imac;
	int val;

#if defined(DEBUG) && SIMU_AMBIENT
	base->ambient  =  0;
#endif
	base->identity = g_string_new(NULL);

	base->machine  = getMachineType();
	g_string_printf (base->identity, "%s", _(getMachineName(base->machine)));

#ifdef DEBUG
	print_msg(PBB_INFO, "DBG: Machine: %x\n", base->machine);

	if (haveKBDIllumination()) {
		if ((check_devorfile (SYSFS_KEYBLIGHT, TYPE_FILE)) == 0)
			print_msg(PBB_INFO, "DBG: SysFS support for keyboard illumination found.\n");
		else
			print_msg(PBB_INFO, "DBG: Using native SMC interface for keyboard illumination.\n");
	} else
		print_msg(PBB_INFO, "DBG: No keyboard illumination available on this MacBook.\n");

	if (haveAmbient()) {
		if ((check_devorfile (SYSFS_AMBIENT, TYPE_FILE)) == 0)
			print_msg(PBB_INFO, "DBG: SysFS support for ambient light sensor found.\n");
		else
			print_msg(PBB_INFO, "DBG: Using native SMC interface for ambient light sensor.\n");
	} else
		print_msg(PBB_INFO, "DBG: No ambient light sensor available on this MacBook.\n");
#endif

	register_function (QUERYQUEUE, imac_query);
	register_function (CONFIGQUEUE, imac_configure);
	return 0;
}

/**
 * @brief  Frees all ressources allocated from module imac
 *
 * This function is usually called on program termination. It
 * frees all allocated ressources and closes all open file
 * handles.
 *
 * @return  always zero
 */
void
imac_exit ()
{
}

void
imac_query (struct tagitem *taglist)
{
	imac_handle_tags (MODE_QUERY, taglist);
}

void
imac_configure (struct tagitem *taglist)
{
	imac_handle_tags (MODE_CONFIG, taglist);
}

void
imac_handle_tags (int cfgure, struct tagitem *taglist)
{
	struct moddata_imac *base = &modbase_imac;

	while (taglist->tag != TAG_END) {
		switch (taglist->tag) {
		case TAG_IDENTITY:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = (long) base->identity->str;
			break;
		case TAG_AMBIENTLIGHT:
			if (!haveAmbient())
				tagerror (taglist, E_NOSUPPORT);
			else if (cfgure)
#if defined(DEBUG) && SIMU_AMBIENT
				base->ambient = taglist->data;
#else
				tagerror (taglist, E_NOWRITE);
#endif
			else 
				taglist->data = getRawAmbient();
			break;
		case TAG_AMBIENTLIGHTMAX:
			if (cfgure)	tagerror (taglist, E_NOWRITE);
			else		taglist->data = AMBIENTLIGHTMAX;
			break;
		case TAG_KEYBLIGHTLEVEL:   /* private tag */
			if (cfgure)
				setKBDIllumination (taglist->data);
			break;
		case TAG_KEYBLIGHTMAX:     /* private tag */
			if (cfgure)	;
			else		taglist->data = KEYBLIGHTMAX;
			break;
		case TAG_SYSINFOSLAVE:
			if (cfgure) ;
			else        taglist->data |= SYSINFO_PB_USB_KBD | SYSINFO_PB_USB_TPAD;
		}
		taglist++;
	}
}

machine_t
getMachineType ()
{
#ifndef WITH_SMBIOS
	/* We can't probe for the machine type if the SMBios library
	 * is not installed. Nevertheless we assume that this machine
	 * is a MacBook but MacBook Pro features will be disabled
	 */
	return CMTYPE(TYPE_MACBOOK,0,0);
#else
	const char *prop;
	char *p;
	int type, ver, rev;

	type = ver = rev = 0;

	if ((prop = SMBIOSGetSystemName())) {

		if (strncmp("MacBookPro", prop, 10) == 0) {
			type = TYPE_MACBOOKPRO;
			if ((p = strchr(prop, ',')) != NULL) {
				ver = atoi(&prop[10]) & 0xf;
				rev = atoi(p+1) & 0xf;
			}
		} else if (strncmp("MacBook", prop, 7) == 0) {
			type = TYPE_MACBOOK;
			if ((p = strchr(prop, ',')) != NULL) {
				ver = atoi(&prop[7]) & 0xf;
				rev = atoi(p+1) & 0xf;
			}
		}

	    SMBIOSFreeMemory(prop);
	}
	return CMTYPE(type,ver,rev);
#endif
}

const char*
getMachineName(machine_t mid)
{
	unsigned int type, ver, rev;

	static char *macbooks[3][10] = {
	{ N_("Unknown MacBook"),
	  NULL, NULL, NULL, NULL, NULL, NULL,
	  NULL, NULL, NULL },
	{ NULL,
	  "MacBook Core Duo (May 2006)",           /* 1,1 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "MacBook Core2 Duo (Nov 2006)",          /* 2,1 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL }
	};

	static char *macbookpros[3][10] = {
	{ N_("Unknown MacBook Pro"),
	  NULL, NULL, NULL, NULL, NULL, NULL,
	  NULL, NULL, NULL },
	{ NULL,
	  "MacBook Pro 15\" Cor22e Duo (Jan 2006)",  /* 1,1 */
	  "MacBook Pro 17\" Core Duo (Apr 2006)",  /* 1,2 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL },
	{ NULL,
	  "MacBook Pro 17\" Core2 Duo (Oct 2006)", /* 2,1 */
	  "MacBook Pro 15\" Core2 Duo (Oct 2006)", /* 2,2 */
	  NULL, NULL, NULL, NULL, NULL, NULL, NULL }
	};
	
	type = MTYPE (mid);
	ver  = MVERSION (mid);
	rev  = MREVISION (mid);

	if (type == TYPE_MACBOOK) {
		if (ver < 3 && rev < 10 && macbooks[ver][rev] != NULL)
			return macbooks[ver][rev];
	} else if (type == TYPE_MACBOOKPRO) {
		if (ver < 3 && rev < 10 && macbookpros[ver][rev] != NULL)
			return macbookpros[ver][rev];
	}

	return N_("Unknown Machine");
}

/* -------------------------------  MISCELLANEOUS FUNCTIONS ----------------------------------- */

gboolean
haveAmbient()
{
	struct moddata_imac *base = &modbase_imac;
	return MTYPE(base->machine) == TYPE_MACBOOKPRO ? TRUE : FALSE;
}

/**
 * @brief Get the raw ambient light sensor value from hardware
 *
 * This function read the ambient sensor from hardware. Two
 * interfaces are supported:
 *   @li  sysfs
 *   @li  direct SMC communication
 *
 * The returned sensor values are checked for boundaries and if
 * any value is suspect the fuction return -1. In this case the
 * suspect is guilty until its innocence is proven.
 *
 * The MacBook Pros have two independent ambient light sensors.
 * To reduce flickering through shadows or spot lights at one of
 * the sensors the sensor with the highest value is returned.
 *
 * The maximum ambient brightness is 255 
 *
 * @return  The raw ambient light value or -1 if something went
 *          wrong and the value is invalid.
 */
int
getRawAmbient ()
{
	FILE *fd;
	unsigned char buffer[16], *p, *buf;
	int left = -1, right = -1;

#if defined(DEBUG) && SIMU_AMBIENT
	struct moddata_imac *base = &modbase_imac;

	left = (base->ambient >> 8) & 0xFF;
	right = base->ambient & 0xFF;
#else
	/* return immediately, if no ambient light sensor is available */
	if (haveAmbient()) {
		if ((fd = fopen(SYSFS_AMBIENT, "r"))) {
			if ((buf = fgets(buffer, sizeof(buffer), fd))) {
				if ((p = strchr(buf, ','))) {
					*p++ = '\0';
					right = atoi (p);
					left  = atoi (buf+1);
				}
			}
			fclose(fd);
		} else if (ioperm(0x300, 0x304, 1) == 0) {
			if (readSMCKey(SMC_LIGHT_SENSOR_LEFT_KEY, 6, buffer))
				left = buffer[2];
			if (readSMCKey(SMC_LIGHT_SENSOR_RIGHT_KEY, 6, buffer))
				right = buffer[2];
		}
	}
#endif
	if (left  < 0 || left  > 255) return -1;
	if (right < 0 || right > 255) return -1;
	return left > right ? left : right;
}

gboolean
haveKBDIllumination()
{
	struct moddata_imac *base = &modbase_imac;
	return MTYPE(base->machine) == TYPE_MACBOOKPRO ? TRUE : FALSE;
}

void
setKBDIllumination (unsigned short level)
{
	FILE *fd;
	unsigned char buffer[8];

	/* return immediately, if keyboard illumination is not supported */
	if (!haveKBDIllumination()) return;

	if ((fd = fopen(SYSFS_KEYBLIGHT, "w"))) {
		if ((snprintf(buffer, sizeof(buffer), "%d", level)) < sizeof(buffer))
			fputs(buffer, fd);
		fclose(fd);
		return;
	}
	
	if (ioperm(0x300, 0x304, 1) < 0)
		return;

	buffer[0] = level & 0xFF;
	buffer[1] = 0x00;
	writeSMCKey(SMC_BACKLIGHT_KEY, 2, buffer);	
}

/************************ Apple SMC interface *********************/
static struct timeval lasttv;
static struct timeval newtv;

void inline ssleep(const int usec) {
   gettimeofday(&lasttv, NULL);
   while (1) {
      gettimeofday(&newtv, NULL);
      if (((newtv.tv_usec - lasttv.tv_usec) + ((newtv.tv_sec - lasttv.tv_sec)*1000000)) > usec) {
         break;
      }
   }
}

unsigned char
get_status()
{
	return inb(0x304);
}

int
waitfree(char num)
{
	char c, pc = -1;
	int retry = 100;

	while (((c = get_status())&0x0F) != num && retry) {
		ssleep(10);
		retry--;
		if (pc != c) {
			pc = c;
		}
	}
	if (retry == 0) {
#if defined(DEBUG)
//		printf("Waitfree failed %x != %x.\n", c, num);
#endif
		return 0;
	}
	return 1;
}

int
writeSMCKey(char* key, char len, unsigned char* buffer)
{
	int i;

	outb(0x11, 0x304);
	if (!waitfree(0x0c)) return 0;
	
	for (i = 0; i < 4; i++) {
		outb(key[i], 0x300);
		if (!waitfree(0x04)) return 0;
	}

	outb(len, 0x300);
	for (i = 0; i < len; i++) {
		if (!waitfree(0x04)) return 0;
		outb(buffer[i], 0x300);
	}
	return 1;
}

int
readSMCKey(char* key, char len, unsigned char* buffer)
{
	int i;
	unsigned char c;

	outb(0x10, 0x304);
	if (!waitfree(0x0c)) return 0;
	
	for (i = 0; i < 4; i++) {
		outb(key[i], 0x300);
		if (!waitfree(0x04)) return 0;
	}

	outb(len, 0x300);

	for (i = 0; i < len; i++) {
		if (!waitfree(0x05)) return 0;
		c = inb(0x300);
		buffer[i] = c;
	}
	return 1;
}


