/*###############################################################################
# Linux Management Providers (LMP), Sensors provider package
# Copyright (C) 2009 Shakhrom RUSTAMOV <shahrombek@gmail.com>
#
# This program is being developed under the "OpenDRIM" project.
# The "OpenDRIM" project web page: http://opendrim.sourceforge.net
# The "OpenDRIM" project mailing list: opendrim@googlegroups.com
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; version 2
# of the License.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
#################################################################################

#################################################################################
# To contributors, please leave your contact information in this section
# AND comment your changes in the source code.
#
# Modified by 2009 Guillaume BOTTEX, ETRI <guillaumebottex@etri.re.kr>
###############################################################################*/

#include "Sensors_Common.h"

string CPU_printProcessors(const vector<_processor_topology>& processors) {
	_E_;
	string processors_string;
	for (unsigned long i = 0; i < processors.size(); i ++) {
		
		processors_string += "Processor " + CF_intToStr(i) + ":\n";
		
		for (unsigned long j = 0; j < processors[i].processor_cores.size(); j++) {
			
			processors_string += "   Core " + CF_intToStr(j) + ":\n";
			processors_string += "      nb_hardware_thread: " + CF_intToStr(processors[i].processor_cores[j].nb_hardware_thread) + "\n";
			
			for (unsigned long k = 0; k < processors[i].processor_cores[j].nb_hardware_thread; k++) {
				processors_string += "      hardware_thread_load_average: " + CF_intToStr(processors[i].processor_cores[j].hardware_thread_load_averages[k]) + "\n";
				processors_string += "      previous_hardware_thread_idle_cputime: " + CF_intToStr(processors[i].processor_cores[j].previous_hardware_thread_idle_cputimes[k]) + "\n";
				processors_string += "      previous_hardware_thread_total_cputime: " + CF_intToStr(processors[i].processor_cores[j].previous_hardware_thread_total_cputimes[k]) + "\n";
			}
			processors_string += "      associated_cache_memory: " + CF_boolToStr(processors[i].processor_cores[j].associated_cache_memory) + "\n";
			processors_string += "      cache_memory_size: " + CF_intToStr(processors[i].processor_cores[j].cache_memory_size) + "\n";
			processors_string += "      current_freq: " + CF_intToStr(processors[i].processor_cores[j].current_freq) + "\n";
			processors_string += "      load_average: " + CF_intToStr(processors[i].processor_cores[j].load_average) + "\n";
		}
		processors_string += "      associated_cache_memory: " + CF_boolToStr(processors[i].associated_cache_memory) + "\n";
		processors_string += "      cache_memory_size: " + CF_intToStr(processors[i].cache_memory_size) + "\n";
		processors_string += "      family: " + CF_intToStr(processors[i].family) + "\n";
		processors_string += "      model: " + CF_intToStr(processors[i].model) + "\n";
		processors_string += "      stepping: " + CF_intToStr(processors[i].stepping) + "\n";
		processors_string += "      architecture: " + processors[i].architecture + "\n";
		processors_string += "      vendor: " + processors[i].vendor + "\n";
		processors_string += "      name: " + processors[i].name + "\n";
		processors_string += "      physical_address_size: " + CF_intToStr(processors[i].physical_address_size) + "\n";
		processors_string += "      virtual_address_size: " + CF_intToStr(processors[i].virtual_address_size) + "\n";
		processors_string += "      load_average: " + CF_intToStr(processors[i].load_average) + "\n";
		
	}
	_L_;
	return processors_string;
}

void CPU_setCoreToDefault(_processor_core& processor_core) {
	_E_;
	processor_core.nb_hardware_thread = 1;
	processor_core.hardware_thread_load_averages.clear();
	processor_core.previous_hardware_thread_idle_cputimes.clear();
	processor_core.previous_hardware_thread_total_cputimes.clear();
	processor_core.associated_cache_memory = false; // by default cache is associated with the processor
	processor_core.cache_memory_size = 0;
	processor_core.current_freq = 0;
	processor_core.load_average = 0;
	_L_;
}

void CPU_setTopologyToDefault(_processor_topology& processor_topology) {
	_E_;
	processor_topology.processor_cores.clear();
	processor_topology.associated_cache_memory = true; // by default cache is associated with the processor
	processor_topology.cache_memory_size = 0;
	processor_topology.family = 0;
	processor_topology.model = 0;
	processor_topology.stepping = 0;
	processor_topology.architecture = "";
	processor_topology.vendor = "";
	processor_topology.name = "";
	processor_topology.physical_address_size = 0;
	processor_topology.virtual_address_size = 0;
	processor_topology.load_average = 0;
	_L_;
}

void CPU_findProcessorArchitecture(_processor_topology& topology) {
	_E_;
	if (topology.vendor == "AuthenticAMD") {
		for (unsigned long i = 0; i < sizeof(_AuthenticAMD_Processors)/sizeof(_AuthenticAMD_Processors[0]); i++) {
			if (topology.family == atoi(_AuthenticAMD_Processors[i][0]) && topology.model == atoi(_AuthenticAMD_Processors[i][1]) && topology.stepping == atoi(_AuthenticAMD_Processors[i][2])) {
				topology.architecture = _AuthenticAMD_Processors[i][3];
				if (_AuthenticAMD_Processors[i][4] == "shared")
					topology.associated_cache_memory = false;
				break;
			}
		}
	}
	if (topology.vendor == "GenuineIntel") {
		for (unsigned long i = 0; i < sizeof(_GenuineIntel_CPUs)/sizeof(_GenuineIntel_CPUs[0]); i++) {
			if (topology.family == atoi(_GenuineIntel_CPUs[i][0]) && topology.model == atoi(_GenuineIntel_CPUs[i][1]) && topology.stepping == atoi(_GenuineIntel_CPUs[i][2])) {
				topology.architecture = _GenuineIntel_CPUs[i][3];
				if (_GenuineIntel_CPUs[i][4] == "shared")
					topology.associated_cache_memory = false;
				break;
			}
		}
	}
	_L_;
}

int CPU_getProcessors(const string& cpuinfo_path, vector<_processor_topology>& processors, string& errorMessage) {
	_E_;
	processors.clear();
	vector<string> cpuinfo_lines;
	string stdOut, stdErr;
	
	// read the cpu information
	CF_assert(CF_readTextFileToLines(cpuinfo_path, cpuinfo_lines, 0, errorMessage));
	
	unsigned long nb_core_per_processor = 1;
	unsigned long nb_hardware_thread_per_core = 1;
	
	long nb_core_to_go = -1;
	long nb_thread_to_go = -1;
	
	unsigned long siblings = 1;
	unsigned long long cache = 0;
	
	_processor_topology current_topology;
	_processor_core current_core;
	CPU_setTopologyToDefault(current_topology);
	// guessing what the architecture is
	// Note: you can install a 32 bit system on a 64 bit architecture which means this info is not reliable
#if defined(INTEL)
	current_topology.architecture = "x86";
#elif defined (X86_64)
	current_topology.architecture = "x86_64";
#elif defined (IA64)
	current_topology.architecture = "ia64";
#elif defined (S390)
	current_topology.architecture = "s390";
#elif defined (PPC)
	current_topology.architecture = "ppc";
#elif defined (GENERIC)
	current_topology.architecture = "generic";
#endif
	CPU_setCoreToDefault(current_core);
	
	// for each line of the cpuinfo file
	for (unsigned long i=0; i < cpuinfo_lines.size(); i++) {
		// skip empty lines
		if (cpuinfo_lines[i].size() == 0)
			continue;
		// split the line
		vector<string> cpuinfo_line;
		CF_splitTextBySpace(cpuinfo_line, CF_untab(cpuinfo_lines[i]));
		
		// who knows...
		if (cpuinfo_line.size() < 2)
			continue;
		
		// found a processor declaration
		if (cpuinfo_line[0] == "processor") {
			
			if (nb_thread_to_go == 0) {
				// add a core to the topology
				if (!current_topology.associated_cache_memory) {
					current_core.associated_cache_memory = true;
					current_core.cache_memory_size = cache / nb_core_per_processor;
				}
				current_core.nb_hardware_thread = nb_hardware_thread_per_core;
				for (unsigned int j = 0; j < nb_hardware_thread_per_core; j++) {
					current_core.hardware_thread_load_averages.push_back(0);
					current_core.previous_hardware_thread_idle_cputimes.push_back(1);
					current_core.previous_hardware_thread_total_cputimes.push_back(1);
				}
				current_topology.processor_cores.push_back(current_core);
				CPU_setCoreToDefault(current_core);
			}
			
			if (nb_core_to_go == 0 && nb_thread_to_go == 0) {
				// we're done for this processor
				if (current_topology.associated_cache_memory)
					current_topology.cache_memory_size = cache;
				processors.push_back(current_topology);
				// reset the processor
				CPU_setTopologyToDefault(current_topology);
				nb_core_per_processor = 1;
				nb_hardware_thread_per_core = 1;
				nb_core_to_go = -1;
				nb_thread_to_go = -1;
				continue;
			}
			
			// go to the next hardware thread
			if (nb_thread_to_go > 0) {
				nb_thread_to_go--;
				continue;
			}
			
			// go to the next core
			if (nb_thread_to_go == 0 && nb_core_to_go > 0) {
				nb_thread_to_go = nb_hardware_thread_per_core - 1;
				nb_core_to_go--;
				continue;
			}
			
		}
		
		// vendor_id
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "vendor_id") {
			if (cpuinfo_line.size() < 3) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_topology.vendor = cpuinfo_line[2];
			continue;
		}
		
		// cpu family
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "cpu" && cpuinfo_line[1] == "family") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_topology.family = atoi(cpuinfo_line[3].c_str());
			continue;
		}
		
		// model
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "model" && cpuinfo_line[1] == ":") {
			if (cpuinfo_line.size() < 3) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_topology.model = atoi(cpuinfo_line[2].c_str());
			continue;
		}
		
		// model name
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "model" && cpuinfo_line[1] == "name") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			for (unsigned int j = 3; j < cpuinfo_line.size(); j++) {
				current_topology.name += cpuinfo_line[j];
				if (j < cpuinfo_line.size() - 1)
					 current_topology.name += " ";
			}
			continue;
		}
		
		// stepping
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "stepping") {
			if (cpuinfo_line.size() < 3) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_topology.stepping = atoi(cpuinfo_line[2].c_str());
			// we know which processor it is so let's try to find it in our own list
			CPU_findProcessorArchitecture(current_topology);
			continue;
		}
		
		// cpu MHz
		if (cpuinfo_line[0] == "cpu" && cpuinfo_line[1] == "MHz") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_core.current_freq = atof(cpuinfo_line[3].c_str())*1000;
			continue;
		}
		
		// cache size
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "cache" && cpuinfo_line[1] == "size") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			cache = atoll(cpuinfo_line[3].c_str());
			continue;
		}
		
		// siblings
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "siblings") {
			if (cpuinfo_line.size() < 3) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			siblings = atol(cpuinfo_line[2].c_str());
			continue;
		}
		
		// cpu cores
		if (nb_core_to_go == -1 && nb_thread_to_go ==-1 && cpuinfo_line[0] == "cpu" && cpuinfo_line[1] == "cores") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			// information for the current processor (first time we read it... and last)
			nb_core_per_processor = atol(cpuinfo_line[3].c_str());
			nb_hardware_thread_per_core = siblings / nb_core_per_processor;
			// set the number of cores and hardware threads we should read next
			nb_core_to_go = nb_core_per_processor - 1;
			nb_thread_to_go = nb_hardware_thread_per_core - 1;
			continue;
		}
		
		// address sizes
		if (nb_core_to_go == nb_core_per_processor - 1 && nb_thread_to_go == nb_hardware_thread_per_core - 1 && cpuinfo_line[0] == "address" && cpuinfo_line[1] == "sizes") {
			if (cpuinfo_line.size() < 4) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			string rest_of_the_line;
			for (unsigned int j = 3; j < cpuinfo_line.size(); j++) {
				rest_of_the_line += cpuinfo_line[j];
				if (j < cpuinfo_line.size()-1)
					rest_of_the_line += " ";
			}
			vector<string> rest_of_the_line_elements;
			CF_splitText(rest_of_the_line_elements, rest_of_the_line, ',');
			if (rest_of_the_line_elements.size() < 2) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			vector<string> rest_of_the_line_left_element, rest_of_the_line_right_element;
			CF_splitTextBySpace(rest_of_the_line_left_element, rest_of_the_line_elements[0]);
			CF_splitTextBySpace(rest_of_the_line_right_element, rest_of_the_line_elements[1]);
			if (rest_of_the_line_left_element.size() != 3 || rest_of_the_line_right_element.size() != 3) {
				errorMessage = "Wrong format (at line " + CF_intToStr(i) + "): " + (string) cpuinfo_path;
				return FAILED;
			}
			current_topology.physical_address_size = atoi(rest_of_the_line_left_element[0].c_str());
			current_topology.virtual_address_size = atoi(rest_of_the_line_right_element[0].c_str());
			continue;
		}
		
	}
	
	// the last core
	if (!current_topology.associated_cache_memory) {
		current_core.associated_cache_memory = true;
		current_core.cache_memory_size = cache / nb_core_per_processor;
	}
	current_core.nb_hardware_thread = nb_hardware_thread_per_core;
	for (unsigned int j = 0; j < nb_hardware_thread_per_core; j++) {
		current_core.hardware_thread_load_averages.push_back(0);
		current_core.previous_hardware_thread_idle_cputimes.push_back(1);
		current_core.previous_hardware_thread_total_cputimes.push_back(1);
	}
	current_topology.processor_cores.push_back(current_core);
	
	// the last processor
	if (current_topology.associated_cache_memory)
		current_topology.cache_memory_size = cache;
	processors.push_back(current_topology);
	
	_L_;
	return OK;
}

int CPU_getLoadAverages(vector<_processor_topology>& new_processors, const vector<_processor_topology>& old_processors, string& errorMessage) {
	_E_;
	vector<string> proc_stat_lines;
	
	// read the cpu information
	CF_assert(CF_readTextFileToLines("/proc/stat", proc_stat_lines, 0, errorMessage));
	
	unsigned int line_index = 1;
	
	for (unsigned int i = 0; i < new_processors.size(); i++) {
		
		for (unsigned int j = 0; j < new_processors[i].processor_cores.size(); j++) {
			
			for (unsigned int k = 0; k < new_processors[i].processor_cores[j].nb_hardware_thread; k++) {
				
				if (line_index >= proc_stat_lines.size()) {
					errorMessage = "Wrong format (not enough lines): /proc/stat";
					return FAILED;
				}
				
				vector<string> proc_stat_line;
				CF_splitTextBySpace(proc_stat_line, proc_stat_lines[line_index]);
				
				// /proc/stat cputime info line should contain at least 8 elemens
				if (proc_stat_line.size() < 8) {
					errorMessage = "Wrong format (at line " + CF_intToStr(line_index+1) + "): /proc/stat";
					return FAILED;
				}
				
				// get fresh idle and total
				unsigned long long idle = CF_strToULL(proc_stat_line[4]);
				unsigned long long total = 0;
				
				for (unsigned int l = 1; l < proc_stat_line.size(); l++) {
					total += CF_strToULL(proc_stat_line[l]);
				}
				
				unsigned long long delta_idle = idle - old_processors[i].processor_cores[j].previous_hardware_thread_idle_cputimes[k];
				unsigned long long delta_total = total - old_processors[i].processor_cores[j].previous_hardware_thread_total_cputimes[k];
				
				// compute the new load average for this hardware thread
				if (delta_total != 0)
					new_processors[i].processor_cores[j].hardware_thread_load_averages[k] = (unsigned short) ((delta_total - delta_idle)*100/delta_total);
				else {
					if (delta_total - delta_idle == 0)
						new_processors[i].processor_cores[j].hardware_thread_load_averages[k] = 0;
					else
						new_processors[i].processor_cores[j].hardware_thread_load_averages[k] = 100;
				}
				
				// set the fresh idle and total
				new_processors[i].processor_cores[j].previous_hardware_thread_idle_cputimes[k] = idle;
				new_processors[i].processor_cores[j].previous_hardware_thread_total_cputimes[k] = total;
				
				line_index++;
				
			}
			
			unsigned long long load_average = 0;
			
			for (unsigned int k = 0; k < new_processors[i].processor_cores[j].nb_hardware_thread; k++)
				load_average += new_processors[i].processor_cores[j].hardware_thread_load_averages[k];
			
			new_processors[i].processor_cores[j].load_average = (load_average/(new_processors[i].processor_cores[j].nb_hardware_thread));
			
		}
		
		unsigned long long load_average = 0;
		
		for (unsigned int j = 0; j < new_processors[i].processor_cores.size(); j++)
			load_average += new_processors[i].processor_cores[j].load_average;
		
		new_processors[i].load_average = (load_average/(new_processors[i].processor_cores.size()));
		
	}
	
	_L_;
	return OK;
}
