#ifndef MODEL_H
#define MODEL_H

#include <stdio.h>
#include <stdlib.h>
#include <QtCore/QObject>
#include <QtCore/QMutex>
#include <QtCore/QSemaphore>
#include <QtCore/QWaitCondition>
#include <QtWidgets/QTreeWidgetItem>
#include "math.h"
#include "voicedata.h"

#define MAX_PARAMS 10
#define MAX_SPLIT 9
#define MAX_SCENES 2
#define MAX_SOURCES 4
#define MAX_HARMONIC_GROUPS 8
#define MAX_OSC 5
#define MAX_SCALES 80
#define MAX_POLY 128
#define STEP 64
#define FILTER_LEN 24
#define SUBHARMONIC_LEN 32
#define MAX_LEVEL 2147483647
#define MAX_FRAMES 16384
#define NUM_CONTROLPOINTS 7

const double freqConst = 8.1757989;
const double log2div12 = log(2.0) / 12.0;

enum tokenCodeEnum { MasterVolume, PitchLfoFreq, PitchLfoDepth, Morphing, Compressor, Panning, PanMode, EvenOdd, Noise,
                     NoiseSubHarmonicMode, NoiseFreqDiffusion, Osc3Mode, Osc3FreqDiffusion, Detune, VolumeTracking, VelocityDepth, 
                     offsetDetune, SpectrumOffset, SpectrumSpacing, SpectrumOffsetIndex, SpectrumSpacingIndex, HarmonicSplit,
                     Scale, ControlPointY, H, FilterMix, FilterMixSwitch, Cutoff, Spacing, KeyTracking, FilterDepth, FilterCutoffIndex,
                     FilterSpacingIndex, FilterTrackingIndex, F, FilterMod, FilterModSwitch, TEXT, NONE };

typedef struct controlObject {
  long uid;
  QTreeWidgetItem *treeWidgetItem;
  QString name;
  int objectType; 
  int objectIndex;
  int groupIndex;
  int modDepth;
  bool full, inverse;
} controlObject;

typedef struct midiController {
  long uid;
  QTreeWidgetItem *treeWidgetItem;
  QList<controlObject> connectedObjects;
  unsigned char eventChannel;
  unsigned int eventParam;
  signed int value;
  int type;
} midiController;

typedef struct groupSwitch {
  QList<midiController> connectedControllers;
  unsigned char eventChannel;
  unsigned int eventParam;
  signed int value;
  int type;
  int radioGroupID, radioGroupIndex;
} groupSwitch;

struct MidiEvent {
  long frame;
  int type;
  int ch;
  int index;
  int val;
};

typedef QList<MidiEvent> MidiEventList;

typedef struct scalaItem {
  double value;
  bool isCent;
} scalaItem;

class Model : public QObject
{
  Q_OBJECT
  
  private:
    long uid;
    QList<midiController> midiControllerList;
    QList<controlObject> controlObjectList;
    QList<groupSwitch> groupSwitchList;
    QList<scalaItem> scalaData;
    MidiEventList currentEvents;
    int frame;
    VoiceData *voiceData[MAX_POLY];
    int currentVoiceDataIndex;
    int currentGroupIndex;
    int editStart, editEnd;
    int numHarmonics;
    int harmonicSplit[MAX_SPLIT];
    double *harmonics[MAX_PARAMS];
    double *scaledHarmonics[MAX_PARAMS];
    double scaleLin[MAX_PARAMS];
    double scaleExp[MAX_PARAMS];
    double minScale[MAX_PARAMS];
    double maxScale[MAX_PARAMS];
    double scale[MAX_SCALES];
    double scalaFreq[128];
    double scalaNote[128];
    double max;
    bool interpolate;
    double masterVolume, pitchBend, lfo, dlfo, lfoFreq, lfoDepth;
    double delta[6];
    double evenOdd[4];
    double noiseBandwidth;
    double morphing;
    double linearLevel, transitionWidth, compression, antialiasing;
    int lfoState;
    int editGroup;
    int rate, bufSize;
    int poly;
    int max_noise_step;
    voiceType *voices;
    envelopeType *envelopes[MAX_POLY];
    bool synthesisFlag;
    double scalaBaseFreq;
    int scalaBaseNote;
    bool scalaMode;
    bool modeToggleState, sceneToggleState;
    int sourceSwitchState;
    int filterSwitchState;
    int groupSwitchState[2];
    bool filterCurveMode;
    
  public:
    QMutex voiceMutex, synthMutex;
    QMutex jackMutex, midiOutMutex, quitMutex;
    QWaitCondition voiceWait, synthWait, jackWait, quitWait;  
    int voiceReadyCounter;
    QSemaphore *synthSemaphore;
    double *buf[2];

  public:
    Model(int p_poly, QObject* parent=0);
    ~Model();

    long getUID();
    void sendXmlData(tokenCodeEnum code, int index1, int index2, int index3, int voiceIndex, double value);
    VoiceData* getVoiceData(int index);
    VoiceData* currentVoiceData();
    void setCurrentVoiceDataIndex(int index);
    void setEditRange(int start, int end);
    void getEditRange(int& start, int& end);
    double getMax(); 
    void setMax(double p_max); 
    int getPoly(); 
    void setPoly(int p_poly); 
    void setMaxNoiseStep(int value);
    int getMaxNoiseStep();
    int getRate(); 
    void setRate(int p_rate); 
    int getBufSize(); 
    void setBufSize(int p_bufSize); 
    double getPitchBend(); 
    void setPitchBend(double p_pitchBend); 
    double getLFO(); 
    void lfoStep(); 
    void setMasterVolume(double value);
    double getMasterVolume();
    double getMinScale(int p_index); 
    double getMaxScale(int p_index); 
    void setGroupScale(int p_group, int p_index, int value); 
    bool doSynthesis(); 
    void setSynthesis(bool on);
    voiceType* getVoices(); 
    envelopeType* getEnvelopes(int voiceIndex); 
    int getNumHarmonics(); 
    void setNumHarmonics(int p_numHarmonics); 
    int getEditGroup(); 
    int calcEditGroup(int p_index); 
    void setEditGroup(int p_group); 
    int getHarmonicSplit(int p_group); 
    void setHarmonicSplit(int p_group, int p_HarmonicSplit); 
    double *getHarmonics(int p_index); 
    double getHarmonic(int p_index, int p_num); 
    double getScaledHarmonic(int p_index, int p_num); 
    void setHarmonic(int p_index, int p_num, double p_value); 
    void setLfoFreq(double value);
    double getLfoFreq(); 
    void setLfoDepth(double value); 
    double getLfoDepth(); 
    void resetLFO(); 
    double getMorphing(); 
    void setMorphing(double value); 
    void setCompressor(int index, double value);
    void getCompressorCoeff(double& x0, double& x1, double& a, double& b, double& c, double& d, double& aa);
    double getDelta(int index); 
    void setDelta(int index, double value);
    void setEvenOdd(int index, double value);
    double getEvenOdd(int index);
    void setNoiseBandwidth(double value);
    double getNoiseBandwidth();
    void setInterpolate(bool on);
    bool getInterpolate();
    void addControlObject(QString name, int objectType, int objectIndex, QTreeWidgetItem *item);
    void addGroupSwitch(int radioGroupID, int radioGroupIndex);
    void updateRadioGroup(int radioGroupID, int radioGroupIndex);
    MidiEventList fetchEvents(int nframes);
    void addMidiEvent(int frame, int type, int ch, int index, int val);
    int hasMidiController(int type, unsigned char eventChannel,
                           unsigned int eventParam);
    void addMidiController(int type, unsigned char eventChannel,
                           unsigned int eventParam, QTreeWidgetItem *item); 
    bool connectController(QTreeWidgetItem *midiControllerItem, 
                           QTreeWidgetItem *controlObjectItem);
    bool connectController(QTreeWidgetItem *midiControllerItem, 
                           int objectType, int objectIndex, int groupIndex, int modDepth, bool full, bool inverse);
    bool disconnectController(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem); 
    void setCurrentGroupIndex(int value);
    int getCurrentGroupIndex();
    midiController getMidiController(int index);
    controlObject getControlObject(int index);
    int getGroupSwitchCount();
    groupSwitch getGroupSwitch(int index);
    void updateMidiControllerList(int index, midiController mc);
    void updateGroupSwitchList(int groupIndex, midiController mc, bool addController);
    int midiControllerCount();
    int connectedObjectCount(int midiControllerIndex);
    controlObject getConnectedObject(int midiIndex, int controllerIndex);
    void setControllerItem(QTreeWidgetItem *midiControllerItem,   
                           QTreeWidgetItem *controlObjectItem,    
                           QTreeWidgetItem *connectedObjectItem);
    void setControllerItem(QTreeWidgetItem *midiControllerItem,   
                           int objectType, int objectIndex,
                           QTreeWidgetItem *connectedObjectItem);
    void setControllerModulation(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem, int modDepth, bool full, bool inverse);                       
    controlObject getControlObject(QTreeWidgetItem *midiControllerItem, QTreeWidgetItem *controlObjectItem, bool& success);                       
    void clearMidiControllerList();
    void clearControlObjectList();                       
    bool isMidiControllerItem(QTreeWidgetItem *midiControllerItem);
    QString getControlObjectName(int objectType, int objectIndex);
    void setScalaBaseFreq(double value);
    double getScalaBaseFreq();
    void setScalaBaseNote(int value);
    int getScalaBaseNote();
    void setScalaMode(bool on);
    bool getScalaMode();
    double getScalaNote(int index);
    double getScalaFreq(int index);
    void updateScala(int &scalaLen, bool &octavePeriod);
    void clearScalaData();
    void appendScalaItem(double value, bool isCent);
    void setModeToggleState(bool on);
    bool getModeToggleState();
    void setSceneToggleState(bool on);
    bool getSceneToggleState();
    void setSourceSwitchState(int val);
    int getSourceSwitchState();
    void setFilterSwitchState(int val);
    int getFilterSwitchState();
    void setGroupSwitchState(bool mode, int val);
    int getGroupSwitchState(bool mode);
    void setFilterCurveMode(bool on);
    bool getFilterCurveMode();

  signals:
    void xmlData(tokenCodeEnum code, int index1, int index2, int index3, int voiceIndex, double value);
    
};
  
#endif
