/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#ifdef ARM_COMPUTE_ENABLE_SVE

#include "arm_gemm.hpp"
#include "../../utils.hpp"

#include <cassert>
#include <cstdint>

namespace arm_gemm {

void sve_hybrid_s8qs_dot_6x4VL (
    unsigned int num_strings, const unsigned int *string_lengths, IndirectInputArg<int8_t> A_arg,
    size_t M, size_t N, const int8_t *B_ptr, IndirectOutputArg<int8_t> output_arg,
    const Requantize32 *qp, const int32_t *col_bias, unsigned int col_base
)
{
    struct KernelArgs {
        const int32_t *multiplier_ptr = {};
        const int32_t *shift_ptr = {};
        unsigned int num_strings = {};
        const unsigned int *string_lengths = {};
        size_t N = {};
        const int8_t *B_ptr = {};
        size_t output_offset = {};
        size_t input_initial_col = {};
        size_t input_offset = {};
        void *output_ptr = {};
    } ka;

    unsigned long flags=0;
    void *input_ptr;

    if (output_arg.is_indirect) {
        ka.output_ptr=(void *)(output_arg.indirect.ptr);
        ka.output_offset=output_arg.indirect.offset;
        flags |= 0x4;
    } else {
        ka.output_ptr=(void *)(output_arg.direct.base);
        ka.output_offset=output_arg.direct.stride;
    }

    if (A_arg.is_indirect) {
        input_ptr=(void *)(A_arg.indirect.ptr);
        ka.input_offset=A_arg.indirect.start_row;
        ka.input_initial_col=A_arg.indirect.start_col;
        flags |= 0x8;
    } else {
        assert(num_strings==1);
        input_ptr=(void *)(A_arg.direct.base);
        ka.input_offset=A_arg.direct.stride;
    }
    ka.num_strings = num_strings;
    ka.string_lengths = string_lengths;
    ka.N = N;
    ka.B_ptr = B_ptr;
    if (qp->per_channel_requant) {
        flags |= 0x10;
        ka.multiplier_ptr=qp->per_channel_muls + col_base;
        ka.shift_ptr=qp->per_channel_right_shifts + col_base;
    }
    if (qp->c_offset > qp->minval) {
        flags |= 0x20;
    }
    __asm__ __volatile__(
      "ptrue p2.b\n"
      "1:"  // Row loop
      "cmp %x[M], #0x6\n"
      "bge 66f\n"
      "cmp %x[M], #0x4\n"
      "bgt 53f\n"
      "beq 40f\n"
      "cmp %x[M], #0x2\n"
      "bgt 27f\n"
      "beq 14f\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "2:"  // Height 1: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "3:"  // Height 1: setup done
      "mov x28, #0x0\n"
      "4:"  // Height 1: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 5f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "cbnz x28, 6f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "b 6f\n"
      "5:"  // Height 1: setup direct input
      "mov x26, %x[input_ptr]\n"
      "6:"  // Height 1: input setup done
      "cmp x27, #0x10\n"
      "ble 8f\n"
      "7:"  // Height 1: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "sdot z8.s, z17.b, z0.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z17.b, z0.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z16.b, z0.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z17.b, z0.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[1]\n"
      "ld1b { z16.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z17.b, z0.b[1]\n"
      "sdot z11.s, z16.b, z0.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z17.b, z0.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z17.b, z0.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z16.b, z0.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z17.b, z0.b[3]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[3]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z17.b, z0.b[3]\n"
      "sdot z11.s, z16.b, z0.b[3]\n"
      "bgt 7b\n"
      "8:"  // Height 1: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "sdot z8.s, z17.b, z0.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[0]\n"
      "sdot z11.s, z16.b, z0.b[0]\n"
      "ble 9f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z17.b, z0.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[1]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[1]\n"
      "sdot z11.s, z16.b, z0.b[1]\n"
      "ble 9f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z17.b, z0.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[2]\n"
      "sdot z11.s, z16.b, z0.b[2]\n"
      "ble 9f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z17.b, z0.b[3]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[3]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[3]\n"
      "sdot z11.s, z16.b, z0.b[3]\n"
      "9:"  // Height 1: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 4b\n"
      "ld1w { z19.s }, p2/Z, [x14]\n"
      "ld1w { z18.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z17.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add z8.s, z8.s, z19.s\n"
      "add z9.s, z9.s, z18.s\n"
      "add z10.s, z10.s, z17.s\n"
      "add z11.s, z11.s, z16.s\n"
      "tbz %x[flags], #4, 10f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 11f\n"
      "10:"  // Height 1: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "11:"  // Height 1: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      "tbz %x[flags], #5, 12f\n"
      "and z19.d, z8.d, z0.d\n"
      "and z18.d, z9.d, z1.d\n"
      "and z17.d, z10.d, z2.d\n"
      "and z16.d, z11.d, z3.d\n"
      "asr z19.s, z19.s, #0x1f\n"
      "asr z18.s, z18.s, #0x1f\n"
      "asr z17.s, z17.s, #0x1f\n"
      "asr z16.s, z16.s, #0x1f\n"
      "sqadd z8.s, z8.s, z19.s\n"
      "sqadd z9.s, z9.s, z18.s\n"
      "sqadd z10.s, z10.s, z17.s\n"
      "sqadd z11.s, z11.s, z16.s\n"
      "12:"  // Height 1: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z18.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "ld1rw { z17.s }, p2/Z, [x20]\n"
      "add z8.s, z8.s, z18.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z9.s, z9.s, z18.s\n"
      "add z10.s, z10.s, z18.s\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "add z11.s, z11.s, z18.s\n"
      "smin z8.s, p2/M, z8.s, z17.s\n"
      "smin z9.s, p2/M, z9.s, z17.s\n"
      "smin z10.s, p2/M, z10.s, z17.s\n"
      "smin z11.s, p2/M, z11.s, z17.s\n"
      "smax z8.s, p2/M, z8.s, z16.s\n"
      "smax z9.s, p2/M, z9.s, z16.s\n"
      "smax z10.s, p2/M, z10.s, z16.s\n"
      "smax z11.s, p2/M, z11.s, z16.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "uzp1 z16.h, z10.h, z11.h\n"
      "uzp1 z8.b, z8.b, z16.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "13:"  // Height 1: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 2b\n"
      "b 80f\n"
      "14:"  // Height 2
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "15:"  // Height 2: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "16:"  // Height 2: setup done
      "mov x28, #0x0\n"
      "17:"  // Height 2: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 18f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "cbnz x28, 19f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "b 19f\n"
      "18:"  // Height 2: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "19:"  // Height 2: input setup done
      "cmp x27, #0x10\n"
      "ble 21f\n"
      "20:"  // Height 2: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "sdot z8.s, z17.b, z1.b[0]\n"
      "sdot z12.s, z17.b, z0.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z1.b[0]\n"
      "sdot z13.s, z16.b, z0.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z17.b, z1.b[0]\n"
      "sdot z14.s, z17.b, z0.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z16.b, z1.b[0]\n"
      "sdot z15.s, z16.b, z0.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z17.b, z1.b[1]\n"
      "sdot z12.s, z17.b, z0.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z16.b, z1.b[1]\n"
      "sdot z13.s, z16.b, z0.b[1]\n"
      "ld1b { z16.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z17.b, z1.b[1]\n"
      "sdot z14.s, z17.b, z0.b[1]\n"
      "sdot z11.s, z16.b, z1.b[1]\n"
      "sdot z15.s, z16.b, z0.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z17.b, z1.b[2]\n"
      "sdot z12.s, z17.b, z0.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z16.b, z1.b[2]\n"
      "sdot z13.s, z16.b, z0.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z17.b, z1.b[2]\n"
      "sdot z14.s, z17.b, z0.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z16.b, z1.b[2]\n"
      "sdot z15.s, z16.b, z0.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z17.b, z1.b[3]\n"
      "sdot z12.s, z17.b, z0.b[3]\n"
      "ld1b { z17.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z16.b, z1.b[3]\n"
      "sdot z13.s, z16.b, z0.b[3]\n"
      "ld1b { z16.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z17.b, z1.b[3]\n"
      "sdot z14.s, z17.b, z0.b[3]\n"
      "sdot z11.s, z16.b, z1.b[3]\n"
      "sdot z15.s, z16.b, z0.b[3]\n"
      "bgt 20b\n"
      "21:"  // Height 2: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "sdot z8.s, z17.b, z0.b[0]\n"
      "sdot z12.s, z17.b, z1.b[0]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[0]\n"
      "sdot z13.s, z16.b, z1.b[0]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[0]\n"
      "sdot z14.s, z17.b, z1.b[0]\n"
      "sdot z11.s, z16.b, z0.b[0]\n"
      "sdot z15.s, z16.b, z1.b[0]\n"
      "ble 22f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z17.b, z0.b[1]\n"
      "sdot z12.s, z17.b, z1.b[1]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[1]\n"
      "sdot z13.s, z16.b, z1.b[1]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[1]\n"
      "sdot z14.s, z17.b, z1.b[1]\n"
      "sdot z11.s, z16.b, z0.b[1]\n"
      "sdot z15.s, z16.b, z1.b[1]\n"
      "ble 22f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z17.b, z0.b[2]\n"
      "sdot z12.s, z17.b, z1.b[2]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[2]\n"
      "sdot z13.s, z16.b, z1.b[2]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[2]\n"
      "sdot z14.s, z17.b, z1.b[2]\n"
      "sdot z11.s, z16.b, z0.b[2]\n"
      "sdot z15.s, z16.b, z1.b[2]\n"
      "ble 22f\n"
      "ld1b { z17.b }, p2/Z, [x10]\n"
      "ld1b { z16.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z17.b, z0.b[3]\n"
      "sdot z12.s, z17.b, z1.b[3]\n"
      "ld1b { z17.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z16.b, z0.b[3]\n"
      "sdot z13.s, z16.b, z1.b[3]\n"
      "ld1b { z16.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z17.b, z0.b[3]\n"
      "sdot z14.s, z17.b, z1.b[3]\n"
      "sdot z11.s, z16.b, z0.b[3]\n"
      "sdot z15.s, z16.b, z1.b[3]\n"
      "22:"  // Height 2: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 17b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z19.s }, p2/Z, [x14]\n"
      "ld1w { z18.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z17.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z16.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add x26, x9, x20\n"
      "add z8.s, z8.s, z19.s\n"
      "add z12.s, z12.s, z19.s\n"
      "add z9.s, z9.s, z18.s\n"
      "add z10.s, z10.s, z17.s\n"
      "add z13.s, z13.s, z18.s\n"
      "add z11.s, z11.s, z16.s\n"
      "add z14.s, z14.s, z17.s\n"
      "add z15.s, z15.s, z16.s\n"
      "tbz %x[flags], #4, 23f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 24f\n"
      "23:"  // Height 2: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "24:"  // Height 2: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      ".inst 0x04a4758c  // sqrdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a575ad  // sqrdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a675ce  // sqrdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a775ef  // sqrdmulh z15.s, z15.s, z7.s\n"
      "tbz %x[flags], #5, 25f\n"
      "and z19.d, z8.d, z0.d\n"
      "and z18.d, z9.d, z1.d\n"
      "and z17.d, z10.d, z2.d\n"
      "and z16.d, z11.d, z3.d\n"
      "asr z19.s, z19.s, #0x1f\n"
      "asr z18.s, z18.s, #0x1f\n"
      "asr z17.s, z17.s, #0x1f\n"
      "asr z16.s, z16.s, #0x1f\n"
      "sqadd z8.s, z8.s, z19.s\n"
      "and z19.d, z12.d, z0.d\n"
      "sqadd z9.s, z9.s, z18.s\n"
      "and z18.d, z13.d, z1.d\n"
      "sqadd z10.s, z10.s, z17.s\n"
      "sqadd z11.s, z11.s, z16.s\n"
      "and z17.d, z14.d, z2.d\n"
      "and z16.d, z15.d, z3.d\n"
      "asr z19.s, z19.s, #0x1f\n"
      "asr z18.s, z18.s, #0x1f\n"
      "asr z17.s, z17.s, #0x1f\n"
      "asr z16.s, z16.s, #0x1f\n"
      "sqadd z12.s, z12.s, z19.s\n"
      "sqadd z13.s, z13.s, z18.s\n"
      "sqadd z14.s, z14.s, z17.s\n"
      "sqadd z15.s, z15.s, z16.s\n"
      "25:"  // Height 2: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z18.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "add z8.s, z8.s, z18.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z9.s, z9.s, z18.s\n"
      "add z10.s, z10.s, z18.s\n"
      "ld1rw { z17.s }, p2/Z, [x20]\n"
      "add z11.s, z11.s, z18.s\n"
      "add z12.s, z12.s, z18.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z13.s, z13.s, z18.s\n"
      "add z14.s, z14.s, z18.s\n"
      "ld1rw { z16.s }, p2/Z, [x20]\n"
      "add z15.s, z15.s, z18.s\n"
      "smin z8.s, p2/M, z8.s, z17.s\n"
      "smin z9.s, p2/M, z9.s, z17.s\n"
      "smin z10.s, p2/M, z10.s, z17.s\n"
      "smin z11.s, p2/M, z11.s, z17.s\n"
      "smin z12.s, p2/M, z12.s, z17.s\n"
      "smin z13.s, p2/M, z13.s, z17.s\n"
      "smin z14.s, p2/M, z14.s, z17.s\n"
      "smin z15.s, p2/M, z15.s, z17.s\n"
      "smax z8.s, p2/M, z8.s, z16.s\n"
      "smax z9.s, p2/M, z9.s, z16.s\n"
      "smax z10.s, p2/M, z10.s, z16.s\n"
      "smax z11.s, p2/M, z11.s, z16.s\n"
      "smax z12.s, p2/M, z12.s, z16.s\n"
      "smax z13.s, p2/M, z13.s, z16.s\n"
      "smax z14.s, p2/M, z14.s, z16.s\n"
      "smax z15.s, p2/M, z15.s, z16.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "uzp1 z17.h, z10.h, z11.h\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z16.h, z14.h, z15.h\n"
      "uzp1 z8.b, z8.b, z17.b\n"
      "uzp1 z12.b, z12.b, z16.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "st1b { z12.b }, p1, [x26]\n"
      "26:"  // Height 2: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 15b\n"
      "b 80f\n"
      "27:"  // Height 3
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "28:"  // Height 3: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "29:"  // Height 3: setup done
      "mov x28, #0x0\n"
      "30:"  // Height 3: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 31f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "cbnz x28, 32f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "b 32f\n"
      "31:"  // Height 3: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "32:"  // Height 3: input setup done
      "cmp x27, #0x10\n"
      "ble 34f\n"
      "33:"  // Height 3: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z21.b }, p2/Z, [x10]\n"
      "ld1b { z20.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "sdot z8.s, z21.b, z2.b[0]\n"
      "sdot z12.s, z21.b, z1.b[0]\n"
      "sdot z9.s, z20.b, z2.b[0]\n"
      "sdot z13.s, z20.b, z1.b[0]\n"
      "sdot z16.s, z21.b, z0.b[0]\n"
      "ld1b { z21.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z20.b, z0.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z21.b, z2.b[0]\n"
      "sdot z14.s, z21.b, z1.b[0]\n"
      "sdot z18.s, z21.b, z0.b[0]\n"
      "ld1b { z21.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z20.b, z2.b[0]\n"
      "sdot z15.s, z20.b, z1.b[0]\n"
      "sdot z19.s, z20.b, z0.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z21.b, z2.b[1]\n"
      "sdot z12.s, z21.b, z1.b[1]\n"
      "sdot z16.s, z21.b, z0.b[1]\n"
      "ld1b { z21.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z20.b, z2.b[1]\n"
      "sdot z13.s, z20.b, z1.b[1]\n"
      "sdot z17.s, z20.b, z0.b[1]\n"
      "ld1b { z20.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z21.b, z2.b[1]\n"
      "sdot z14.s, z21.b, z1.b[1]\n"
      "sdot z18.s, z21.b, z0.b[1]\n"
      "sdot z11.s, z20.b, z2.b[1]\n"
      "sdot z15.s, z20.b, z1.b[1]\n"
      "sdot z19.s, z20.b, z0.b[1]\n"
      "ld1b { z21.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "ld1b { z20.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z21.b, z2.b[2]\n"
      "sdot z12.s, z21.b, z1.b[2]\n"
      "sdot z16.s, z21.b, z0.b[2]\n"
      "ld1b { z21.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z20.b, z2.b[2]\n"
      "sdot z13.s, z20.b, z1.b[2]\n"
      "sdot z17.s, z20.b, z0.b[2]\n"
      "ld1b { z20.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z21.b, z2.b[2]\n"
      "sdot z14.s, z21.b, z1.b[2]\n"
      "sdot z18.s, z21.b, z0.b[2]\n"
      "ld1b { z21.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z20.b, z2.b[2]\n"
      "sdot z15.s, z20.b, z1.b[2]\n"
      "sdot z19.s, z20.b, z0.b[2]\n"
      "ld1b { z20.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z21.b, z2.b[3]\n"
      "sdot z12.s, z21.b, z1.b[3]\n"
      "sdot z16.s, z21.b, z0.b[3]\n"
      "ld1b { z21.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z20.b, z2.b[3]\n"
      "sdot z13.s, z20.b, z1.b[3]\n"
      "sdot z17.s, z20.b, z0.b[3]\n"
      "ld1b { z20.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z21.b, z2.b[3]\n"
      "sdot z14.s, z21.b, z1.b[3]\n"
      "sdot z18.s, z21.b, z0.b[3]\n"
      "sdot z11.s, z20.b, z2.b[3]\n"
      "sdot z15.s, z20.b, z1.b[3]\n"
      "sdot z19.s, z20.b, z0.b[3]\n"
      "bgt 33b\n"
      "34:"  // Height 3: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z21.b }, p2/Z, [x10]\n"
      "ld1b { z20.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "sdot z8.s, z21.b, z0.b[0]\n"
      "sdot z12.s, z21.b, z1.b[0]\n"
      "sdot z9.s, z20.b, z0.b[0]\n"
      "sdot z13.s, z20.b, z1.b[0]\n"
      "sdot z16.s, z21.b, z2.b[0]\n"
      "ld1b { z21.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z20.b, z2.b[0]\n"
      "ld1b { z20.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z21.b, z0.b[0]\n"
      "sdot z14.s, z21.b, z1.b[0]\n"
      "sdot z18.s, z21.b, z2.b[0]\n"
      "sdot z11.s, z20.b, z0.b[0]\n"
      "sdot z15.s, z20.b, z1.b[0]\n"
      "sdot z19.s, z20.b, z2.b[0]\n"
      "ble 35f\n"
      "ld1b { z21.b }, p2/Z, [x10]\n"
      "ld1b { z20.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z21.b, z0.b[1]\n"
      "sdot z12.s, z21.b, z1.b[1]\n"
      "sdot z16.s, z21.b, z2.b[1]\n"
      "ld1b { z21.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z20.b, z0.b[1]\n"
      "sdot z13.s, z20.b, z1.b[1]\n"
      "sdot z17.s, z20.b, z2.b[1]\n"
      "ld1b { z20.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z21.b, z0.b[1]\n"
      "sdot z14.s, z21.b, z1.b[1]\n"
      "sdot z18.s, z21.b, z2.b[1]\n"
      "sdot z11.s, z20.b, z0.b[1]\n"
      "sdot z15.s, z20.b, z1.b[1]\n"
      "sdot z19.s, z20.b, z2.b[1]\n"
      "ble 35f\n"
      "ld1b { z21.b }, p2/Z, [x10]\n"
      "ld1b { z20.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z21.b, z0.b[2]\n"
      "sdot z12.s, z21.b, z1.b[2]\n"
      "sdot z16.s, z21.b, z2.b[2]\n"
      "ld1b { z21.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z20.b, z0.b[2]\n"
      "sdot z13.s, z20.b, z1.b[2]\n"
      "sdot z17.s, z20.b, z2.b[2]\n"
      "ld1b { z20.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z21.b, z0.b[2]\n"
      "sdot z14.s, z21.b, z1.b[2]\n"
      "sdot z18.s, z21.b, z2.b[2]\n"
      "sdot z11.s, z20.b, z0.b[2]\n"
      "sdot z15.s, z20.b, z1.b[2]\n"
      "sdot z19.s, z20.b, z2.b[2]\n"
      "ble 35f\n"
      "ld1b { z21.b }, p2/Z, [x10]\n"
      "ld1b { z20.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z21.b, z0.b[3]\n"
      "sdot z12.s, z21.b, z1.b[3]\n"
      "sdot z16.s, z21.b, z2.b[3]\n"
      "ld1b { z21.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z20.b, z0.b[3]\n"
      "sdot z13.s, z20.b, z1.b[3]\n"
      "sdot z17.s, z20.b, z2.b[3]\n"
      "ld1b { z20.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z21.b, z0.b[3]\n"
      "sdot z14.s, z21.b, z1.b[3]\n"
      "sdot z18.s, z21.b, z2.b[3]\n"
      "sdot z11.s, z20.b, z0.b[3]\n"
      "sdot z15.s, z20.b, z1.b[3]\n"
      "sdot z19.s, z20.b, z2.b[3]\n"
      "35:"  // Height 3: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 30b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z23.s }, p2/Z, [x14]\n"
      "ld1w { z22.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z21.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z20.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add x26, x9, x20\n"
      "add x25, x26, x20\n"
      "add z8.s, z8.s, z23.s\n"
      "add z12.s, z12.s, z23.s\n"
      "add z9.s, z9.s, z22.s\n"
      "add z10.s, z10.s, z21.s\n"
      "add z11.s, z11.s, z20.s\n"
      "add z13.s, z13.s, z22.s\n"
      "add z14.s, z14.s, z21.s\n"
      "add z15.s, z15.s, z20.s\n"
      "add z16.s, z16.s, z23.s\n"
      "add z17.s, z17.s, z22.s\n"
      "add z18.s, z18.s, z21.s\n"
      "add z19.s, z19.s, z20.s\n"
      "tbz %x[flags], #4, 36f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 37f\n"
      "36:"  // Height 3: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "37:"  // Height 3: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      ".inst 0x04a4758c  // sqrdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a575ad  // sqrdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a675ce  // sqrdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a775ef  // sqrdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x04a47610  // sqrdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57631  // sqrdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x04a67652  // sqrdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77673  // sqrdmulh z19.s, z19.s, z7.s\n"
      "tbz %x[flags], #5, 38f\n"
      "and z23.d, z8.d, z0.d\n"
      "and z22.d, z9.d, z1.d\n"
      "and z21.d, z10.d, z2.d\n"
      "and z20.d, z11.d, z3.d\n"
      "asr z23.s, z23.s, #0x1f\n"
      "asr z22.s, z22.s, #0x1f\n"
      "asr z21.s, z21.s, #0x1f\n"
      "asr z20.s, z20.s, #0x1f\n"
      "sqadd z8.s, z8.s, z23.s\n"
      "and z23.d, z12.d, z0.d\n"
      "sqadd z9.s, z9.s, z22.s\n"
      "and z22.d, z13.d, z1.d\n"
      "sqadd z10.s, z10.s, z21.s\n"
      "sqadd z11.s, z11.s, z20.s\n"
      "and z21.d, z14.d, z2.d\n"
      "and z20.d, z15.d, z3.d\n"
      "asr z23.s, z23.s, #0x1f\n"
      "asr z22.s, z22.s, #0x1f\n"
      "asr z21.s, z21.s, #0x1f\n"
      "asr z20.s, z20.s, #0x1f\n"
      "sqadd z12.s, z12.s, z23.s\n"
      "and z23.d, z16.d, z0.d\n"
      "sqadd z13.s, z13.s, z22.s\n"
      "and z22.d, z17.d, z1.d\n"
      "sqadd z14.s, z14.s, z21.s\n"
      "sqadd z15.s, z15.s, z20.s\n"
      "and z21.d, z18.d, z2.d\n"
      "and z20.d, z19.d, z3.d\n"
      "asr z23.s, z23.s, #0x1f\n"
      "asr z22.s, z22.s, #0x1f\n"
      "asr z21.s, z21.s, #0x1f\n"
      "asr z20.s, z20.s, #0x1f\n"
      "sqadd z16.s, z16.s, z23.s\n"
      "sqadd z17.s, z17.s, z22.s\n"
      "sqadd z18.s, z18.s, z21.s\n"
      "sqadd z19.s, z19.s, z20.s\n"
      "38:"  // Height 3: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z22.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "add z8.s, z8.s, z22.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z9.s, z9.s, z22.s\n"
      "add z10.s, z10.s, z22.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "add z11.s, z11.s, z22.s\n"
      "add z12.s, z12.s, z22.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z13.s, z13.s, z22.s\n"
      "add z14.s, z14.s, z22.s\n"
      "ld1rw { z21.s }, p2/Z, [x20]\n"
      "add z15.s, z15.s, z22.s\n"
      "add z16.s, z16.s, z22.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z17.s, z17.s, z22.s\n"
      "add z18.s, z18.s, z22.s\n"
      "ld1rw { z20.s }, p2/Z, [x20]\n"
      "add z19.s, z19.s, z22.s\n"
      "smin z8.s, p2/M, z8.s, z21.s\n"
      "smin z9.s, p2/M, z9.s, z21.s\n"
      "smin z10.s, p2/M, z10.s, z21.s\n"
      "smin z11.s, p2/M, z11.s, z21.s\n"
      "smin z12.s, p2/M, z12.s, z21.s\n"
      "smin z13.s, p2/M, z13.s, z21.s\n"
      "smin z14.s, p2/M, z14.s, z21.s\n"
      "smin z15.s, p2/M, z15.s, z21.s\n"
      "smin z16.s, p2/M, z16.s, z21.s\n"
      "smin z17.s, p2/M, z17.s, z21.s\n"
      "smin z18.s, p2/M, z18.s, z21.s\n"
      "smin z19.s, p2/M, z19.s, z21.s\n"
      "smax z8.s, p2/M, z8.s, z20.s\n"
      "smax z9.s, p2/M, z9.s, z20.s\n"
      "smax z10.s, p2/M, z10.s, z20.s\n"
      "smax z11.s, p2/M, z11.s, z20.s\n"
      "smax z12.s, p2/M, z12.s, z20.s\n"
      "smax z13.s, p2/M, z13.s, z20.s\n"
      "smax z14.s, p2/M, z14.s, z20.s\n"
      "smax z15.s, p2/M, z15.s, z20.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z20.s\n"
      "smax z17.s, p2/M, z17.s, z20.s\n"
      "uzp1 z21.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z20.s\n"
      "smax z19.s, p2/M, z19.s, z20.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "uzp1 z20.h, z14.h, z15.h\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z21.b\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "uzp1 z12.b, z12.b, z20.b\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "st1b { z12.b }, p1, [x26]\n"
      "st1b { z16.b }, p1, [x25]\n"
      "39:"  // Height 3: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 28b\n"
      "b 80f\n"
      "40:"  // Height 4
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "41:"  // Height 4: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "42:"  // Height 4: setup done
      "mov x28, #0x0\n"
      "43:"  // Height 4: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 44f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "cbnz x28, 45f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "b 45f\n"
      "44:"  // Height 4: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "45:"  // Height 4: input setup done
      "cmp x27, #0x10\n"
      "ble 47f\n"
      "46:"  // Height 4: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z25.b }, p2/Z, [x10]\n"
      "ld1b { z24.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z3.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z0.b }, p0/Z, [x23]\n"
      "add x23, x23, #0x10\n"
      "sdot z8.s, z25.b, z3.b[0]\n"
      "sdot z12.s, z25.b, z2.b[0]\n"
      "sdot z9.s, z24.b, z3.b[0]\n"
      "sdot z13.s, z24.b, z2.b[0]\n"
      "sdot z16.s, z25.b, z1.b[0]\n"
      "sdot z20.s, z25.b, z0.b[0]\n"
      "ld1b { z25.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z24.b, z1.b[0]\n"
      "sdot z21.s, z24.b, z0.b[0]\n"
      "ld1b { z24.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z25.b, z3.b[0]\n"
      "sdot z14.s, z25.b, z2.b[0]\n"
      "sdot z18.s, z25.b, z1.b[0]\n"
      "sdot z22.s, z25.b, z0.b[0]\n"
      "ld1b { z25.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z24.b, z3.b[0]\n"
      "sdot z15.s, z24.b, z2.b[0]\n"
      "sdot z19.s, z24.b, z1.b[0]\n"
      "sdot z23.s, z24.b, z0.b[0]\n"
      "ld1b { z24.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z25.b, z3.b[1]\n"
      "sdot z12.s, z25.b, z2.b[1]\n"
      "sdot z16.s, z25.b, z1.b[1]\n"
      "sdot z20.s, z25.b, z0.b[1]\n"
      "ld1b { z25.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z24.b, z3.b[1]\n"
      "sdot z13.s, z24.b, z2.b[1]\n"
      "sdot z17.s, z24.b, z1.b[1]\n"
      "sdot z21.s, z24.b, z0.b[1]\n"
      "ld1b { z24.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z25.b, z3.b[1]\n"
      "sdot z14.s, z25.b, z2.b[1]\n"
      "sdot z18.s, z25.b, z1.b[1]\n"
      "sdot z22.s, z25.b, z0.b[1]\n"
      "sdot z11.s, z24.b, z3.b[1]\n"
      "sdot z15.s, z24.b, z2.b[1]\n"
      "ld1b { z25.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z19.s, z24.b, z1.b[1]\n"
      "sdot z23.s, z24.b, z0.b[1]\n"
      "ld1b { z24.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z25.b, z3.b[2]\n"
      "sdot z12.s, z25.b, z2.b[2]\n"
      "sdot z16.s, z25.b, z1.b[2]\n"
      "sdot z20.s, z25.b, z0.b[2]\n"
      "ld1b { z25.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z24.b, z3.b[2]\n"
      "sdot z13.s, z24.b, z2.b[2]\n"
      "sdot z17.s, z24.b, z1.b[2]\n"
      "sdot z21.s, z24.b, z0.b[2]\n"
      "ld1b { z24.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z25.b, z3.b[2]\n"
      "sdot z14.s, z25.b, z2.b[2]\n"
      "sdot z18.s, z25.b, z1.b[2]\n"
      "sdot z22.s, z25.b, z0.b[2]\n"
      "ld1b { z25.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z24.b, z3.b[2]\n"
      "sdot z15.s, z24.b, z2.b[2]\n"
      "sdot z19.s, z24.b, z1.b[2]\n"
      "sdot z23.s, z24.b, z0.b[2]\n"
      "ld1b { z24.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z25.b, z3.b[3]\n"
      "sdot z12.s, z25.b, z2.b[3]\n"
      "sdot z16.s, z25.b, z1.b[3]\n"
      "sdot z20.s, z25.b, z0.b[3]\n"
      "ld1b { z25.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z24.b, z3.b[3]\n"
      "sdot z13.s, z24.b, z2.b[3]\n"
      "sdot z17.s, z24.b, z1.b[3]\n"
      "sdot z21.s, z24.b, z0.b[3]\n"
      "ld1b { z24.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z25.b, z3.b[3]\n"
      "sdot z14.s, z25.b, z2.b[3]\n"
      "sdot z18.s, z25.b, z1.b[3]\n"
      "sdot z22.s, z25.b, z0.b[3]\n"
      "sdot z11.s, z24.b, z3.b[3]\n"
      "sdot z15.s, z24.b, z2.b[3]\n"
      "sdot z19.s, z24.b, z1.b[3]\n"
      "sdot z23.s, z24.b, z0.b[3]\n"
      "bgt 46b\n"
      "47:"  // Height 4: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z25.b }, p2/Z, [x10]\n"
      "ld1b { z24.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "sdot z8.s, z25.b, z0.b[0]\n"
      "sdot z12.s, z25.b, z1.b[0]\n"
      "sdot z9.s, z24.b, z0.b[0]\n"
      "sdot z13.s, z24.b, z1.b[0]\n"
      "sdot z16.s, z25.b, z2.b[0]\n"
      "sdot z20.s, z25.b, z3.b[0]\n"
      "ld1b { z25.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z24.b, z2.b[0]\n"
      "sdot z21.s, z24.b, z3.b[0]\n"
      "ld1b { z24.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z25.b, z0.b[0]\n"
      "sdot z14.s, z25.b, z1.b[0]\n"
      "sdot z18.s, z25.b, z2.b[0]\n"
      "sdot z22.s, z25.b, z3.b[0]\n"
      "sdot z11.s, z24.b, z0.b[0]\n"
      "sdot z15.s, z24.b, z1.b[0]\n"
      "sdot z19.s, z24.b, z2.b[0]\n"
      "sdot z23.s, z24.b, z3.b[0]\n"
      "ble 48f\n"
      "ld1b { z25.b }, p2/Z, [x10]\n"
      "ld1b { z24.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z25.b, z0.b[1]\n"
      "sdot z12.s, z25.b, z1.b[1]\n"
      "sdot z16.s, z25.b, z2.b[1]\n"
      "sdot z20.s, z25.b, z3.b[1]\n"
      "ld1b { z25.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z24.b, z0.b[1]\n"
      "sdot z13.s, z24.b, z1.b[1]\n"
      "sdot z17.s, z24.b, z2.b[1]\n"
      "sdot z21.s, z24.b, z3.b[1]\n"
      "ld1b { z24.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z25.b, z0.b[1]\n"
      "sdot z14.s, z25.b, z1.b[1]\n"
      "sdot z18.s, z25.b, z2.b[1]\n"
      "sdot z22.s, z25.b, z3.b[1]\n"
      "sdot z11.s, z24.b, z0.b[1]\n"
      "sdot z15.s, z24.b, z1.b[1]\n"
      "sdot z19.s, z24.b, z2.b[1]\n"
      "sdot z23.s, z24.b, z3.b[1]\n"
      "ble 48f\n"
      "ld1b { z25.b }, p2/Z, [x10]\n"
      "ld1b { z24.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z25.b, z0.b[2]\n"
      "sdot z12.s, z25.b, z1.b[2]\n"
      "sdot z16.s, z25.b, z2.b[2]\n"
      "sdot z20.s, z25.b, z3.b[2]\n"
      "ld1b { z25.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z24.b, z0.b[2]\n"
      "sdot z13.s, z24.b, z1.b[2]\n"
      "sdot z17.s, z24.b, z2.b[2]\n"
      "sdot z21.s, z24.b, z3.b[2]\n"
      "ld1b { z24.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z25.b, z0.b[2]\n"
      "sdot z14.s, z25.b, z1.b[2]\n"
      "sdot z18.s, z25.b, z2.b[2]\n"
      "sdot z22.s, z25.b, z3.b[2]\n"
      "sdot z11.s, z24.b, z0.b[2]\n"
      "sdot z15.s, z24.b, z1.b[2]\n"
      "sdot z19.s, z24.b, z2.b[2]\n"
      "sdot z23.s, z24.b, z3.b[2]\n"
      "ble 48f\n"
      "ld1b { z25.b }, p2/Z, [x10]\n"
      "ld1b { z24.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z25.b, z0.b[3]\n"
      "sdot z12.s, z25.b, z1.b[3]\n"
      "sdot z16.s, z25.b, z2.b[3]\n"
      "sdot z20.s, z25.b, z3.b[3]\n"
      "ld1b { z25.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z24.b, z0.b[3]\n"
      "sdot z13.s, z24.b, z1.b[3]\n"
      "sdot z17.s, z24.b, z2.b[3]\n"
      "sdot z21.s, z24.b, z3.b[3]\n"
      "ld1b { z24.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z25.b, z0.b[3]\n"
      "sdot z14.s, z25.b, z1.b[3]\n"
      "sdot z18.s, z25.b, z2.b[3]\n"
      "sdot z22.s, z25.b, z3.b[3]\n"
      "sdot z11.s, z24.b, z0.b[3]\n"
      "sdot z15.s, z24.b, z1.b[3]\n"
      "sdot z19.s, z24.b, z2.b[3]\n"
      "sdot z23.s, z24.b, z3.b[3]\n"
      "48:"  // Height 4: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 43b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z27.s }, p2/Z, [x14]\n"
      "ld1w { z26.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z25.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z24.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add x26, x9, x20\n"
      "add x25, x26, x20\n"
      "add z8.s, z8.s, z27.s\n"
      "add z12.s, z12.s, z27.s\n"
      "add x24, x25, x20\n"
      "add z9.s, z9.s, z26.s\n"
      "add z10.s, z10.s, z25.s\n"
      "add z11.s, z11.s, z24.s\n"
      "add z13.s, z13.s, z26.s\n"
      "add z14.s, z14.s, z25.s\n"
      "add z15.s, z15.s, z24.s\n"
      "add z16.s, z16.s, z27.s\n"
      "add z17.s, z17.s, z26.s\n"
      "add z18.s, z18.s, z25.s\n"
      "add z19.s, z19.s, z24.s\n"
      "add z20.s, z20.s, z27.s\n"
      "add z21.s, z21.s, z26.s\n"
      "add z22.s, z22.s, z25.s\n"
      "add z23.s, z23.s, z24.s\n"
      "tbz %x[flags], #4, 49f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 50f\n"
      "49:"  // Height 4: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "50:"  // Height 4: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      ".inst 0x04a4758c  // sqrdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a575ad  // sqrdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a675ce  // sqrdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a775ef  // sqrdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x04a47610  // sqrdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57631  // sqrdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x04a67652  // sqrdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77673  // sqrdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x04a47694  // sqrdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a576b5  // sqrdmulh z21.s, z21.s, z5.s\n"
      ".inst 0x04a676d6  // sqrdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a776f7  // sqrdmulh z23.s, z23.s, z7.s\n"
      "tbz %x[flags], #5, 51f\n"
      "and z27.d, z8.d, z0.d\n"
      "and z26.d, z9.d, z1.d\n"
      "and z25.d, z10.d, z2.d\n"
      "and z24.d, z11.d, z3.d\n"
      "asr z27.s, z27.s, #0x1f\n"
      "asr z26.s, z26.s, #0x1f\n"
      "asr z25.s, z25.s, #0x1f\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z8.s, z8.s, z27.s\n"
      "and z27.d, z12.d, z0.d\n"
      "sqadd z9.s, z9.s, z26.s\n"
      "and z26.d, z13.d, z1.d\n"
      "sqadd z10.s, z10.s, z25.s\n"
      "sqadd z11.s, z11.s, z24.s\n"
      "and z25.d, z14.d, z2.d\n"
      "and z24.d, z15.d, z3.d\n"
      "asr z27.s, z27.s, #0x1f\n"
      "asr z26.s, z26.s, #0x1f\n"
      "asr z25.s, z25.s, #0x1f\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z12.s, z12.s, z27.s\n"
      "and z27.d, z16.d, z0.d\n"
      "sqadd z13.s, z13.s, z26.s\n"
      "and z26.d, z17.d, z1.d\n"
      "sqadd z14.s, z14.s, z25.s\n"
      "sqadd z15.s, z15.s, z24.s\n"
      "and z25.d, z18.d, z2.d\n"
      "and z24.d, z19.d, z3.d\n"
      "asr z27.s, z27.s, #0x1f\n"
      "asr z26.s, z26.s, #0x1f\n"
      "asr z25.s, z25.s, #0x1f\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z16.s, z16.s, z27.s\n"
      "and z27.d, z20.d, z0.d\n"
      "sqadd z17.s, z17.s, z26.s\n"
      "and z26.d, z21.d, z1.d\n"
      "sqadd z18.s, z18.s, z25.s\n"
      "sqadd z19.s, z19.s, z24.s\n"
      "and z25.d, z22.d, z2.d\n"
      "and z24.d, z23.d, z3.d\n"
      "asr z27.s, z27.s, #0x1f\n"
      "asr z26.s, z26.s, #0x1f\n"
      "asr z25.s, z25.s, #0x1f\n"
      "asr z24.s, z24.s, #0x1f\n"
      "sqadd z20.s, z20.s, z27.s\n"
      "sqadd z21.s, z21.s, z26.s\n"
      "sqadd z22.s, z22.s, z25.s\n"
      "sqadd z23.s, z23.s, z24.s\n"
      "51:"  // Height 4: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z25.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "add z8.s, z8.s, z25.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z9.s, z9.s, z25.s\n"
      "add z10.s, z10.s, z25.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "add z11.s, z11.s, z25.s\n"
      "add z12.s, z12.s, z25.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z13.s, z13.s, z25.s\n"
      "add z14.s, z14.s, z25.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      "add z15.s, z15.s, z25.s\n"
      "add z16.s, z16.s, z25.s\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z17.s, z17.s, z25.s\n"
      "add z18.s, z18.s, z25.s\n"
      "ld1rw { z24.s }, p2/Z, [x20]\n"
      "add z19.s, z19.s, z25.s\n"
      "add z20.s, z20.s, z25.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z21.s, z21.s, z25.s\n"
      "add z22.s, z22.s, z25.s\n"
      "ld1rw { z26.s }, p2/Z, [x20]\n"
      "add z23.s, z23.s, z25.s\n"
      "smin z8.s, p2/M, z8.s, z24.s\n"
      "smin z9.s, p2/M, z9.s, z24.s\n"
      "smin z10.s, p2/M, z10.s, z24.s\n"
      "smin z11.s, p2/M, z11.s, z24.s\n"
      "smin z12.s, p2/M, z12.s, z24.s\n"
      "smin z13.s, p2/M, z13.s, z24.s\n"
      "smin z14.s, p2/M, z14.s, z24.s\n"
      "smin z15.s, p2/M, z15.s, z24.s\n"
      "smin z16.s, p2/M, z16.s, z24.s\n"
      "smin z17.s, p2/M, z17.s, z24.s\n"
      "smin z18.s, p2/M, z18.s, z24.s\n"
      "smin z19.s, p2/M, z19.s, z24.s\n"
      "smin z20.s, p2/M, z20.s, z24.s\n"
      "smin z21.s, p2/M, z21.s, z24.s\n"
      "smin z22.s, p2/M, z22.s, z24.s\n"
      "smin z23.s, p2/M, z23.s, z24.s\n"
      "smax z8.s, p2/M, z8.s, z26.s\n"
      "smax z9.s, p2/M, z9.s, z26.s\n"
      "smax z10.s, p2/M, z10.s, z26.s\n"
      "smax z11.s, p2/M, z11.s, z26.s\n"
      "smax z12.s, p2/M, z12.s, z26.s\n"
      "smax z13.s, p2/M, z13.s, z26.s\n"
      "smax z14.s, p2/M, z14.s, z26.s\n"
      "smax z15.s, p2/M, z15.s, z26.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z26.s\n"
      "smax z17.s, p2/M, z17.s, z26.s\n"
      "uzp1 z25.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z26.s\n"
      "smax z19.s, p2/M, z19.s, z26.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z26.s\n"
      "smax z21.s, p2/M, z21.s, z26.s\n"
      "uzp1 z24.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z26.s\n"
      "smax z23.s, p2/M, z23.s, z26.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z25.b\n"
      "uzp1 z18.h, z18.h, z19.h\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z24.b\n"
      "uzp1 z17.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z16.b, z16.b, z18.b\n"
      "uzp1 z20.b, z20.b, z17.b\n"
      "st1b { z12.b }, p1, [x26]\n"
      "st1b { z16.b }, p1, [x25]\n"
      "st1b { z20.b }, p1, [x24]\n"
      "52:"  // Height 4: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 41b\n"
      "b 80f\n"
      "53:"  // Height 5
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "54:"  // Height 5: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "55:"  // Height 5: setup done
      "mov x28, #0x0\n"
      "56:"  // Height 5: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 57f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "cbnz x28, 58f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "b 58f\n"
      "57:"  // Height 5: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "58:"  // Height 5: input setup done
      "cmp x27, #0x10\n"
      "ble 60f\n"
      "59:"  // Height 5: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z29.b }, p2/Z, [x10]\n"
      "ld1b { z28.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z4.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z3.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z1.b }, p0/Z, [x23]\n"
      "ld1rqb { z0.b }, p0/Z, [x22]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "sdot z8.s, z29.b, z4.b[0]\n"
      "sdot z12.s, z29.b, z3.b[0]\n"
      "sdot z9.s, z28.b, z4.b[0]\n"
      "sdot z16.s, z29.b, z2.b[0]\n"
      "sdot z20.s, z29.b, z1.b[0]\n"
      "sdot z24.s, z29.b, z0.b[0]\n"
      "sdot z13.s, z28.b, z3.b[0]\n"
      "ld1b { z29.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z28.b, z2.b[0]\n"
      "sdot z21.s, z28.b, z1.b[0]\n"
      "sdot z25.s, z28.b, z0.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z29.b, z4.b[0]\n"
      "sdot z14.s, z29.b, z3.b[0]\n"
      "sdot z18.s, z29.b, z2.b[0]\n"
      "sdot z22.s, z29.b, z1.b[0]\n"
      "sdot z26.s, z29.b, z0.b[0]\n"
      "ld1b { z29.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z28.b, z4.b[0]\n"
      "sdot z15.s, z28.b, z3.b[0]\n"
      "sdot z19.s, z28.b, z2.b[0]\n"
      "sdot z23.s, z28.b, z1.b[0]\n"
      "sdot z27.s, z28.b, z0.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z29.b, z4.b[1]\n"
      "sdot z12.s, z29.b, z3.b[1]\n"
      "sdot z16.s, z29.b, z2.b[1]\n"
      "sdot z20.s, z29.b, z1.b[1]\n"
      "sdot z24.s, z29.b, z0.b[1]\n"
      "ld1b { z29.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z28.b, z4.b[1]\n"
      "sdot z13.s, z28.b, z3.b[1]\n"
      "sdot z17.s, z28.b, z2.b[1]\n"
      "sdot z21.s, z28.b, z1.b[1]\n"
      "sdot z25.s, z28.b, z0.b[1]\n"
      "ld1b { z28.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z29.b, z4.b[1]\n"
      "sdot z14.s, z29.b, z3.b[1]\n"
      "sdot z18.s, z29.b, z2.b[1]\n"
      "sdot z22.s, z29.b, z1.b[1]\n"
      "sdot z26.s, z29.b, z0.b[1]\n"
      "sdot z11.s, z28.b, z4.b[1]\n"
      "sdot z15.s, z28.b, z3.b[1]\n"
      "ld1b { z29.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z19.s, z28.b, z2.b[1]\n"
      "sdot z23.s, z28.b, z1.b[1]\n"
      "sdot z27.s, z28.b, z0.b[1]\n"
      "ld1b { z28.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z29.b, z4.b[2]\n"
      "sdot z12.s, z29.b, z3.b[2]\n"
      "sdot z16.s, z29.b, z2.b[2]\n"
      "sdot z20.s, z29.b, z1.b[2]\n"
      "sdot z24.s, z29.b, z0.b[2]\n"
      "ld1b { z29.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z28.b, z4.b[2]\n"
      "sdot z13.s, z28.b, z3.b[2]\n"
      "sdot z17.s, z28.b, z2.b[2]\n"
      "sdot z21.s, z28.b, z1.b[2]\n"
      "sdot z25.s, z28.b, z0.b[2]\n"
      "ld1b { z28.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z29.b, z4.b[2]\n"
      "sdot z14.s, z29.b, z3.b[2]\n"
      "sdot z18.s, z29.b, z2.b[2]\n"
      "sdot z22.s, z29.b, z1.b[2]\n"
      "sdot z26.s, z29.b, z0.b[2]\n"
      "ld1b { z29.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z28.b, z4.b[2]\n"
      "sdot z15.s, z28.b, z3.b[2]\n"
      "sdot z19.s, z28.b, z2.b[2]\n"
      "sdot z23.s, z28.b, z1.b[2]\n"
      "sdot z27.s, z28.b, z0.b[2]\n"
      "ld1b { z28.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z29.b, z4.b[3]\n"
      "sdot z12.s, z29.b, z3.b[3]\n"
      "sdot z16.s, z29.b, z2.b[3]\n"
      "sdot z20.s, z29.b, z1.b[3]\n"
      "sdot z24.s, z29.b, z0.b[3]\n"
      "ld1b { z29.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z28.b, z4.b[3]\n"
      "sdot z13.s, z28.b, z3.b[3]\n"
      "sdot z17.s, z28.b, z2.b[3]\n"
      "sdot z21.s, z28.b, z1.b[3]\n"
      "sdot z25.s, z28.b, z0.b[3]\n"
      "ld1b { z28.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z29.b, z4.b[3]\n"
      "sdot z14.s, z29.b, z3.b[3]\n"
      "sdot z18.s, z29.b, z2.b[3]\n"
      "sdot z22.s, z29.b, z1.b[3]\n"
      "sdot z26.s, z29.b, z0.b[3]\n"
      "sdot z11.s, z28.b, z4.b[3]\n"
      "sdot z15.s, z28.b, z3.b[3]\n"
      "sdot z19.s, z28.b, z2.b[3]\n"
      "sdot z23.s, z28.b, z1.b[3]\n"
      "sdot z27.s, z28.b, z0.b[3]\n"
      "bgt 59b\n"
      "60:"  // Height 5: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z29.b }, p2/Z, [x10]\n"
      "ld1b { z28.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "sdot z8.s, z29.b, z0.b[0]\n"
      "sdot z12.s, z29.b, z1.b[0]\n"
      "sdot z9.s, z28.b, z0.b[0]\n"
      "sdot z13.s, z28.b, z1.b[0]\n"
      "sdot z16.s, z29.b, z2.b[0]\n"
      "sdot z20.s, z29.b, z3.b[0]\n"
      "sdot z24.s, z29.b, z4.b[0]\n"
      "sdot z17.s, z28.b, z2.b[0]\n"
      "ld1b { z29.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z21.s, z28.b, z3.b[0]\n"
      "sdot z25.s, z28.b, z4.b[0]\n"
      "ld1b { z28.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z29.b, z0.b[0]\n"
      "sdot z14.s, z29.b, z1.b[0]\n"
      "sdot z18.s, z29.b, z2.b[0]\n"
      "sdot z22.s, z29.b, z3.b[0]\n"
      "sdot z26.s, z29.b, z4.b[0]\n"
      "sdot z11.s, z28.b, z0.b[0]\n"
      "sdot z15.s, z28.b, z1.b[0]\n"
      "sdot z19.s, z28.b, z2.b[0]\n"
      "sdot z23.s, z28.b, z3.b[0]\n"
      "sdot z27.s, z28.b, z4.b[0]\n"
      "ble 61f\n"
      "ld1b { z29.b }, p2/Z, [x10]\n"
      "ld1b { z28.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z29.b, z0.b[1]\n"
      "sdot z12.s, z29.b, z1.b[1]\n"
      "sdot z16.s, z29.b, z2.b[1]\n"
      "sdot z20.s, z29.b, z3.b[1]\n"
      "sdot z24.s, z29.b, z4.b[1]\n"
      "ld1b { z29.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z28.b, z0.b[1]\n"
      "sdot z13.s, z28.b, z1.b[1]\n"
      "sdot z17.s, z28.b, z2.b[1]\n"
      "sdot z21.s, z28.b, z3.b[1]\n"
      "sdot z25.s, z28.b, z4.b[1]\n"
      "ld1b { z28.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z29.b, z0.b[1]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z29.b, z1.b[1]\n"
      "sdot z18.s, z29.b, z2.b[1]\n"
      "sdot z22.s, z29.b, z3.b[1]\n"
      "sdot z26.s, z29.b, z4.b[1]\n"
      "sdot z11.s, z28.b, z0.b[1]\n"
      "sdot z15.s, z28.b, z1.b[1]\n"
      "sdot z19.s, z28.b, z2.b[1]\n"
      "sdot z23.s, z28.b, z3.b[1]\n"
      "sdot z27.s, z28.b, z4.b[1]\n"
      "ble 61f\n"
      "ld1b { z29.b }, p2/Z, [x10]\n"
      "ld1b { z28.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z29.b, z0.b[2]\n"
      "sdot z12.s, z29.b, z1.b[2]\n"
      "sdot z16.s, z29.b, z2.b[2]\n"
      "sdot z20.s, z29.b, z3.b[2]\n"
      "sdot z24.s, z29.b, z4.b[2]\n"
      "ld1b { z29.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z28.b, z0.b[2]\n"
      "sdot z13.s, z28.b, z1.b[2]\n"
      "sdot z17.s, z28.b, z2.b[2]\n"
      "sdot z21.s, z28.b, z3.b[2]\n"
      "sdot z25.s, z28.b, z4.b[2]\n"
      "ld1b { z28.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z29.b, z0.b[2]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z29.b, z1.b[2]\n"
      "sdot z18.s, z29.b, z2.b[2]\n"
      "sdot z22.s, z29.b, z3.b[2]\n"
      "sdot z26.s, z29.b, z4.b[2]\n"
      "sdot z11.s, z28.b, z0.b[2]\n"
      "sdot z15.s, z28.b, z1.b[2]\n"
      "sdot z19.s, z28.b, z2.b[2]\n"
      "sdot z23.s, z28.b, z3.b[2]\n"
      "sdot z27.s, z28.b, z4.b[2]\n"
      "ble 61f\n"
      "ld1b { z29.b }, p2/Z, [x10]\n"
      "ld1b { z28.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z29.b, z0.b[3]\n"
      "sdot z12.s, z29.b, z1.b[3]\n"
      "sdot z16.s, z29.b, z2.b[3]\n"
      "sdot z20.s, z29.b, z3.b[3]\n"
      "sdot z24.s, z29.b, z4.b[3]\n"
      "ld1b { z29.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z28.b, z0.b[3]\n"
      "sdot z13.s, z28.b, z1.b[3]\n"
      "sdot z17.s, z28.b, z2.b[3]\n"
      "sdot z21.s, z28.b, z3.b[3]\n"
      "sdot z25.s, z28.b, z4.b[3]\n"
      "ld1b { z28.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z29.b, z0.b[3]\n"
      "sdot z14.s, z29.b, z1.b[3]\n"
      "sdot z18.s, z29.b, z2.b[3]\n"
      "sdot z22.s, z29.b, z3.b[3]\n"
      "sdot z26.s, z29.b, z4.b[3]\n"
      "sdot z11.s, z28.b, z0.b[3]\n"
      "sdot z15.s, z28.b, z1.b[3]\n"
      "sdot z19.s, z28.b, z2.b[3]\n"
      "sdot z23.s, z28.b, z3.b[3]\n"
      "sdot z27.s, z28.b, z4.b[3]\n"
      "61:"  // Height 5: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 56b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z31.s }, p2/Z, [x14]\n"
      "ld1w { z30.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z29.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z28.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add x26, x9, x20\n"
      "add x25, x26, x20\n"
      "add z8.s, z8.s, z31.s\n"
      "add z12.s, z12.s, z31.s\n"
      "add x24, x25, x20\n"
      "add x23, x24, x20\n"
      "add z9.s, z9.s, z30.s\n"
      "add z10.s, z10.s, z29.s\n"
      "add z11.s, z11.s, z28.s\n"
      "add z13.s, z13.s, z30.s\n"
      "add z14.s, z14.s, z29.s\n"
      "add z15.s, z15.s, z28.s\n"
      "add z16.s, z16.s, z31.s\n"
      "add z17.s, z17.s, z30.s\n"
      "add z18.s, z18.s, z29.s\n"
      "add z19.s, z19.s, z28.s\n"
      "add z20.s, z20.s, z31.s\n"
      "add z21.s, z21.s, z30.s\n"
      "add z22.s, z22.s, z29.s\n"
      "add z23.s, z23.s, z28.s\n"
      "add z24.s, z24.s, z31.s\n"
      "add z25.s, z25.s, z30.s\n"
      "add z26.s, z26.s, z29.s\n"
      "add z27.s, z27.s, z28.s\n"
      "tbz %x[flags], #4, 62f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 63f\n"
      "62:"  // Height 5: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "63:"  // Height 5: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      ".inst 0x04a4758c  // sqrdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a575ad  // sqrdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a675ce  // sqrdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a775ef  // sqrdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x04a47610  // sqrdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57631  // sqrdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x04a67652  // sqrdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77673  // sqrdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x04a47694  // sqrdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a576b5  // sqrdmulh z21.s, z21.s, z5.s\n"
      ".inst 0x04a676d6  // sqrdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a776f7  // sqrdmulh z23.s, z23.s, z7.s\n"
      ".inst 0x04a47718  // sqrdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a57739  // sqrdmulh z25.s, z25.s, z5.s\n"
      ".inst 0x04a6775a  // sqrdmulh z26.s, z26.s, z6.s\n"
      ".inst 0x04a7777b  // sqrdmulh z27.s, z27.s, z7.s\n"
      "tbz %x[flags], #5, 64f\n"
      "and z31.d, z8.d, z0.d\n"
      "and z30.d, z9.d, z1.d\n"
      "and z29.d, z10.d, z2.d\n"
      "and z28.d, z11.d, z3.d\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z8.s, z8.s, z31.s\n"
      "and z31.d, z12.d, z0.d\n"
      "sqadd z9.s, z9.s, z30.s\n"
      "and z30.d, z13.d, z1.d\n"
      "sqadd z10.s, z10.s, z29.s\n"
      "sqadd z11.s, z11.s, z28.s\n"
      "and z29.d, z14.d, z2.d\n"
      "and z28.d, z15.d, z3.d\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z12.s, z12.s, z31.s\n"
      "and z31.d, z16.d, z0.d\n"
      "sqadd z13.s, z13.s, z30.s\n"
      "and z30.d, z17.d, z1.d\n"
      "sqadd z14.s, z14.s, z29.s\n"
      "sqadd z15.s, z15.s, z28.s\n"
      "and z29.d, z18.d, z2.d\n"
      "and z28.d, z19.d, z3.d\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z16.s, z16.s, z31.s\n"
      "and z31.d, z20.d, z0.d\n"
      "sqadd z17.s, z17.s, z30.s\n"
      "and z30.d, z21.d, z1.d\n"
      "sqadd z18.s, z18.s, z29.s\n"
      "sqadd z19.s, z19.s, z28.s\n"
      "and z29.d, z22.d, z2.d\n"
      "and z28.d, z23.d, z3.d\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z20.s, z20.s, z31.s\n"
      "and z31.d, z24.d, z0.d\n"
      "sqadd z21.s, z21.s, z30.s\n"
      "and z30.d, z25.d, z1.d\n"
      "sqadd z22.s, z22.s, z29.s\n"
      "sqadd z23.s, z23.s, z28.s\n"
      "and z29.d, z26.d, z2.d\n"
      "and z28.d, z27.d, z3.d\n"
      "asr z31.s, z31.s, #0x1f\n"
      "asr z30.s, z30.s, #0x1f\n"
      "asr z29.s, z29.s, #0x1f\n"
      "asr z28.s, z28.s, #0x1f\n"
      "sqadd z24.s, z24.s, z31.s\n"
      "sqadd z25.s, z25.s, z30.s\n"
      "sqadd z26.s, z26.s, z29.s\n"
      "sqadd z27.s, z27.s, z28.s\n"
      "64:"  // Height 5: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z29.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "add z8.s, z8.s, z29.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z9.s, z9.s, z29.s\n"
      "add z10.s, z10.s, z29.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "add z11.s, z11.s, z29.s\n"
      "add z12.s, z12.s, z29.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z13.s, z13.s, z29.s\n"
      "add z14.s, z14.s, z29.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      "add z15.s, z15.s, z29.s\n"
      "add z16.s, z16.s, z29.s\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      "add z17.s, z17.s, z29.s\n"
      "add z18.s, z18.s, z29.s\n"
      ".inst 0x44828839  // srshl z25.s, p2/M, z25.s, z1.s\n"
      ".inst 0x4482885a  // srshl z26.s, p2/M, z26.s, z2.s\n"
      "add z19.s, z19.s, z29.s\n"
      "add z20.s, z20.s, z29.s\n"
      ".inst 0x4482887b  // srshl z27.s, p2/M, z27.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z21.s, z21.s, z29.s\n"
      "add z22.s, z22.s, z29.s\n"
      "ld1rw { z28.s }, p2/Z, [x20]\n"
      "add z23.s, z23.s, z29.s\n"
      "add z24.s, z24.s, z29.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z25.s, z25.s, z29.s\n"
      "add z26.s, z26.s, z29.s\n"
      "ld1rw { z30.s }, p2/Z, [x20]\n"
      "add z27.s, z27.s, z29.s\n"
      "smin z8.s, p2/M, z8.s, z28.s\n"
      "smin z9.s, p2/M, z9.s, z28.s\n"
      "smin z10.s, p2/M, z10.s, z28.s\n"
      "smin z11.s, p2/M, z11.s, z28.s\n"
      "smin z12.s, p2/M, z12.s, z28.s\n"
      "smin z13.s, p2/M, z13.s, z28.s\n"
      "smin z14.s, p2/M, z14.s, z28.s\n"
      "smin z15.s, p2/M, z15.s, z28.s\n"
      "smin z16.s, p2/M, z16.s, z28.s\n"
      "smin z17.s, p2/M, z17.s, z28.s\n"
      "smin z18.s, p2/M, z18.s, z28.s\n"
      "smin z19.s, p2/M, z19.s, z28.s\n"
      "smin z20.s, p2/M, z20.s, z28.s\n"
      "smin z21.s, p2/M, z21.s, z28.s\n"
      "smin z22.s, p2/M, z22.s, z28.s\n"
      "smin z23.s, p2/M, z23.s, z28.s\n"
      "smin z24.s, p2/M, z24.s, z28.s\n"
      "smin z25.s, p2/M, z25.s, z28.s\n"
      "smin z26.s, p2/M, z26.s, z28.s\n"
      "smin z27.s, p2/M, z27.s, z28.s\n"
      "smax z8.s, p2/M, z8.s, z30.s\n"
      "smax z9.s, p2/M, z9.s, z30.s\n"
      "smax z10.s, p2/M, z10.s, z30.s\n"
      "smax z11.s, p2/M, z11.s, z30.s\n"
      "smax z12.s, p2/M, z12.s, z30.s\n"
      "smax z13.s, p2/M, z13.s, z30.s\n"
      "smax z14.s, p2/M, z14.s, z30.s\n"
      "smax z15.s, p2/M, z15.s, z30.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z30.s\n"
      "smax z17.s, p2/M, z17.s, z30.s\n"
      "uzp1 z29.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z30.s\n"
      "smax z19.s, p2/M, z19.s, z30.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z30.s\n"
      "smax z21.s, p2/M, z21.s, z30.s\n"
      "uzp1 z28.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z30.s\n"
      "smax z23.s, p2/M, z23.s, z30.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z29.b\n"
      "smax z24.s, p2/M, z24.s, z30.s\n"
      "smax z25.s, p2/M, z25.s, z30.s\n"
      "uzp1 z17.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z30.s\n"
      "smax z27.s, p2/M, z27.s, z30.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z28.b\n"
      "uzp1 z18.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z17.b\n"
      "uzp1 z17.h, z26.h, z27.h\n"
      "st1b { z12.b }, p1, [x26]\n"
      "uzp1 z20.b, z20.b, z18.b\n"
      "uzp1 z24.b, z24.b, z17.b\n"
      "st1b { z16.b }, p1, [x25]\n"
      "st1b { z20.b }, p1, [x24]\n"
      "st1b { z24.b }, p1, [x23]\n"
      "65:"  // Height 5: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 54b\n"
      "b 80f\n"
      "66:"  // Height 6
      "ldr x21, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ldr x9, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "mov x20, #0x6\n"
      "mov x14, %x[col_bias]\n"
      "ldr x13, [%x[args_ptr], %[offsetof_multiplier_ptr]]\n"
      "ldr x12, [%x[args_ptr], %[offsetof_shift_ptr]]\n"
      "ldr x11, [%x[args_ptr], %[offsetof_N]]\n"
      "ldr x10, [%x[args_ptr], %[offsetof_B_ptr]]\n"
      "madd x20, x21, x20, x9\n"
      "str x20, [%x[args_ptr], %[offsetof_output_ptr]]\n"
      "67:"  // Height 6: Column loop
      "mov x20, #0x0\n"
      "mov z8.s, #0x0\n"
      "mov z9.s, #0x0\n"
      "mov z10.s, #0x0\n"
      "mov z11.s, #0x0\n"
      "mov z12.s, #0x0\n"
      "mov z13.s, #0x0\n"
      "whilelt p1.b, x20, x11\n"
      "mov z14.s, #0x0\n"
      "mov z15.s, #0x0\n"
      "mov z16.s, #0x0\n"
      "mov z17.s, #0x0\n"
      "mov z18.s, #0x0\n"
      "mov z19.s, #0x0\n"
      "mov z20.s, #0x0\n"
      "mov z21.s, #0x0\n"
      "mov z22.s, #0x0\n"
      "mov z23.s, #0x0\n"
      "mov z24.s, #0x0\n"
      "mov z25.s, #0x0\n"
      "mov z26.s, #0x0\n"
      "mov z27.s, #0x0\n"
      "mov z28.s, #0x0\n"
      "mov z29.s, #0x0\n"
      "mov z30.s, #0x0\n"
      "mov z31.s, #0x0\n"
      "68:"  // Height 6: setup done
      "mov x28, #0x0\n"
      "69:"  // Height 6: String loop
      "ldr x20, [%x[args_ptr], %[offsetof_string_lengths]]\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "ldr w27, [x20, x28, LSL #0x2]\n"
      "tbz %x[flags], #3, 70f\n"
      "ldr x20, [%x[input_ptr], x28, LSL #0x3]\n"
      "add x20, x20, x21, LSL #3\n"
      "ldr x26, [x20, #0x0]\n"
      "ldr x25, [x20, #0x8]\n"
      "ldr x24, [x20, #0x10]\n"
      "ldr x23, [x20, #0x18]\n"
      "ldr x22, [x20, #0x20]\n"
      "ldr x21, [x20, #0x28]\n"
      "cbnz x28, 71f\n"
      "ldr x20, [%x[args_ptr], %[offsetof_input_initial_col]]\n"
      "add x26, x26, x20\n"
      "add x25, x25, x20\n"
      "add x24, x24, x20\n"
      "add x23, x23, x20\n"
      "add x22, x22, x20\n"
      "add x21, x21, x20\n"
      "b 71f\n"
      "70:"  // Height 6: setup direct input
      "mov x26, %x[input_ptr]\n"
      "add x25, x26, x21\n"
      "add x24, x25, x21\n"
      "add x23, x24, x21\n"
      "add x22, x23, x21\n"
      "add x21, x22, x21\n"
      "71:"  // Height 6: input setup done
      "cmp x27, #0x10\n"
      "ble 73f\n"
      "72:"  // Height 6: Multiply loop: Main loop head
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z1.b }, p2/Z, [x10]\n"
      "ld1b { z0.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sub x27, x27, #0x10\n"
      "cmp x27, #0x10\n"
      "ld1rqb { z7.b }, p0/Z, [x26]\n"
      "add x26, x26, #0x10\n"
      "ld1rqb { z6.b }, p0/Z, [x25]\n"
      "add x25, x25, #0x10\n"
      "ld1rqb { z5.b }, p0/Z, [x24]\n"
      "add x24, x24, #0x10\n"
      "ld1rqb { z4.b }, p0/Z, [x23]\n"
      "ld1rqb { z3.b }, p0/Z, [x22]\n"
      "ld1rqb { z2.b }, p0/Z, [x21]\n"
      "add x23, x23, #0x10\n"
      "add x22, x22, #0x10\n"
      "sdot z8.s, z1.b, z7.b[0]\n"
      "sdot z12.s, z1.b, z6.b[0]\n"
      "add x21, x21, #0x10\n"
      "sdot z16.s, z1.b, z5.b[0]\n"
      "sdot z20.s, z1.b, z4.b[0]\n"
      "sdot z24.s, z1.b, z3.b[0]\n"
      "sdot z28.s, z1.b, z2.b[0]\n"
      "ld1b { z1.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z0.b, z7.b[0]\n"
      "sdot z13.s, z0.b, z6.b[0]\n"
      "sdot z17.s, z0.b, z5.b[0]\n"
      "sdot z21.s, z0.b, z4.b[0]\n"
      "sdot z25.s, z0.b, z3.b[0]\n"
      "sdot z29.s, z0.b, z2.b[0]\n"
      "ld1b { z0.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z1.b, z7.b[0]\n"
      "sdot z14.s, z1.b, z6.b[0]\n"
      "sdot z18.s, z1.b, z5.b[0]\n"
      "sdot z22.s, z1.b, z4.b[0]\n"
      "sdot z26.s, z1.b, z3.b[0]\n"
      "sdot z30.s, z1.b, z2.b[0]\n"
      "ld1b { z1.b }, p2/Z, [x10, #4, MUL VL]\n"
      "sdot z11.s, z0.b, z7.b[0]\n"
      "sdot z15.s, z0.b, z6.b[0]\n"
      "sdot z19.s, z0.b, z5.b[0]\n"
      "sdot z23.s, z0.b, z4.b[0]\n"
      "sdot z27.s, z0.b, z3.b[0]\n"
      "sdot z31.s, z0.b, z2.b[0]\n"
      "ld1b { z0.b }, p2/Z, [x10, #5, MUL VL]\n"
      "sdot z8.s, z1.b, z7.b[1]\n"
      "sdot z12.s, z1.b, z6.b[1]\n"
      "sdot z16.s, z1.b, z5.b[1]\n"
      "sdot z20.s, z1.b, z4.b[1]\n"
      "sdot z24.s, z1.b, z3.b[1]\n"
      "sdot z28.s, z1.b, z2.b[1]\n"
      "ld1b { z1.b }, p2/Z, [x10, #6, MUL VL]\n"
      "sdot z9.s, z0.b, z7.b[1]\n"
      "sdot z13.s, z0.b, z6.b[1]\n"
      "sdot z17.s, z0.b, z5.b[1]\n"
      "sdot z21.s, z0.b, z4.b[1]\n"
      "sdot z25.s, z0.b, z3.b[1]\n"
      "sdot z29.s, z0.b, z2.b[1]\n"
      "ld1b { z0.b }, p2/Z, [x10, #7, MUL VL]\n"
      "addvl x10, x10, #16\n"
      "sdot z10.s, z1.b, z7.b[1]\n"
      "sdot z14.s, z1.b, z6.b[1]\n"
      "sdot z18.s, z1.b, z5.b[1]\n"
      "sdot z22.s, z1.b, z4.b[1]\n"
      "sdot z26.s, z1.b, z3.b[1]\n"
      "sdot z30.s, z1.b, z2.b[1]\n"
      "ld1b { z1.b }, p2/Z, [x10, #-8, MUL VL]\n"
      "sdot z11.s, z0.b, z7.b[1]\n"
      "sdot z15.s, z0.b, z6.b[1]\n"
      "sdot z19.s, z0.b, z5.b[1]\n"
      "sdot z23.s, z0.b, z4.b[1]\n"
      "sdot z27.s, z0.b, z3.b[1]\n"
      "sdot z31.s, z0.b, z2.b[1]\n"
      "ld1b { z0.b }, p2/Z, [x10, #-7, MUL VL]\n"
      "sdot z8.s, z1.b, z7.b[2]\n"
      "sdot z12.s, z1.b, z6.b[2]\n"
      "sdot z16.s, z1.b, z5.b[2]\n"
      "sdot z20.s, z1.b, z4.b[2]\n"
      "sdot z24.s, z1.b, z3.b[2]\n"
      "sdot z28.s, z1.b, z2.b[2]\n"
      "ld1b { z1.b }, p2/Z, [x10, #-6, MUL VL]\n"
      "sdot z9.s, z0.b, z7.b[2]\n"
      "sdot z13.s, z0.b, z6.b[2]\n"
      "sdot z17.s, z0.b, z5.b[2]\n"
      "sdot z21.s, z0.b, z4.b[2]\n"
      "sdot z25.s, z0.b, z3.b[2]\n"
      "sdot z29.s, z0.b, z2.b[2]\n"
      "ld1b { z0.b }, p2/Z, [x10, #-5, MUL VL]\n"
      "sdot z10.s, z1.b, z7.b[2]\n"
      "sdot z14.s, z1.b, z6.b[2]\n"
      "sdot z18.s, z1.b, z5.b[2]\n"
      "sdot z22.s, z1.b, z4.b[2]\n"
      "sdot z26.s, z1.b, z3.b[2]\n"
      "sdot z30.s, z1.b, z2.b[2]\n"
      "ld1b { z1.b }, p2/Z, [x10, #-4, MUL VL]\n"
      "sdot z11.s, z0.b, z7.b[2]\n"
      "sdot z15.s, z0.b, z6.b[2]\n"
      "sdot z19.s, z0.b, z5.b[2]\n"
      "sdot z23.s, z0.b, z4.b[2]\n"
      "sdot z27.s, z0.b, z3.b[2]\n"
      "sdot z31.s, z0.b, z2.b[2]\n"
      "ld1b { z0.b }, p2/Z, [x10, #-3, MUL VL]\n"
      "sdot z8.s, z1.b, z7.b[3]\n"
      "sdot z12.s, z1.b, z6.b[3]\n"
      "sdot z16.s, z1.b, z5.b[3]\n"
      "sdot z20.s, z1.b, z4.b[3]\n"
      "sdot z24.s, z1.b, z3.b[3]\n"
      "sdot z28.s, z1.b, z2.b[3]\n"
      "ld1b { z1.b }, p2/Z, [x10, #-2, MUL VL]\n"
      "sdot z9.s, z0.b, z7.b[3]\n"
      "sdot z13.s, z0.b, z6.b[3]\n"
      "sdot z17.s, z0.b, z5.b[3]\n"
      "sdot z21.s, z0.b, z4.b[3]\n"
      "sdot z25.s, z0.b, z3.b[3]\n"
      "sdot z29.s, z0.b, z2.b[3]\n"
      "ld1b { z0.b }, p2/Z, [x10, #-1, MUL VL]\n"
      "sdot z10.s, z1.b, z7.b[3]\n"
      "sdot z14.s, z1.b, z6.b[3]\n"
      "sdot z18.s, z1.b, z5.b[3]\n"
      "sdot z22.s, z1.b, z4.b[3]\n"
      "sdot z26.s, z1.b, z3.b[3]\n"
      "sdot z30.s, z1.b, z2.b[3]\n"
      "sdot z11.s, z0.b, z7.b[3]\n"
      "sdot z15.s, z0.b, z6.b[3]\n"
      "sdot z19.s, z0.b, z5.b[3]\n"
      "sdot z23.s, z0.b, z4.b[3]\n"
      "sdot z27.s, z0.b, z3.b[3]\n"
      "sdot z31.s, z0.b, z2.b[3]\n"
      "bgt 72b\n"
      "73:"  // Height 6: Multiply loop: Single iteration only
      "whilelt p0.b, XZR, x27\n"
      "ld1b { z7.b }, p2/Z, [x10]\n"
      "ld1b { z6.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "ld1rqb { z0.b }, p0/Z, [x26]\n"
      "ld1rqb { z1.b }, p0/Z, [x25]\n"
      "ld1rqb { z2.b }, p0/Z, [x24]\n"
      "ld1rqb { z3.b }, p0/Z, [x23]\n"
      "ld1rqb { z4.b }, p0/Z, [x22]\n"
      "ld1rqb { z5.b }, p0/Z, [x21]\n"
      "sdot z8.s, z7.b, z0.b[0]\n"
      "sdot z12.s, z7.b, z1.b[0]\n"
      "sdot z9.s, z6.b, z0.b[0]\n"
      "sdot z13.s, z6.b, z1.b[0]\n"
      "sdot z16.s, z7.b, z2.b[0]\n"
      "sdot z20.s, z7.b, z3.b[0]\n"
      "sdot z24.s, z7.b, z4.b[0]\n"
      "sdot z28.s, z7.b, z5.b[0]\n"
      "ld1b { z7.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z17.s, z6.b, z2.b[0]\n"
      "sdot z21.s, z6.b, z3.b[0]\n"
      "sdot z25.s, z6.b, z4.b[0]\n"
      "sdot z29.s, z6.b, z5.b[0]\n"
      "ld1b { z6.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z7.b, z0.b[0]\n"
      "sdot z14.s, z7.b, z1.b[0]\n"
      "sdot z18.s, z7.b, z2.b[0]\n"
      "sdot z22.s, z7.b, z3.b[0]\n"
      "sdot z26.s, z7.b, z4.b[0]\n"
      "sdot z30.s, z7.b, z5.b[0]\n"
      "sdot z11.s, z6.b, z0.b[0]\n"
      "sdot z15.s, z6.b, z1.b[0]\n"
      "sdot z19.s, z6.b, z2.b[0]\n"
      "sdot z23.s, z6.b, z3.b[0]\n"
      "sdot z27.s, z6.b, z4.b[0]\n"
      "sdot z31.s, z6.b, z5.b[0]\n"
      "ble 74f\n"
      "ld1b { z7.b }, p2/Z, [x10]\n"
      "ld1b { z6.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z7.b, z0.b[1]\n"
      "sdot z12.s, z7.b, z1.b[1]\n"
      "sdot z16.s, z7.b, z2.b[1]\n"
      "sdot z20.s, z7.b, z3.b[1]\n"
      "sdot z24.s, z7.b, z4.b[1]\n"
      "sdot z28.s, z7.b, z5.b[1]\n"
      "sdot z9.s, z6.b, z0.b[1]\n"
      "ld1b { z7.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z13.s, z6.b, z1.b[1]\n"
      "sdot z17.s, z6.b, z2.b[1]\n"
      "sdot z21.s, z6.b, z3.b[1]\n"
      "sdot z25.s, z6.b, z4.b[1]\n"
      "sdot z29.s, z6.b, z5.b[1]\n"
      "ld1b { z6.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z7.b, z0.b[1]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z7.b, z1.b[1]\n"
      "sdot z18.s, z7.b, z2.b[1]\n"
      "sdot z22.s, z7.b, z3.b[1]\n"
      "sdot z26.s, z7.b, z4.b[1]\n"
      "sdot z30.s, z7.b, z5.b[1]\n"
      "sdot z11.s, z6.b, z0.b[1]\n"
      "sdot z15.s, z6.b, z1.b[1]\n"
      "sdot z19.s, z6.b, z2.b[1]\n"
      "sdot z23.s, z6.b, z3.b[1]\n"
      "sdot z27.s, z6.b, z4.b[1]\n"
      "sdot z31.s, z6.b, z5.b[1]\n"
      "ble 74f\n"
      "ld1b { z7.b }, p2/Z, [x10]\n"
      "ld1b { z6.b }, p2/Z, [x10, #1, MUL VL]\n"
      "subs x27, x27, #0x4\n"
      "sdot z8.s, z7.b, z0.b[2]\n"
      "sdot z12.s, z7.b, z1.b[2]\n"
      "sdot z16.s, z7.b, z2.b[2]\n"
      "sdot z20.s, z7.b, z3.b[2]\n"
      "sdot z24.s, z7.b, z4.b[2]\n"
      "sdot z28.s, z7.b, z5.b[2]\n"
      "sdot z9.s, z6.b, z0.b[2]\n"
      "ld1b { z7.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z13.s, z6.b, z1.b[2]\n"
      "sdot z17.s, z6.b, z2.b[2]\n"
      "sdot z21.s, z6.b, z3.b[2]\n"
      "sdot z25.s, z6.b, z4.b[2]\n"
      "sdot z29.s, z6.b, z5.b[2]\n"
      "ld1b { z6.b }, p2/Z, [x10, #3, MUL VL]\n"
      "sdot z10.s, z7.b, z0.b[2]\n"
      "addvl x10, x10, #4\n"
      "sdot z14.s, z7.b, z1.b[2]\n"
      "sdot z18.s, z7.b, z2.b[2]\n"
      "sdot z22.s, z7.b, z3.b[2]\n"
      "sdot z26.s, z7.b, z4.b[2]\n"
      "sdot z30.s, z7.b, z5.b[2]\n"
      "sdot z11.s, z6.b, z0.b[2]\n"
      "sdot z15.s, z6.b, z1.b[2]\n"
      "sdot z19.s, z6.b, z2.b[2]\n"
      "sdot z23.s, z6.b, z3.b[2]\n"
      "sdot z27.s, z6.b, z4.b[2]\n"
      "sdot z31.s, z6.b, z5.b[2]\n"
      "ble 74f\n"
      "ld1b { z7.b }, p2/Z, [x10]\n"
      "ld1b { z6.b }, p2/Z, [x10, #1, MUL VL]\n"
      "sdot z8.s, z7.b, z0.b[3]\n"
      "sdot z12.s, z7.b, z1.b[3]\n"
      "sdot z16.s, z7.b, z2.b[3]\n"
      "sdot z20.s, z7.b, z3.b[3]\n"
      "sdot z24.s, z7.b, z4.b[3]\n"
      "sdot z28.s, z7.b, z5.b[3]\n"
      "ld1b { z7.b }, p2/Z, [x10, #2, MUL VL]\n"
      "sdot z9.s, z6.b, z0.b[3]\n"
      "sdot z13.s, z6.b, z1.b[3]\n"
      "sdot z17.s, z6.b, z2.b[3]\n"
      "sdot z21.s, z6.b, z3.b[3]\n"
      "sdot z25.s, z6.b, z4.b[3]\n"
      "sdot z29.s, z6.b, z5.b[3]\n"
      "ld1b { z6.b }, p2/Z, [x10, #3, MUL VL]\n"
      "addvl x10, x10, #4\n"
      "sdot z10.s, z7.b, z0.b[3]\n"
      "sdot z14.s, z7.b, z1.b[3]\n"
      "sdot z18.s, z7.b, z2.b[3]\n"
      "sdot z22.s, z7.b, z3.b[3]\n"
      "sdot z26.s, z7.b, z4.b[3]\n"
      "sdot z30.s, z7.b, z5.b[3]\n"
      "sdot z11.s, z6.b, z0.b[3]\n"
      "sdot z15.s, z6.b, z1.b[3]\n"
      "sdot z19.s, z6.b, z2.b[3]\n"
      "sdot z23.s, z6.b, z3.b[3]\n"
      "sdot z27.s, z6.b, z4.b[3]\n"
      "sdot z31.s, z6.b, z5.b[3]\n"
      "74:"  // Height 6: Multiply loop: multiply skip
      "ldr w20, [%x[args_ptr], %[offsetof_num_strings]]\n"
      "add x28, x28, #0x1\n"
      "cmp x28, x20\n"
      "bne 69b\n"
      "ldr x20, [%x[args_ptr], %[offsetof_output_offset]]\n"
      "ld1w { z3.s }, p2/Z, [x14]\n"
      "ld1w { z2.s }, p2/Z, [x14, #1, MUL VL]\n"
      "ld1w { z1.s }, p2/Z, [x14, #2, MUL VL]\n"
      "ld1w { z0.s }, p2/Z, [x14, #3, MUL VL]\n"
      "addvl x14, x14, #4\n"
      "add x26, x9, x20\n"
      "add x25, x26, x20\n"
      "add z8.s, z8.s, z3.s\n"
      "add z12.s, z12.s, z3.s\n"
      "add x24, x25, x20\n"
      "add x23, x24, x20\n"
      "add z9.s, z9.s, z2.s\n"
      "add z10.s, z10.s, z1.s\n"
      "add x22, x23, x20\n"
      "add z11.s, z11.s, z0.s\n"
      "add z13.s, z13.s, z2.s\n"
      "add z14.s, z14.s, z1.s\n"
      "add z15.s, z15.s, z0.s\n"
      "add z16.s, z16.s, z3.s\n"
      "add z17.s, z17.s, z2.s\n"
      "add z18.s, z18.s, z1.s\n"
      "add z19.s, z19.s, z0.s\n"
      "add z20.s, z20.s, z3.s\n"
      "add z21.s, z21.s, z2.s\n"
      "add z22.s, z22.s, z1.s\n"
      "add z23.s, z23.s, z0.s\n"
      "add z24.s, z24.s, z3.s\n"
      "add z25.s, z25.s, z2.s\n"
      "add z26.s, z26.s, z1.s\n"
      "add z27.s, z27.s, z0.s\n"
      "add z28.s, z28.s, z3.s\n"
      "add z29.s, z29.s, z2.s\n"
      "add z30.s, z30.s, z1.s\n"
      "add z31.s, z31.s, z0.s\n"
      "tbz %x[flags], #4, 75f\n"
      "ld1w { z0.s }, p2/Z, [x12]\n"
      "ld1w { z4.s }, p2/Z, [x13]\n"
      "ld1w { z1.s }, p2/Z, [x12, #1, MUL VL]\n"
      "ld1w { z5.s }, p2/Z, [x13, #1, MUL VL]\n"
      "ld1w { z2.s }, p2/Z, [x12, #2, MUL VL]\n"
      "ld1w { z6.s }, p2/Z, [x13, #2, MUL VL]\n"
      "ld1w { z3.s }, p2/Z, [x12, #3, MUL VL]\n"
      "ld1w { z7.s }, p2/Z, [x13, #3, MUL VL]\n"
      "addvl x12, x12, #4\n"
      "addvl x13, x13, #4\n"
      "b 76f\n"
      "75:"  // Height 6: per layer parameters
      "add x21, %x[qp], %[per_layer_right_shift]\n"
      "add x20, %x[qp], %[per_layer_mul]\n"
      "ld1rw { z0.s }, p2/Z, [x21]\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      "mov z1.d, z0.d\n"
      "mov z5.d, z4.d\n"
      "mov z2.d, z0.d\n"
      "mov z6.d, z4.d\n"
      "mov z3.d, z0.d\n"
      "mov z7.d, z4.d\n"
      "76:"  // Height 6: parameters loaded
      ".inst 0x04a47508  // sqrdmulh z8.s, z8.s, z4.s\n"
      ".inst 0x04a57529  // sqrdmulh z9.s, z9.s, z5.s\n"
      ".inst 0x04a6754a  // sqrdmulh z10.s, z10.s, z6.s\n"
      ".inst 0x04a7756b  // sqrdmulh z11.s, z11.s, z7.s\n"
      ".inst 0x04a4758c  // sqrdmulh z12.s, z12.s, z4.s\n"
      ".inst 0x04a575ad  // sqrdmulh z13.s, z13.s, z5.s\n"
      ".inst 0x04a675ce  // sqrdmulh z14.s, z14.s, z6.s\n"
      ".inst 0x04a775ef  // sqrdmulh z15.s, z15.s, z7.s\n"
      ".inst 0x04a47610  // sqrdmulh z16.s, z16.s, z4.s\n"
      ".inst 0x04a57631  // sqrdmulh z17.s, z17.s, z5.s\n"
      ".inst 0x04a67652  // sqrdmulh z18.s, z18.s, z6.s\n"
      ".inst 0x04a77673  // sqrdmulh z19.s, z19.s, z7.s\n"
      ".inst 0x04a47694  // sqrdmulh z20.s, z20.s, z4.s\n"
      ".inst 0x04a576b5  // sqrdmulh z21.s, z21.s, z5.s\n"
      ".inst 0x04a676d6  // sqrdmulh z22.s, z22.s, z6.s\n"
      ".inst 0x04a776f7  // sqrdmulh z23.s, z23.s, z7.s\n"
      ".inst 0x04a47718  // sqrdmulh z24.s, z24.s, z4.s\n"
      ".inst 0x04a57739  // sqrdmulh z25.s, z25.s, z5.s\n"
      ".inst 0x04a6775a  // sqrdmulh z26.s, z26.s, z6.s\n"
      ".inst 0x04a7777b  // sqrdmulh z27.s, z27.s, z7.s\n"
      ".inst 0x04a4779c  // sqrdmulh z28.s, z28.s, z4.s\n"
      ".inst 0x04a577bd  // sqrdmulh z29.s, z29.s, z5.s\n"
      ".inst 0x04a677de  // sqrdmulh z30.s, z30.s, z6.s\n"
      ".inst 0x04a777ff  // sqrdmulh z31.s, z31.s, z7.s\n"
      "tbz %x[flags], #5, 77f\n"
      "and z7.d, z8.d, z0.d\n"
      "and z6.d, z9.d, z1.d\n"
      "and z5.d, z10.d, z2.d\n"
      "and z4.d, z11.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z8.s, z8.s, z7.s\n"
      "and z7.d, z12.d, z0.d\n"
      "sqadd z9.s, z9.s, z6.s\n"
      "and z6.d, z13.d, z1.d\n"
      "sqadd z10.s, z10.s, z5.s\n"
      "sqadd z11.s, z11.s, z4.s\n"
      "and z5.d, z14.d, z2.d\n"
      "and z4.d, z15.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z12.s, z12.s, z7.s\n"
      "and z7.d, z16.d, z0.d\n"
      "sqadd z13.s, z13.s, z6.s\n"
      "and z6.d, z17.d, z1.d\n"
      "sqadd z14.s, z14.s, z5.s\n"
      "sqadd z15.s, z15.s, z4.s\n"
      "and z5.d, z18.d, z2.d\n"
      "and z4.d, z19.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z16.s, z16.s, z7.s\n"
      "and z7.d, z20.d, z0.d\n"
      "sqadd z17.s, z17.s, z6.s\n"
      "and z6.d, z21.d, z1.d\n"
      "sqadd z18.s, z18.s, z5.s\n"
      "sqadd z19.s, z19.s, z4.s\n"
      "and z5.d, z22.d, z2.d\n"
      "and z4.d, z23.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z20.s, z20.s, z7.s\n"
      "and z7.d, z24.d, z0.d\n"
      "sqadd z21.s, z21.s, z6.s\n"
      "and z6.d, z25.d, z1.d\n"
      "sqadd z22.s, z22.s, z5.s\n"
      "sqadd z23.s, z23.s, z4.s\n"
      "and z5.d, z26.d, z2.d\n"
      "and z4.d, z27.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z24.s, z24.s, z7.s\n"
      "and z7.d, z28.d, z0.d\n"
      "sqadd z25.s, z25.s, z6.s\n"
      "and z6.d, z29.d, z1.d\n"
      "sqadd z26.s, z26.s, z5.s\n"
      "sqadd z27.s, z27.s, z4.s\n"
      "and z5.d, z30.d, z2.d\n"
      "and z4.d, z31.d, z3.d\n"
      "asr z7.s, z7.s, #0x1f\n"
      "asr z6.s, z6.s, #0x1f\n"
      "asr z5.s, z5.s, #0x1f\n"
      "asr z4.s, z4.s, #0x1f\n"
      "sqadd z28.s, z28.s, z7.s\n"
      "sqadd z29.s, z29.s, z6.s\n"
      "sqadd z30.s, z30.s, z5.s\n"
      "sqadd z31.s, z31.s, z4.s\n"
      "77:"  // Height 6: no shift correction
      "add x20, %x[qp], %[c_offset]\n"
      ".inst 0x44828808  // srshl z8.s, p2/M, z8.s, z0.s\n"
      "ld1rw { z4.s }, p2/Z, [x20]\n"
      ".inst 0x44828829  // srshl z9.s, p2/M, z9.s, z1.s\n"
      ".inst 0x4482884a  // srshl z10.s, p2/M, z10.s, z2.s\n"
      ".inst 0x4482886b  // srshl z11.s, p2/M, z11.s, z3.s\n"
      ".inst 0x4482880c  // srshl z12.s, p2/M, z12.s, z0.s\n"
      ".inst 0x4482882d  // srshl z13.s, p2/M, z13.s, z1.s\n"
      ".inst 0x4482884e  // srshl z14.s, p2/M, z14.s, z2.s\n"
      "add z8.s, z8.s, z4.s\n"
      ".inst 0x4482886f  // srshl z15.s, p2/M, z15.s, z3.s\n"
      ".inst 0x44828810  // srshl z16.s, p2/M, z16.s, z0.s\n"
      "add z9.s, z9.s, z4.s\n"
      "add z10.s, z10.s, z4.s\n"
      ".inst 0x44828831  // srshl z17.s, p2/M, z17.s, z1.s\n"
      ".inst 0x44828852  // srshl z18.s, p2/M, z18.s, z2.s\n"
      "add z11.s, z11.s, z4.s\n"
      "add z12.s, z12.s, z4.s\n"
      ".inst 0x44828873  // srshl z19.s, p2/M, z19.s, z3.s\n"
      ".inst 0x44828814  // srshl z20.s, p2/M, z20.s, z0.s\n"
      "add z13.s, z13.s, z4.s\n"
      "add z14.s, z14.s, z4.s\n"
      ".inst 0x44828835  // srshl z21.s, p2/M, z21.s, z1.s\n"
      ".inst 0x44828856  // srshl z22.s, p2/M, z22.s, z2.s\n"
      "add z15.s, z15.s, z4.s\n"
      "add z16.s, z16.s, z4.s\n"
      ".inst 0x44828877  // srshl z23.s, p2/M, z23.s, z3.s\n"
      ".inst 0x44828818  // srshl z24.s, p2/M, z24.s, z0.s\n"
      "add z17.s, z17.s, z4.s\n"
      "add z18.s, z18.s, z4.s\n"
      ".inst 0x44828839  // srshl z25.s, p2/M, z25.s, z1.s\n"
      ".inst 0x4482885a  // srshl z26.s, p2/M, z26.s, z2.s\n"
      "add z19.s, z19.s, z4.s\n"
      "add z20.s, z20.s, z4.s\n"
      ".inst 0x4482887b  // srshl z27.s, p2/M, z27.s, z3.s\n"
      ".inst 0x4482881c  // srshl z28.s, p2/M, z28.s, z0.s\n"
      "add z21.s, z21.s, z4.s\n"
      "add z22.s, z22.s, z4.s\n"
      ".inst 0x4482883d  // srshl z29.s, p2/M, z29.s, z1.s\n"
      ".inst 0x4482885e  // srshl z30.s, p2/M, z30.s, z2.s\n"
      "add z23.s, z23.s, z4.s\n"
      "add z24.s, z24.s, z4.s\n"
      ".inst 0x4482887f  // srshl z31.s, p2/M, z31.s, z3.s\n"
      "add x20, %x[qp], %[maxval]\n"
      "add z25.s, z25.s, z4.s\n"
      "add z26.s, z26.s, z4.s\n"
      "ld1rw { z0.s }, p2/Z, [x20]\n"
      "add z27.s, z27.s, z4.s\n"
      "add z28.s, z28.s, z4.s\n"
      "add x20, %x[qp], %[minval]\n"
      "add z29.s, z29.s, z4.s\n"
      "add z30.s, z30.s, z4.s\n"
      "ld1rw { z2.s }, p2/Z, [x20]\n"
      "add z31.s, z31.s, z4.s\n"
      "smin z8.s, p2/M, z8.s, z0.s\n"
      "smin z9.s, p2/M, z9.s, z0.s\n"
      "smin z10.s, p2/M, z10.s, z0.s\n"
      "smin z11.s, p2/M, z11.s, z0.s\n"
      "smin z12.s, p2/M, z12.s, z0.s\n"
      "smin z13.s, p2/M, z13.s, z0.s\n"
      "smin z14.s, p2/M, z14.s, z0.s\n"
      "smin z15.s, p2/M, z15.s, z0.s\n"
      "smin z16.s, p2/M, z16.s, z0.s\n"
      "smin z17.s, p2/M, z17.s, z0.s\n"
      "smin z18.s, p2/M, z18.s, z0.s\n"
      "smin z19.s, p2/M, z19.s, z0.s\n"
      "smin z20.s, p2/M, z20.s, z0.s\n"
      "smin z21.s, p2/M, z21.s, z0.s\n"
      "smin z22.s, p2/M, z22.s, z0.s\n"
      "smin z23.s, p2/M, z23.s, z0.s\n"
      "smin z24.s, p2/M, z24.s, z0.s\n"
      "smin z25.s, p2/M, z25.s, z0.s\n"
      "smin z26.s, p2/M, z26.s, z0.s\n"
      "smin z27.s, p2/M, z27.s, z0.s\n"
      "smin z28.s, p2/M, z28.s, z0.s\n"
      "smin z29.s, p2/M, z29.s, z0.s\n"
      "smin z30.s, p2/M, z30.s, z0.s\n"
      "smin z31.s, p2/M, z31.s, z0.s\n"
      "smax z8.s, p2/M, z8.s, z2.s\n"
      "smax z9.s, p2/M, z9.s, z2.s\n"
      "smax z10.s, p2/M, z10.s, z2.s\n"
      "smax z11.s, p2/M, z11.s, z2.s\n"
      "smax z12.s, p2/M, z12.s, z2.s\n"
      "smax z13.s, p2/M, z13.s, z2.s\n"
      "smax z14.s, p2/M, z14.s, z2.s\n"
      "smax z15.s, p2/M, z15.s, z2.s\n"
      "uzp1 z8.h, z8.h, z9.h\n"
      "smax z16.s, p2/M, z16.s, z2.s\n"
      "smax z17.s, p2/M, z17.s, z2.s\n"
      "uzp1 z1.h, z10.h, z11.h\n"
      "smax z18.s, p2/M, z18.s, z2.s\n"
      "smax z19.s, p2/M, z19.s, z2.s\n"
      "uzp1 z12.h, z12.h, z13.h\n"
      "smax z20.s, p2/M, z20.s, z2.s\n"
      "smax z21.s, p2/M, z21.s, z2.s\n"
      "uzp1 z0.h, z14.h, z15.h\n"
      "smax z22.s, p2/M, z22.s, z2.s\n"
      "smax z23.s, p2/M, z23.s, z2.s\n"
      "uzp1 z16.h, z16.h, z17.h\n"
      "uzp1 z8.b, z8.b, z1.b\n"
      "smax z24.s, p2/M, z24.s, z2.s\n"
      "smax z25.s, p2/M, z25.s, z2.s\n"
      "uzp1 z18.h, z18.h, z19.h\n"
      "smax z26.s, p2/M, z26.s, z2.s\n"
      "smax z27.s, p2/M, z27.s, z2.s\n"
      "uzp1 z20.h, z20.h, z21.h\n"
      "uzp1 z12.b, z12.b, z0.b\n"
      "smax z28.s, p2/M, z28.s, z2.s\n"
      "smax z29.s, p2/M, z29.s, z2.s\n"
      "uzp1 z17.h, z22.h, z23.h\n"
      "st1b { z8.b }, p1, [x9]\n"
      "smax z30.s, p2/M, z30.s, z2.s\n"
      "smax z31.s, p2/M, z31.s, z2.s\n"
      "uzp1 z24.h, z24.h, z25.h\n"
      "uzp1 z16.b, z16.b, z18.b\n"
      "uzp1 z18.h, z26.h, z27.h\n"
      "st1b { z12.b }, p1, [x26]\n"
      "addvl x9, x9, #1\n"
      "uzp1 z28.h, z28.h, z29.h\n"
      "uzp1 z20.b, z20.b, z17.b\n"
      "uzp1 z17.h, z30.h, z31.h\n"
      "st1b { z16.b }, p1, [x25]\n"
      "uzp1 z24.b, z24.b, z18.b\n"
      "uzp1 z28.b, z28.b, z17.b\n"
      "st1b { z20.b }, p1, [x24]\n"
      "st1b { z24.b }, p1, [x23]\n"
      "st1b { z28.b }, p1, [x22]\n"
      "78:"  // Height 6: Writeback done
      "decw x11, ALL, MUL #4\n"
      "cmp x11, XZR\n"
      "bgt 67b\n"
      "subs %x[M], %x[M], #0x6\n"
      "beq 80f\n"
      "ldr x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "tbz %x[flags], #3, 79f\n"
      "add x21, x21, #0x6\n"
      "str x21, [%x[args_ptr], %[offsetof_input_offset]]\n"
      "b 1b\n"
      "79:"  // Update direct input
      "mov x20, #0x6\n"
      "madd %x[input_ptr], x20, x21, %x[input_ptr]\n"
      "b 1b\n"
      "80:"  // Exit
      : [M] "+&r" (M), [input_ptr] "+&r" (input_ptr)
      : [args_ptr] "r" (&ka), [c_offset] "I" (offsetof(Requantize32, c_offset)), [col_bias] "r" (col_bias), [flags] "r" (flags), [maxval] "I" (offsetof(Requantize32, maxval)), [minval] "I" (offsetof(Requantize32, minval)), [offsetof_B_ptr] "I" (offsetof(KernelArgs, B_ptr)), [offsetof_N] "I" (offsetof(KernelArgs, N)), [offsetof_input_initial_col] "I" (offsetof(KernelArgs, input_initial_col)), [offsetof_input_offset] "I" (offsetof(KernelArgs, input_offset)), [offsetof_multiplier_ptr] "I" (offsetof(KernelArgs, multiplier_ptr)), [offsetof_num_strings] "I" (offsetof(KernelArgs, num_strings)), [offsetof_output_offset] "I" (offsetof(KernelArgs, output_offset)), [offsetof_output_ptr] "I" (offsetof(KernelArgs, output_ptr)), [offsetof_shift_ptr] "I" (offsetof(KernelArgs, shift_ptr)), [offsetof_string_lengths] "I" (offsetof(KernelArgs, string_lengths)), [per_layer_mul] "I" (offsetof(Requantize32, per_layer_mul)), [per_layer_right_shift] "I" (offsetof(Requantize32, per_layer_right_shift)), [qp] "r" (qp)
      : "cc", "memory", "p0", "p1", "p2", "x9", "x10", "x11", "x12", "x13", "x14", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
    );
}

} // namespace arm_gemm
#endif  // ARM_COMPUTE_ENABLE_SVE
