# -*- coding: utf-8 -*-
#
# Copyright (C) 2009 Canonical Ltd
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

import os, sys
from distutils import log
from distutils.core import Command

try:
    # python 2.5
    from xml.etree.ElementTree import ElementTree, Element
except ImportError:
    # python 2.4
    from elementtree.ElementTree import ElementTree, Element

try:
    from bzrlib import trace
except ImportError:
    trace = None

import ide_menu #from bzrlib.plugins.explorer.lib import ide_menu


class build_menus(Command):
    description = "build the menu definitions for IDEs and shells to use"
    user_options = [
        ]
    boolean_options = []

    def initialize_options(self):
        pass

    def finalize_options(self):
        pass

    def run(self):
        src_file = "ui/explorer.ui"
        dest_file = "menus-for-ides.xml"
        log.info("compiling %s -> %s", src_file, dest_file)
        store = ide_menu.MenuStore()
        importer = QtFormImporter(src_file, store.root())
        store.save(dest_file)


class QtFormImporter(object):
    """Converts the menu in a Qt form to our IDE menu definition."""

    def __init__(self, path, target_menu):
        self._path = path
        self._target_menu = target_menu
        self._menus = {}
        self._actions = {}
        self.mapper = _ETreeFormMapper()
        self.load()
        goodies = ['menu_Settings']
        self._add_goodies_to_bazaar_menu(goodies)
        self._append_menu(target_menu, "menu_Bazaar")

    def load(self):
        """Load the form."""
        try:
            etree = ElementTree(file=self._path)
        except Exception:
            log.info("failed to parse form file %s", self._path)
            if trace:
                trace.mutter("failed to parse form file %s" % (self._path,))
                trace.report_exception(sys.exc_info(), sys.stderr)
        else:
            self._menus, self._actions = \
                self.mapper.etree_to_menus_and_actions(etree)

    def _add_goodies_to_bazaar_menu(self, goodies):
        if not goodies:
            return
        bzr_menu = self.menu_info('menu_Bazaar')
        if not bzr_menu:
            return
        title, children = bzr_menu
        children.append("separator")
        children.extend(goodies)

    def _dump(self):
        print "Menus:"
        for name, data in sorted(self._menus.items()):
            print "  %s => %s" % (name, data)
        print "Actions:"
        for name, data in sorted(self._actions.items()):
            print "  %s => %s" % (name, data)


    def action(self, name):
        """Get the MenuAction matching a name."""
        return self._actions.get(name)

    def menu_info(self, name):
        """Get the title and list of children names matching a menu name."""
        return self._menus.get(name)

    def _append_menu(self, target, source_name):
        menu_info = self.menu_info(source_name)
        if menu_info is None:
            return
        menu = ide_menu.Menu(menu_info[0])
        target.append(menu)
        for child in menu_info[1]:
            if child.startswith("action_"):
                menu.append(self.action(child))
            elif child.startswith("menu_"):
                self._append_menu(menu, child)
            elif child == 'separator':
                menu.append(ide_menu.MenuSeparator())


class _ETreeFormMapper(object):

    def etree_to_menus_and_actions(self, etree):
        """Convert an ElementTree to a list of menus and actions.

        :param etree: the ElementTree to convert
        :return: menus, actions where
          menus is a map from names to (title, child-names) tuples
          actions is a map from action names to MenuActions
        """
        self._menus = {}
        self._actions = {}
        root = etree.getroot()
        self._process_element(root)
        return self._menus, self._actions

    def _process_element(self, element):
        if element.tag == 'widget':
            qclass = element.get('class')
            if qclass == 'QMenu':
                self._process_menu(element)
            for child in element:
                self._process_element(child)
        elif element.tag == 'action':
            self._process_action(element)
        else:
            for child in element:
                self._process_element(child)

    def _process_menu(self, element):
        name = element.get("name")
        props, children = self._parse_properties(iter(element))
        text = props.get('title')
        menu_tuple = (text, children)
        self._menus[name] = menu_tuple

    def _process_action(self, element):
        name = element.get("name")
        props, _ = self._parse_properties(iter(element))
        command = self._get_command_for(name)
        conditions = self._get_conditions_for(name)
        # Remove the theme and sizing info from the icon
        icon = props.get("icon")
        if icon and icon.startswith(":/tango/16x16/"):
            icon = icon[len(":/tango/16x16/"):]
        act = ide_menu.MenuAction(
            props.get("text"),
            command,
            icon=icon,
            text_short=props.get("iconText"),
            text_tip=props.get("statusTip"),
            conditions=conditions)
        self._actions[name] = act

    def _get_command_for(self, name):
        specials = {
            # Temp hacks until some actions renamed
            'action_New': 'init',
            'action_Information': 'info',
            # Settings
            'action_Configuration':             'config:bazaar.conf',
            'action_Ignores':                   'config:ignore',
            'action_Rules':                     'config:rules',
            'action_Projects':                  'config:locations.conf',
            'action_Credentials':               'config:authenication.conf',
            'action_Branch_Configuration':      'config:branch.conf',
            'action_Branch_Ignores':            'config:.bzrignore',
            }
        if name in specials:
            return specials[name]
        if name.startswith("action_"):
            return name[7:].lower().replace('_', '-')
        return "????"

    def _get_conditions_for(self, name):
        needs_wt = [
            'Add', 'Diff', 'Commit',
            'Pull', 'Merge', 'Update', 'Push', 'Send',
            'List', 'Log', 'Info',
            'Bind', 'Unbind', 'Revert', 'Uncommit',
            'Shelve', 'Unshelve', 'Rename', 'Delete', 'Tag',
            ]
        name = name[len("action_"):]
        if name in needs_wt:
            return ["working-tree"]
        else:
            return None

    def _parse_properties(self, elements):
        result = {}
        children = []
        for e in elements:
            if e.tag == 'property':
                name = e.get("name")
                value = self._parse_value(iter(e))
                result[name] = value
            elif e.tag == 'addaction':
                child = e.get("name")
                children.append(child)
            elif e.tag != 'widget':
                print "hmm - got %s parsing properties" % (e.tag,)
                continue
        return result, children

    def _parse_value(self, elements):
        result = []
        for e in elements:
            if e.tag == 'string':
                result.append(e.text)
            elif e.tag == 'iconset':
                result.append(e[0].text)
            elif e.tag == 'enum':
                result.append(e.text)
        if len(result) == 1:
            return result[0]
        elif len(result) == 0:
            return None
        else:
            return result


if __name__ == '__main__':
    store = ide_menu.MenuStore()
    importer = QtFormImporter("../ui/explorer.ui", store.root())
    store.save("../menus-for-ides.xml")
