/*
 * Copyright 2013 Canonical Ltd.
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * telephony-app is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


import QtQuick 2.0
import Ubuntu.Components 0.1
import Ubuntu.Components.ListItems 0.1 as ListItem
import Friends 0.2

Page {
    anchors.fill: parent
    clip: true

    property alias flickable: listView

    property alias stream: streamModel.stream
    property alias service: streamModel.service

    property string search

    /* Jump to the bottom of the view and set the currentIndex */
    function jumpToBottom () {
        listView.currentIndex = listView.count - 1;
        listView.positionViewAtIndex(listView.currentIndex, ListView.End);
    }

    /* Jump to the top of the view and set the currentIndex */
    function jumpToTop () {
        listView.positionViewAtIndex(0, ListView.Contain);
        listView.currentIndex = 0;
    }

    Component.onCompleted: {
        listView.focus = true
    }

    Timer {
        id: updateTimestampsTimer
        running: true
        interval: Qt.application.active ? 30000 : 120000
        repeat: true
    }

    Component {
        id: statusDelegate

        ListItem.Empty {
            height: tile.height

            StatusUpdateTile {
                id: tile

                Connections {
                    target: updateTimestampsTimer
                    onTriggered: {
                        tile.recalculateTimeString();
                    }
                }
            }
        }
    }

    StreamModel {
        id: streamModel
        objectName: "streamModel"
    }

    RefreshBar {
        id: refreshBar

        opacity: visible ? 1.0 : 0.0
        visible: listView.dragging && listView.onTop

        Behavior on opacity {
            NumberAnimation { duration: 200; }
        }
    }

    ListView {
        id: listView
        objectName: "listView"

        anchors {
            fill: parent
        }

        // Save scrolling state when app is closed :)
        StateSaver.properties: contentY

        property bool onTop : listView.contentY <= -main.header.height

        opacity: 1
        spacing: units.gu(1)
        model: streamModel
        snapMode: ListView.NoSnap
        cacheBuffer: parent.height
        boundsBehavior: Flickable.DragOverBounds
        currentIndex: 0
        clip: true
        delegate: statusDelegate

        preferredHighlightBegin: 0
        preferredHighlightEnd: preferredHighlightBegin

        /* We need highlightRangeMode for keyboard navigation,
         * but it forces the page header to stay hidden
         * https://bugs.launchpad.net/ubuntu-ui-toolkit/+bug/1201936
         *
         * highlightRangeMode: ListView.StrictlyEnforceRange
        */

        /* Expand details on enter */
        Keys.onReturnPressed: {
            listView.currentItem.showConversation()
        }

        /* Handle home and end nav */
        Keys.onPressed: {
            if (event.key === Qt.Key_Home) {
                jumpToTop();
            }
            else if (event.key === Qt.Key_End) {
                jumpToBottom();
            }
        }
    }

    Scrollbar {
        flickableItem: listView
        align: Qt.AlignTrailing
    }


    tools: ToolbarItems {
        ToolbarButton {
            action: topAction
        }
        ToolbarButton {
            action: refreshAction
        }
        ToolbarButton {
            action: postAction
        }

        Item {
            anchors {
                top: parent.top
                bottom: parent.bottom

                margins: units.gu(1)
            }

            width: previewImagesLabel.width

            CheckBox {
                id: previewImagesSwitch

                height: units.gu(3.8)

                checked: true

                anchors.horizontalCenter: parent.horizontalCenter

                onCheckedChanged: {
                    previewImagesSetting.value = checked
                }

                Connections {
                    target: settingsStorage

                    onSettingsLoaded: {
                        previewImagesSwitch.checked = previewImagesSetting.value
                    }
                }
            }

            Label {
                id: previewImagesLabel

                anchors {
                    top: previewImagesSwitch.bottom
                    bottom: parent.bottom
                }

                color: "gray"
                font.pixelSize: units.dp(11)

                text: i18n.tr("Preview Images")
            }
        }
    }
}
