#ifndef _VMSTATE_H // -*-C++-*-
#define _VMSTATE_H

/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

using namespace std;

#ifndef WIN32
#include <pthread.h>
#define GC_THREADS
#else
#include <io.h>
#define GC_WIN32_THREADS
#endif
#include <gc/gc_cpp.h>
#include <setjmp.h>
#include <iostream>

#include <map>
#include <stack>

#include "Heap.h"

// Defined by compiling Builtins.k, so obviously any binary of a kaya
// program needs to be linked to at least Builtins.o!
#define PROJECT_FROM_NON_UNION K_ns__D_Builtins__D_Project_UN_From_UN_Non_UN_Union
#define LOOKUP_FROM_NON_ARRAY K_ns__D_Builtins__D_Lookup_UN_From_UN_Non_UN_Array
#define NEGATIVE_ARRAY_INDEX K_ns__D_Builtins__D_Negative_UN_Array_UN_Index
#define WRONG_CONSTRUCTOR K_ns__D_Builtins__D_Wrong_UN_Constructor
#define GETTING_TAG_FROM_NON_UNION K_ns__D_Builtins__D_Getting_UN_Tag_UN_From_UN_Non_UN_Union
#define INVALID_MARSHALLING_ID K_ns__D_Builtins__D_Invalid_UN_Marshalling_UN_ID
#define INVALID_FUNCTION_TABLE_HASH K_ns__D_Builtins__D_Invalid_UN_Function_UN_Table_UN_Hash
#define CANT_MARSHAL_EXCEPTIONS K_ns__D_Builtins__D_Cant_UN_Marshal_UN_Exceptions
#define CANT_REFLECT_EXCEPTIONS K_ns__D_Builtins__D_Cant_UN_Reflect_UN_Exceptions
#define INVALID_VALUE K_ns__D_Builtins__D_Invalid_UN_Value
#define NOT_IMPLEMENTED K_ns__D_Builtins__D_Not_UN_Implemented
#define INVALID_CIRCULAR_STRUCTURE K_ns__D_Builtins__D_Invalid_UN_Circular_UN_Structure
//#define UNKNOWN_FUNCTION_ID K_ns__D_Builtins__D_Unknown_UN_Function_UN_ID
#define DIVIDE_BY_ZERO K_ns__D_Builtins__D_Divide_UN_By_UN_Zero

extern "C" const char* PROJECT_FROM_NON_UNION;
extern "C" const char* LOOKUP_FROM_NON_ARRAY;
extern "C" const char* NEGATIVE_ARRAY_INDEX;
extern "C" const char* WRONG_CONSTRUCTOR;
extern "C" const char* GETTING_TAG_FROM_NON_UNION;
extern "C" const char* INVALID_MARSHALLING_ID;
extern "C" const char* INVALID_FUNCTION_TABLE_HASH;
extern "C" const char* INVALID_VALUE;
extern "C" const char* NOT_IMPLEMENTED;
extern "C" const char* INVALID_CIRCULAR_STRUCTURE;
extern "C" const char* CANT_MARSHAL_EXCEPTIONS;
extern "C" const char* CANT_REFLECT_EXCEPTIONS;
//extern "C" const char* UNKNOWN_FUNCTION_ID;
extern "C" const char* DIVIDE_BY_ZERO;

/// Function map is global to all threads.
//extern map<int,func> m_funmap;

void initFunMap(kint sz, kint fmhash);
void addToFunMap(kint id, func fn);
func getFn(kint id);
kint getFnID(func fn);
kint getFnHash();

class Exception;


class VMState: public gc {
    friend class Exception;
public:
    VMState(bool panic=false);
    
    Value* doPop() {
	Value* tmp=*(--m_stackptr);
	*m_stackptr=NULL;
	return tmp;
    }

  // sometimes just shortening the stack so this is faster than doPop()
    void discardPop() {
        *(--m_stackptr) = NULL;
    }
  
    void push(Value* val) {
	// TODO: doing this check on every single push() is quite slow
        // Is there a way to calculate the maximum stack increase a
        // single function will use and do a single check at the start
        // of the function?
        // In normal code the default stack size is enough anyway...

        if ((m_stackptr-m_valstack)>=m_stackalloc) {
	  realloc_stack();
        }


	*m_stackptr = val;
	++m_stackptr;
    }

    // Next ones are for efficiency
    void push2(Value* val, Value* val2);
    void push3(Value* val, Value* val2, Value* val3);
    void push4(Value* val, Value* val2, Value* val3,Value *val4);

    // Push then settop is common, so combine.
    void pushsettop(Value* val) {
	val->setPtr(*(--m_stackptr));
	*m_stackptr=NULL;
    }

    void pushrv(void* ptr, valtype type) {
        Value t(ptr,type);
	(*(m_stackptr-1))->setPtr(&t);
    }

    // Setting the top as a temp is also common.
    void tmpsettop(kint val) {
	(*(--m_stackptr))->setInt(val);
	*m_stackptr=NULL;
    }

    double topint2real();

    void mkArray(kint size);
    void mkArray(Value* v, kint size);
    void finish();
    bool emptyStack();

/// Replace the top stack item with its ith argument.
/// Let's see if inlining helps
    void projarg(ukint i, ukint t)
    {
#ifndef NOCHECK
      Value*& topval = (*(m_stackptr-1));
      if (topval==NULL) {
	  kaya_rtsError(INVALID_VALUE);
      }
      if (topval->getType()<KVT_UNION) {
	//	cout << "Debug2" << topval->getType() << endl;
	  kaya_rtsError(PROJECT_FROM_NON_UNION);
      }
      Union* top = (Union*)topval->getRaw();
      if ((ukint)(topval->getType()-KVT_UNION)!=t) {
	  kaya_rtsError(WRONG_CONSTRUCTOR);
      }
      if (U_ARITY(top) <= i) {
	  kaya_rtsError(INVALID_VALUE);
      }
      topval=top->args[i];
#else
      (*(m_stackptr-1)) = ((Union*)(((Value*&)(*(m_stackptr-1)))->getRaw()))->args[i];
#endif
      //    push(arg);
    }

    void setprojarg(Value*& src, ukint i, ukint t, Value* dest)
    {
#ifndef NOCHECK
      if (src==NULL) {
	  kaya_rtsError(INVALID_VALUE);
      }
      if (src->getType()<KVT_UNION) {
	  kaya_rtsError(PROJECT_FROM_NON_UNION);
      }
      Union* top = (Union*)src->getRaw();
      if ((ukint)(src->getType()-KVT_UNION)!=t) {
	  kaya_rtsError(WRONG_CONSTRUCTOR);
      }
      if (U_ARITY(top) <= i) {
	  kaya_rtsError(INVALID_VALUE);
      }
      top->args[i] = dest;
#else
      ((Value*&)(((Union*)(src->getRaw()))->args[i])) = dest;
#endif
      //    push(arg);
    }

    void goToIndex();

    void pushToIndex(Value* v)
    {
	kint idx = v->getInt();
	(*(m_stackptr-1)) = (*(m_stackptr-1))->lookup(this,idx);
    }

/** Replace the top stack item with the contents of the second stack item,
    then remove both (mutation, not evaluation) */
    void setTop()
    {
	(*(m_stackptr-1))->setPtr((*(m_stackptr-2)));
	(*(m_stackptr-2)) = (*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void setRV()
    {
	(*(m_stackptr-2))->setPtr((*(m_stackptr-1)));
	*(m_stackptr-1)=NULL;
	m_stackptr--;
    }

    void setRVreal(double r)
    {
	(*(m_stackptr-1))->setReal(m_rpool,r);
    }

// Add the top stack value to the next one, then remove both.
    void addTop()
    {
	(*(m_stackptr-2))->addVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void subTop()
    {
	(*(m_stackptr-2))->subVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void mulTop()
    {
	(*(m_stackptr-2))->mulVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void divTop()
    {
	(*(m_stackptr-2))->divVal(*(m_stackptr-1));
	*(m_stackptr-1)=NULL;
	m_stackptr-=2;
    }

    void appendTop()
    {
	// append top stack item onto top-1 stack item
        String* topstr = (*(--m_stackptr))->getString();
	(*(m_stackptr-1))->getString()->append(topstr);
	*m_stackptr = NULL;
    }

    void appendTopInt()
    {
	// append top stack item onto top-1 stack item
	wchar_t topstr = (wchar_t)((*(--m_stackptr))->getInt());
	// and now adjust the stack to correct for an explicit coercion
	// being removed.
	// FIXME: probably a better way to do this than a peephole optimisation
	// anyway!
	discardPop();
	// add the right values
	(*(m_stackptr-1))->getString()->append(topstr);
	*m_stackptr = NULL;

    }

    /// Return the top stack item.
    Value* topItem() {
	return (*(m_stackptr-1));
    }

    
    void pushglobal(Value**& globtable, kint i);
    void createglobal(wchar_t* modid,kint i);

    kint tag();

    void doAppend();
    void doAppendChar(const wchar_t x);
    void doAppendStr(const wchar_t* x);
    void doEqExcept(bool inv);
    void doEqString();
    void doEqString(const wchar_t* str);
    void doNeString();
    void doNeString(const wchar_t* str);
    bool doTopStrEq(const wchar_t* str);

    void str2int();
    void int2str();
    void real2str();
    void str2real();
    void str2chr();
    void chr2str();
    void bool2str();
    void real2int();
    void int2real();

    void newjmp_buf();

    void kaya_throw(wchar_t* msg, kint code);
    void kaya_internalError(kint code);
    void kaya_rtsError(const char* exceptionID);
    void throw_ex();
    jmp_buf* top_ex();
    void tried();
    void restore();

    void getindex();

    void pushgetindex(Value* v) {
	kint idx = v->getInt();
	--m_stackptr;
	Value* array = *m_stackptr;
	Value* el = array->lookup(this,idx);
	push(el);
    }

    void setlookup(kint idx, Value* v) {
	Value* array = *(m_stackptr-1);
	Value* el = array->lookup(this,idx);
	v->setPtr(el);
    }

    Value* mkstr(wchar_t* str);
    Value* mkint(void* i);

    void lineno(const wchar_t* src, int ln); 
    void pushBT(const wchar_t* src, const wchar_t* mod, int ln);
    void popBT();

    void memstat();

    int maxMemUsage() { 
	return m_maxmem; 
    }

    // Implement print via write
    void writestdout();

    void returnVal(Value* ret) { 
	m_return.setPtr(ret); 
    }

    Value* getReturnVal() { return &m_return; }

    // Throw divide by zero exception
    void divideByZero() {
	kaya_rtsError(DIVIDE_BY_ZERO);
    }

    void realloc_stack();
    void realloc_callstack();

    ValuePool* getVPool() {
        return m_vpool;
    }
    UnionPool* getUPool() {
        return m_upool;
    }
    StringPool* getSPool() {
        return m_spool;
    }
    RealPool* getRPool() {
        return m_rpool;
    }
    ArrayPool* getAPool() {
        return m_apool;
    }

  // Temporary string buffer to reduce allocations needed in string handling
  // If this works well then it should be done in a neater way!
    wchar_t* allocString(kint alloc) {
      strbuffer = (wchar_t*)KMALLOC_BLOCK(alloc*sizeof(wchar_t));
      strbuflen = alloc;
      strbuffer[0] = '\0';
      return strbuffer;
    }
    wchar_t* strbuffer;
    kint strbuflen;

private:
    Value** m_valstack;
    Value** m_stackptr;
    CallStackEntry** m_btptr;

//    int m_stacksize;

    ValuePool* m_vpool;
    UnionPool* m_upool;
    StringPool* m_spool;
    RealPool* m_rpool;
    ArrayPool* m_apool;

    int m_stackalloc;
    int m_callstackalloc;
    int m_globalloc;

    struct StackData {
	Value** stackptr;
	CallStackEntry** csptr;
    };

    stack<jmp_buf*> m_except_stack;
    stack<StackData> m_stack_stack; // Stack size when an exception happens.

    // Tracing details
    CallStackEntry** m_backtrace;
    wchar_t* m_sourcefile;
    int m_lineno;

    int m_maxmem;

    Value m_return; // Place to put return values (sort of a register).

};

#endif // whole file
