/*
 * Copyright 2010-2017 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jetbrains.kotlin.generators.tests

import java.io.File
import com.intellij.openapi.util.io.FileUtil
import java.io.PrintWriter
import java.io.StringWriter

object GenerateInRangeExpressionTestData {
    private val TEST_DATA_DIR = File("compiler/testData/codegen/box/ranges/contains")
    private val GENERATED_DIR = File(TEST_DATA_DIR, "generated")

    private val PREAMBLE_MESSAGE = "Auto-generated by ${GenerateInRangeExpressionTestData::class.java.simpleName}. Do not edit!"

    private fun generateMatrixTestCase(
            fileName: String,
            rangeExpressions: List<String>,
            elementExpressions: List<String>,
            header: String = ""
    ) {
        PrintWriter(File(GENERATED_DIR, fileName)).use {
            it.generateTestCaseBody(header, rangeExpressions, elementExpressions)
        }
    }

    private fun PrintWriter.generateTestCaseBody(header: String, rangeExpressions: List<String>, elementExpressions: List<String>) {
        println("// $PREAMBLE_MESSAGE")
        println("// WITH_RUNTIME")
        println()
        println(header)
        println()

        val rangeValNames = generateGlobalValDefinitions(rangeExpressions, "range")

        val elementValNames = generateGlobalValDefinitions(elementExpressions, "element")

        val testFunctions = StringWriter()
        val testFunctionsWriter = PrintWriter(testFunctions)

        println("fun box(): String {")
        rangeValNames.zip(rangeExpressions).forEachIndexed { i, (rangeValName, rangeExpression) ->
            elementValNames.zip(elementExpressions).forEachIndexed { j, (elementValName, elementExpression) ->
                val functionName = "testR${i}xE${j}"

                println("    $functionName()")

                testFunctionsWriter.generateTestCaseFunction(functionName, rangeValName, rangeExpression, elementValName, elementExpression)
            }
        }
        println("    return \"OK\"")
        println("}")
        println()
        println(testFunctions.toString())
    }

    private fun PrintWriter.generateGlobalValDefinitions(expressions: List<String>, prefix: String): List<String> {
        val valNames = expressions.indices.map { "$prefix$it" }
        valNames.zip(expressions).forEach { (name, expression) -> println("val $name = $expression") }
        println()
        return valNames
    }

    private fun PrintWriter.generateTestCaseFunction(
            functionName: String,
            rangeValName: String,
            rangeExpression: String,
            elementValName: String,
            elementExpression: String
    ) {
        println("fun $functionName() {")
        println("    // with possible local optimizations")
        println("    if ($elementExpression in $rangeExpression != $rangeValName.contains($elementExpression)) throw AssertionError()")
        println("    if ($elementExpression !in $rangeExpression != !$rangeValName.contains($elementExpression)) throw AssertionError()")
        println("    if (!($elementExpression in $rangeExpression) != !$rangeValName.contains($elementExpression)) throw AssertionError()")
        println("    if (!($elementExpression !in $rangeExpression) != $rangeValName.contains($elementExpression)) throw AssertionError()")
        println("    // no local optimizations")
        println("    if ($elementValName in $rangeExpression != $rangeValName.contains($elementValName)) throw AssertionError()")
        println("    if ($elementValName !in $rangeExpression != !$rangeValName.contains($elementValName)) throw AssertionError()")
        println("    if (!($elementValName in $rangeExpression) != !$rangeValName.contains($elementValName)) throw AssertionError()")
        println("    if (!($elementValName !in $rangeExpression) != $rangeValName.contains($elementValName)) throw AssertionError()")
        println("}")
        println()
    }

    private fun generateRangeOperatorTestCase(
            name: String,
            aExpression: String,
            op: String,
            bExpression: String,
            elementExpressions: List<String>
    ) {
        generateMatrixTestCase(
                name,
                listOf(
                        "$aExpression $op $bExpression",
                        "$bExpression $op $aExpression"
                ),
                elementExpressions
        )
    }

    @JvmStatic
    fun main(args: Array<String>) {
        if (!TEST_DATA_DIR.exists()) throw AssertionError("${TEST_DATA_DIR.path} doesn't exist")

        FileUtil.delete(GENERATED_DIR)
        GENERATED_DIR.mkdirs()

        val charLiterals = listOf("'0'", "'1'", "'2'", "'3'", "'4'")

        val numericLiterals =
                listOf("(-1)", "0", "1", "2", "3", "4").flatMap {
                    listOf("$it.toByte()", "$it.toShort()", it, "$it.toLong()", "$it.toFloat()", "$it.toDouble()")
                }

        generateRangeOperatorTestCase("charRangeLiteral.kt", "'1'", "..", "'3'", charLiterals)
        generateRangeOperatorTestCase("charUntil.kt", "'1'", "until", "'3'", charLiterals)
        generateRangeOperatorTestCase("charDownTo.kt", "'3'", "downTo", "'1'", charLiterals)

        generateRangeOperatorTestCase("intRangeLiteral.kt", "1", "..", "3", numericLiterals)
        generateRangeOperatorTestCase("intUntil.kt", "1", "until", "3", numericLiterals)
        generateRangeOperatorTestCase("intDownTo.kt", "3", "downTo", "1", listOf("1"))

        generateRangeOperatorTestCase("longRangeLiteral.kt", "1L", "..", "3L", numericLiterals)
        generateRangeOperatorTestCase("longUntil.kt", "1L", "until", "3L", numericLiterals)
        generateRangeOperatorTestCase("longDownTo.kt", "3L", "downTo", "1L", listOf("1L"))

        generateRangeOperatorTestCase("floatRangeLiteral.kt", "1.0F", "..", "3.0F", numericLiterals)

        generateRangeOperatorTestCase("doubleRangeLiteral.kt", "1.0", "..", "3.0", numericLiterals)

        generateMatrixTestCase(
                "arrayIndices.kt",
                listOf("intArray.indices", "objectArray.indices", "emptyIntArray.indices", "emptyObjectArray.indices"),
                numericLiterals,
                """val intArray = intArrayOf(1, 2, 3)
                    |val objectArray = arrayOf(1, 2, 3)
                    |val emptyIntArray = intArrayOf()
                    |val emptyObjectArray = arrayOf<Any>()
                """.trimMargin()
        )

        generateMatrixTestCase(
                "collectionIndices.kt",
                listOf("collection.indices", "emptyCollection.indices"),
                numericLiterals,
                """val collection = listOf(1, 2, 3)
                    |val emptyCollection = listOf<Any>()
                """.trimMargin()
        )

        generateMatrixTestCase(
                "charSequenceIndices.kt",
                listOf("charSequence.indices", "emptyCharSequence.indices"),
                numericLiterals,
                """val charSequence: CharSequence = "123"
                    |val emptyCharSequence: CharSequence = ""
                """.trimMargin()
        )
    }
}