/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2000 - 2008, Red Hat Middleware LLC, and individual contributors
 * as indicated by the @author tags. See the copyright.txt file in the
 * distribution for a full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.cache;

import org.jboss.cache.factories.annotations.NonVolatile;

import java.util.concurrent.ConcurrentHashMap;

/**
 * An extension of the ConcurrentHashMap that acts as a container for regions.
 *
 * @author Manik Surtani (<a href="mailto:manik AT jboss DOT org">manik AT jboss DOT org</a>)
 * @since 3.0
 */
@NonVolatile
public class RegionRegistry extends ConcurrentHashMap<Fqn, Region>
{
   private static final long serialVersionUID = -2480683437308480252L;
   
   private volatile int largestFqnLength;

   public RegionRegistry()
   {
      // default CHM constructor.  Potentially fine-tune later?
      super();
   }

   @Override
   public Region put(Fqn f, Region r)
   {
      largestFqnLength = Math.max(largestFqnLength, f.size());
      return super.put(f, r);
   }

   @Override
   public Region putIfAbsent(Fqn f, Region r)
   {
      Region rV = super.putIfAbsent(f, r);
      if (rV == null) largestFqnLength = Math.max(largestFqnLength, f.size());
      return rV;
   }

   @Override
   public Region replace(Fqn f, Region r)
   {
      Region rV = super.replace(f, r);
      if (rV != null) largestFqnLength = Math.max(largestFqnLength, f.size());
      return rV;
   }

   @Override
   public boolean replace(Fqn f, Region oldR, Region newR)
   {
      boolean rV = super.replace(f, oldR, newR);
      if (rV) largestFqnLength = Math.max(largestFqnLength, f.size());
      return rV;
   }

   public int getLargestFqnLength()
   {
      return largestFqnLength;
   }
}
