/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.common.server;

import com.google.common.util.concurrent.Uninterruptibles;
import java.io.Closeable;
import java.io.IOException;
import java.net.MalformedURLException;
import java.rmi.server.RMIClientSocketFactory;
import java.rmi.server.RMISocketFactory;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.BooleanSupplier;
import java.util.function.Supplier;
import javax.management.JMX;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.Notification;
import javax.management.NotificationListener;
import javax.management.ObjectName;
import javax.management.remote.JMXConnectionNotification;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import javax.rmi.ssl.SslRMIClientSocketFactory;
import org.apache.cassandra.sidecar.common.DataObjectBuilder;
import org.apache.cassandra.sidecar.common.server.exceptions.JmxAuthenticationException;
import org.apache.cassandra.sidecar.common.server.utils.DurationSpec;
import org.apache.cassandra.sidecar.common.server.utils.MillisecondBoundConfiguration;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class JmxClient
implements NotificationListener,
Closeable {
    public static final String JMX_PROTOCOL = "rmi";
    public static final String JMX_URL_PATH_FORMAT = "/jndi/rmi://%s:%d/jmxrmi";
    public static final String REGISTRY_CONTEXT_SOCKET_FACTORY = "com.sun.jndi.rmi.factory.socket";
    private static final Logger LOGGER = LoggerFactory.getLogger(JmxClient.class);
    private final JMXServiceURL jmxServiceURL;
    private MBeanServerConnection mBeanServerConnection;
    private boolean connected = false;
    private JMXConnector jmxConnector;
    private final Supplier<String> roleSupplier;
    private final Supplier<String> passwordSupplier;
    private final BooleanSupplier enableSslSupplier;
    private final int connectionMaxRetries;
    private final DurationSpec connectionRetryDelay;
    private final Set<NotificationListener> registeredNotificationListeners = Collections.newSetFromMap(new ConcurrentHashMap());

    protected JmxClient(Builder builder) {
        this.jmxServiceURL = builder.jmxServiceURL != null ? builder.jmxServiceURL : JmxClient.buildJmxServiceURL(Objects.requireNonNull(builder.host, "host is required"), builder.port);
        Objects.requireNonNull(this.jmxServiceURL, "jmxServiceUrl is required");
        this.roleSupplier = Objects.requireNonNull(builder.roleSupplier, "roleSupplier is required");
        this.passwordSupplier = Objects.requireNonNull(builder.passwordSupplier, "passwordSupplier is required");
        this.enableSslSupplier = Objects.requireNonNull(builder.enableSslSupplier, "enableSslSupplier is required");
        Preconditions.checkArgument((builder.connectionMaxRetries > 0 ? 1 : 0) != 0, (String)"connectionMaxRetries must be a positive integer");
        this.connectionMaxRetries = builder.connectionMaxRetries;
        this.connectionRetryDelay = builder.connectionRetryDelay;
    }

    public <C> C proxy(Class<C> clientClass, String remoteName) {
        this.checkConnection();
        try {
            ObjectName name = new ObjectName(remoteName);
            return JMX.newMBeanProxy(this.mBeanServerConnection, name, clientClass);
        }
        catch (MalformedObjectNameException e) {
            throw new RuntimeException(String.format("Invalid remote object name '%s'", remoteName), e);
        }
    }

    public void registerListener(NotificationListener notificationListener) {
        this.registeredNotificationListeners.add(notificationListener);
    }

    public void unregisterListener(NotificationListener notificationListener) {
        this.registeredNotificationListeners.remove(notificationListener);
    }

    private RMIClientSocketFactory rmiClientSocketFactory(boolean enableSsl) {
        return enableSsl ? new SslRMIClientSocketFactory() : RMISocketFactory.getDefaultSocketFactory();
    }

    protected synchronized void checkConnection() {
        if (!this.connected) {
            this.connect();
        }
    }

    protected void connect() {
        int attempts;
        int maxAttempts = this.connectionMaxRetries;
        Throwable lastThrown = null;
        for (attempts = 1; attempts <= maxAttempts; ++attempts) {
            try {
                this.connectInternal(attempts);
                return;
            }
            catch (SecurityException securityException) {
                this.connected = false;
                String errorMessage = securityException.getMessage() != null ? securityException.getMessage() : "JMX Authentication failed";
                throw new JmxAuthenticationException(errorMessage, securityException);
            }
            catch (RuntimeException runtimeException) {
                throw new JmxAuthenticationException(runtimeException);
            }
            catch (Throwable t) {
                lastThrown = t;
                if (attempts >= maxAttempts) continue;
                LOGGER.info("Could not connect to JMX on {} after {} attempts. Will retry.", new Object[]{this.jmxServiceURL, attempts, t});
                Uninterruptibles.sleepUninterruptibly((long)this.connectionRetryDelay.quantity(), (TimeUnit)this.connectionRetryDelay.unit());
                continue;
            }
        }
        String error = "Failed to connect to JMX, which was unreachable after " + attempts + " attempts.";
        LOGGER.error(error, lastThrown);
        throw new RuntimeException(error, lastThrown);
    }

    protected void connectInternal(int currentAttempt) throws IOException {
        this.jmxConnector = JMXConnectorFactory.connect(this.jmxServiceURL, this.buildJmxEnv());
        this.jmxConnector.addConnectionNotificationListener(this, null, null);
        this.mBeanServerConnection = this.jmxConnector.getMBeanServerConnection();
        this.connected = true;
        LOGGER.info("Connected to JMX server at {} after {} attempt(s)", (Object)this.jmxServiceURL, (Object)currentAttempt);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void handleNotification(Notification notification, Object handback) {
        JMXConnectionNotification connectNotice;
        String type;
        if (notification instanceof JMXConnectionNotification && ((type = (connectNotice = (JMXConnectionNotification)notification).getType()).equals("jmx.remote.connection.closed") || type.equals("jmx.remote.connection.failed") || type.equals("jmx.remote.connection.notifs.lost") || type.equals("jmx.remote.connection.opened"))) {
            boolean justConnected = type.equals("jmx.remote.connection.opened");
            JmxClient jmxClient = this;
            synchronized (jmxClient) {
                this.connected = justConnected;
            }
            this.forwardNotification(notification, handback);
        }
    }

    private void forwardNotification(Notification notification, Object handback) {
        this.registeredNotificationListeners.forEach(listener -> listener.handleNotification(notification, handback));
    }

    public boolean isConnected() {
        return this.connected;
    }

    public String host() {
        return this.jmxServiceURL.getHost();
    }

    public int port() {
        return this.jmxServiceURL.getPort();
    }

    private static JMXServiceURL buildJmxServiceURL(String host, int port) {
        if (host == null) {
            return null;
        }
        try {
            return new JMXServiceURL(JMX_PROTOCOL, host, port, JmxClient.jmxUrlPath(host, port));
        }
        catch (MalformedURLException e) {
            String errorMessage = String.format("Unable to build JMXServiceURL for host=%s, port=%d", host, port);
            throw new RuntimeException(errorMessage, e);
        }
    }

    private Map<String, Object> buildJmxEnv() {
        String role = this.roleSupplier.get();
        String password = this.passwordSupplier.get();
        boolean enableSsl = this.enableSslSupplier.getAsBoolean();
        HashMap<String, Object> jmxEnv = new HashMap<String, Object>();
        if (role != null && password != null) {
            String[] credentials = new String[]{role, password};
            jmxEnv.put("jmx.remote.credentials", credentials);
        }
        jmxEnv.put(REGISTRY_CONTEXT_SOCKET_FACTORY, this.rmiClientSocketFactory(enableSsl));
        return jmxEnv;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() throws IOException {
        JMXConnector connector;
        JmxClient jmxClient = this;
        synchronized (jmxClient) {
            connector = this.jmxConnector;
            this.jmxConnector = null;
            this.connected = false;
        }
        if (connector != null) {
            connector.close();
        }
    }

    private static String jmxUrlPath(String host, int port) {
        return String.format(JMX_URL_PATH_FORMAT, JmxClient.maybeAddSquareBrackets(host), port);
    }

    private static String maybeAddSquareBrackets(String host) {
        if (host == null || host.isEmpty() || host.charAt(0) == '[' || !host.contains(":")) {
            return host;
        }
        return "[" + host + "]";
    }

    public static Builder builder() {
        return new Builder();
    }

    public static final class Builder
    implements DataObjectBuilder<Builder, JmxClient> {
        private JMXServiceURL jmxServiceURL;
        private String host;
        private int port;
        private Supplier<String> roleSupplier = () -> null;
        private Supplier<String> passwordSupplier = () -> null;
        private BooleanSupplier enableSslSupplier = () -> false;
        private int connectionMaxRetries = 3;
        private DurationSpec connectionRetryDelay = MillisecondBoundConfiguration.parse("200ms");

        private Builder() {
        }

        public Builder self() {
            return this;
        }

        public Builder host(String host) {
            return (Builder)this.update(b -> {
                b.host = host;
            });
        }

        public Builder port(int port) {
            return (Builder)this.update(b -> {
                b.port = port;
            });
        }

        public Builder jmxServiceURL(JMXServiceURL jmxServiceURL) {
            return (Builder)this.update(b -> {
                b.jmxServiceURL = jmxServiceURL;
            });
        }

        public Builder roleSupplier(Supplier<String> roleSupplier) {
            return (Builder)this.update(b -> {
                b.roleSupplier = Objects.requireNonNull(roleSupplier, "roleSupplier must be provided");
            });
        }

        public Builder role(String role) {
            return (Builder)this.update(b -> {
                b.roleSupplier = () -> role;
            });
        }

        public Builder passwordSupplier(Supplier<String> passwordSupplier) {
            return (Builder)this.update(b -> {
                b.passwordSupplier = Objects.requireNonNull(passwordSupplier, "passwordSupplier must be provided");
            });
        }

        public Builder password(String password) {
            return (Builder)this.update(b -> {
                b.passwordSupplier = () -> password;
            });
        }

        public Builder enableSslSupplier(BooleanSupplier enableSslSupplier) {
            return (Builder)this.update(b -> {
                b.enableSslSupplier = enableSslSupplier;
            });
        }

        public Builder enableSsl(boolean enableSsl) {
            return (Builder)this.update(b -> {
                b.enableSslSupplier = () -> enableSsl;
            });
        }

        public Builder connectionMaxRetries(int connectionMaxRetries) {
            return (Builder)this.update(b -> {
                b.connectionMaxRetries = connectionMaxRetries;
            });
        }

        public Builder connectionRetryDelay(DurationSpec connectionRetryDelay) {
            return (Builder)this.update(b -> {
                b.connectionRetryDelay = connectionRetryDelay;
            });
        }

        public JmxClient build() {
            return new JmxClient(this);
        }
    }
}

