/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.io.channels;

import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.ClosedChannelException;
import java.nio.channels.SeekableByteChannel;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.Random;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;
import org.junit.jupiter.params.ParameterizedTest;
import org.junit.jupiter.params.provider.CsvSource;
import org.junit.jupiter.params.provider.ValueSource;

abstract class AbstractSeekableByteChannelTest {
    private SeekableByteChannel channel;
    @TempDir
    protected Path tempDir;
    protected Path tempFile;
    private final Random random = new Random(42L);

    AbstractSeekableByteChannelTest() {
    }

    protected abstract SeekableByteChannel createChannel() throws IOException;

    @BeforeEach
    void setUp() throws IOException {
        this.tempFile = this.tempDir.resolve(this.getClass().getSimpleName() + ".tmp");
        this.channel = this.createChannel();
    }

    @AfterEach
    void tearDown() throws IOException {
        if (this.channel != null && this.channel.isOpen()) {
            this.channel.close();
        }
        if (this.tempFile != null && Files.exists(this.tempFile, new LinkOption[0])) {
            Files.delete(this.tempFile);
        }
    }

    @Test
    void testCloseMultipleTimes() throws IOException {
        this.channel.close();
        this.channel.close();
        Assertions.assertFalse((boolean)this.channel.isOpen());
    }

    @Test
    void testConcurrentPositionAndSizeQueries() throws IOException {
        byte[] data = "test data".getBytes();
        this.channel.write(ByteBuffer.wrap(data));
        long size = this.channel.size();
        long position = this.channel.position();
        Assertions.assertEquals((long)data.length, (long)size);
        Assertions.assertEquals((long)data.length, (long)position);
        Assertions.assertEquals((long)this.channel.size(), (long)size);
        Assertions.assertEquals((long)this.channel.position(), (long)position);
    }

    @Test
    void testIsOpenAfterClose() throws IOException {
        this.channel.close();
        Assertions.assertFalse((boolean)this.channel.isOpen());
    }

    @Test
    void testIsOpennOnNew() {
        Assertions.assertTrue((boolean)this.channel.isOpen());
    }

    @Test
    void testPartialWritesAndReads() throws IOException {
        byte[] data = "0123456789".getBytes();
        this.channel.write(ByteBuffer.wrap(data, 0, 5));
        this.channel.write(ByteBuffer.wrap(data, 5, 5));
        Assertions.assertEquals((long)10L, (long)this.channel.size());
        this.channel.position(0L);
        ByteBuffer buffer1 = ByteBuffer.allocate(3);
        ByteBuffer buffer2 = ByteBuffer.allocate(7);
        Assertions.assertEquals((int)3, (int)this.channel.read(buffer1));
        Assertions.assertEquals((int)7, (int)this.channel.read(buffer2));
        Assertions.assertArrayEquals((byte[])Arrays.copyOf(data, 3), (byte[])buffer1.array());
        Assertions.assertArrayEquals((byte[])Arrays.copyOfRange(data, 3, 10), (byte[])buffer2.array());
    }

    @Test
    void testPositionBeyondSize() throws IOException {
        this.channel.write(ByteBuffer.wrap("test".getBytes()));
        this.channel.position(100L);
        Assertions.assertEquals((long)100L, (long)this.channel.position());
        Assertions.assertEquals((long)4L, (long)this.channel.size());
    }

    @ParameterizedTest
    @CsvSource(value={"0, 0", "5, 5", "10, 10", "100, 100"})
    void testPositionInBounds(long newPosition, long expectedPosition) throws IOException {
        byte[] data = new byte[200];
        this.random.nextBytes(data);
        this.channel.write(ByteBuffer.wrap(data));
        SeekableByteChannel result = this.channel.position(newPosition);
        Assertions.assertSame((Object)this.channel, (Object)result);
        Assertions.assertEquals((long)expectedPosition, (long)this.channel.position());
    }

    @Test
    void testPositionNegative() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.channel.position(-1L));
    }

    @Test
    void testPositionOnClosed() throws IOException {
        this.channel.close();
        Assertions.assertThrows(ClosedChannelException.class, () -> this.channel.position());
        Assertions.assertThrows(ClosedChannelException.class, () -> this.channel.position(0L));
    }

    @Test
    void testPositionOnNew() throws IOException {
        Assertions.assertEquals((long)0L, (long)this.channel.position());
    }

    @Test
    void testRandomAccess() throws IOException {
        int[] positions;
        byte[] data = new byte[1000];
        this.random.nextBytes(data);
        this.channel.write(ByteBuffer.wrap(data));
        for (int pos : positions = new int[]{100, 500, 0, 999, 250}) {
            this.channel.position(pos);
            Assertions.assertEquals((long)pos, (long)this.channel.position());
            ByteBuffer buffer = ByteBuffer.allocate(1);
            int read = this.channel.read(buffer);
            if (pos >= data.length) continue;
            Assertions.assertEquals((int)1, (int)read);
            Assertions.assertEquals((byte)data[pos], (byte)buffer.get(0));
        }
    }

    @Test
    void testReadAtEndOfFile() throws IOException {
        this.channel.write(ByteBuffer.wrap("test".getBytes()));
        ByteBuffer buffer = ByteBuffer.allocate(10);
        int read = this.channel.read(buffer);
        Assertions.assertEquals((int)-1, (int)read);
    }

    @Test
    void testReadBuffer() throws IOException {
        byte[] data = "test".getBytes();
        this.channel.write(ByteBuffer.wrap(data));
        this.channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(100);
        int read = this.channel.read(buffer);
        Assertions.assertEquals((int)data.length, (int)read);
        Assertions.assertEquals((long)data.length, (long)this.channel.position());
        byte[] readData = new byte[read];
        buffer.flip();
        buffer.get(readData);
        Assertions.assertArrayEquals((byte[])data, (byte[])readData);
    }

    @Test
    void testReadBytes() throws IOException {
        byte[] data = "Hello, World!".getBytes();
        this.channel.write(ByteBuffer.wrap(data));
        this.channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(data.length);
        int read = this.channel.read(buffer);
        Assertions.assertEquals((int)data.length, (int)read);
        Assertions.assertArrayEquals((byte[])data, (byte[])buffer.array());
        Assertions.assertEquals((long)data.length, (long)this.channel.position());
    }

    @Test
    void testReadClosed() throws IOException {
        this.channel.close();
        ByteBuffer buffer = ByteBuffer.allocate(10);
        Assertions.assertThrows(ClosedChannelException.class, () -> this.channel.read(buffer));
    }

    @Test
    void testReadEmpty() throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(10);
        int read = this.channel.read(buffer);
        Assertions.assertEquals((int)-1, (int)read);
        Assertions.assertEquals((int)0, (int)buffer.position());
    }

    @Test
    void testReadNull() {
        Assertions.assertThrows(NullPointerException.class, () -> this.channel.read(null));
    }

    @Test
    void testReadSingleByte() throws IOException {
        this.channel.write(ByteBuffer.wrap(new byte[]{42}));
        this.channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(1);
        int read = this.channel.read(buffer);
        Assertions.assertEquals((int)1, (int)read);
        Assertions.assertEquals((int)42, (int)buffer.get(0));
        Assertions.assertEquals((long)1L, (long)this.channel.position());
    }

    @Test
    void testSizeAfterTruncateToLargerSize() throws IOException {
        this.channel.write(ByteBuffer.wrap("Hello".getBytes()));
        Assertions.assertEquals((long)5L, (long)this.channel.size());
        this.channel.truncate(10L);
        Assertions.assertEquals((long)5L, (long)this.channel.size());
    }

    @Test
    void testSizeAfterWrite() throws IOException {
        Assertions.assertEquals((long)0L, (long)this.channel.size());
        this.channel.write(ByteBuffer.wrap("Hello".getBytes()));
        Assertions.assertEquals((long)5L, (long)this.channel.size());
        this.channel.write(ByteBuffer.wrap(" World".getBytes()));
        Assertions.assertEquals((long)11L, (long)this.channel.size());
    }

    @Test
    void testSizeOnClosed() throws IOException {
        this.channel.close();
        Assertions.assertThrows(ClosedChannelException.class, () -> this.channel.size());
    }

    @Test
    void testSizeOnNew() throws IOException {
        Assertions.assertEquals((long)0L, (long)this.channel.size());
    }

    @Test
    void testSizeSameOnOverwrite() throws IOException {
        this.channel.write(ByteBuffer.wrap("Hello World".getBytes()));
        Assertions.assertEquals((long)11L, (long)this.channel.size());
        this.channel.position(6L);
        this.channel.write(ByteBuffer.wrap("Test".getBytes()));
        Assertions.assertEquals((long)11L, (long)this.channel.size());
    }

    @Test
    void testTruncateNegative() {
        Assertions.assertThrows(IllegalArgumentException.class, () -> this.channel.truncate(-1L));
    }

    @Test
    void testTruncateShrinks() throws IOException {
        this.channel.write(ByteBuffer.wrap("Hello World".getBytes()));
        Assertions.assertEquals((long)11L, (long)this.channel.size());
        this.channel.truncate(5L);
        Assertions.assertEquals((long)5L, (long)this.channel.size());
        if (this.channel.position() > 5L) {
            Assertions.assertEquals((long)5L, (long)this.channel.position());
        }
    }

    @Test
    void testWriteBeyondSizeGrows() throws IOException {
        this.channel.position(100L);
        byte[] data = "test".getBytes();
        this.channel.write(ByteBuffer.wrap(data));
        Assertions.assertEquals((long)104L, (long)this.channel.size());
        Assertions.assertEquals((long)104L, (long)this.channel.position());
        this.channel.position(0L);
        ByteBuffer buffer = ByteBuffer.allocate(100);
        this.channel.read(buffer);
        byte[] expectedGap = new byte[100];
        Assertions.assertArrayEquals((byte[])expectedGap, (byte[])buffer.array());
    }

    @ParameterizedTest
    @ValueSource(ints={1, 10, 100, 1000, 10000})
    void testWriteDifferentSizes(int size) throws IOException {
        byte[] data = new byte[size];
        this.random.nextBytes(data);
        ByteBuffer buffer = ByteBuffer.wrap(data);
        int byteCount = this.channel.write(buffer);
        Assertions.assertEquals((int)size, (int)byteCount);
        Assertions.assertEquals((long)size, (long)this.channel.position());
        Assertions.assertEquals((long)size, (long)this.channel.size());
    }

    @Test
    void testWriteEmpty() throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(0);
        int byteCount = this.channel.write(buffer);
        Assertions.assertEquals((int)0, (int)byteCount);
        Assertions.assertEquals((long)0L, (long)this.channel.position());
        Assertions.assertEquals((long)0L, (long)this.channel.size());
    }

    @Test
    void testWriteNull() {
        Assertions.assertThrows(NullPointerException.class, () -> this.channel.write(null));
    }

    @Test
    void testWritePositionReadVerify() throws IOException {
        byte[] originalData = "Hello, SeekableByteChannel World!".getBytes();
        this.channel.write(ByteBuffer.wrap(originalData));
        this.channel.position(0L);
        ByteBuffer readBuffer = ByteBuffer.allocate(originalData.length);
        int bytesRead = this.channel.read(readBuffer);
        Assertions.assertEquals((int)originalData.length, (int)bytesRead);
        Assertions.assertArrayEquals((byte[])originalData, (byte[])readBuffer.array());
    }

    @Test
    void testWriteSingleByte() throws IOException {
        ByteBuffer buffer = ByteBuffer.allocate(1);
        buffer.put((byte)42);
        buffer.flip();
        int written = this.channel.write(buffer);
        Assertions.assertEquals((int)1, (int)written);
        Assertions.assertEquals((long)1L, (long)this.channel.position());
        Assertions.assertEquals((long)1L, (long)this.channel.size());
    }

    @Test
    void testWriteToClosedChannel() throws IOException {
        this.channel.close();
        ByteBuffer buffer = ByteBuffer.wrap("test".getBytes());
        Assertions.assertThrows(ClosedChannelException.class, () -> this.channel.write(buffer));
    }

    @Test
    void tesWriteBytes() throws IOException {
        byte[] data = "Hello, World!".getBytes();
        ByteBuffer buffer = ByteBuffer.wrap(data);
        int byteCount = this.channel.write(buffer);
        Assertions.assertEquals((int)data.length, (int)byteCount);
        Assertions.assertEquals((long)data.length, (long)this.channel.position());
        Assertions.assertEquals((long)data.length, (long)this.channel.size());
    }
}

