/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.storage.s3;

import com.amazonaws.AmazonClientException;
import com.amazonaws.ClientConfiguration;
import com.amazonaws.Protocol;
import com.amazonaws.SdkClientException;
import com.amazonaws.services.s3.model.AccessControlList;
import com.amazonaws.services.s3.model.AmazonS3Exception;
import com.amazonaws.services.s3.model.CanonicalGrantee;
import com.amazonaws.services.s3.model.DeleteObjectsRequest;
import com.amazonaws.services.s3.model.Grant;
import com.amazonaws.services.s3.model.Grantee;
import com.amazonaws.services.s3.model.ObjectMetadata;
import com.amazonaws.services.s3.model.Permission;
import com.amazonaws.services.s3.model.PutObjectRequest;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Joiner;
import com.google.common.base.Predicate;
import com.google.common.collect.ImmutableList;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.druid.common.aws.AWSClientConfig;
import org.apache.druid.common.aws.AWSClientUtil;
import org.apache.druid.common.aws.AWSEndpointConfig;
import org.apache.druid.common.aws.AWSProxyConfig;
import org.apache.druid.data.input.impl.CloudObjectLocation;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.RetryUtils;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.URIs;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.storage.s3.ObjectSummaryIterator;
import org.apache.druid.storage.s3.ServerSideEncryptingAmazonS3;

public class S3Utils {
    private static final String SCHEME = "s3";
    private static final Joiner JOINER = Joiner.on((String)"/").skipNulls();
    private static final Logger log = new Logger(S3Utils.class);
    public static final String ERROR_ENTITY_TOO_LARGE = "EntityTooLarge";
    public static final Predicate<Throwable> S3RETRY = new Predicate<Throwable>(){

        public boolean apply(Throwable e) {
            if (e == null) {
                return false;
            }
            if (e instanceof IOException) {
                if (e.getCause() != null) {
                    return this.apply(e.getCause());
                }
                return true;
            }
            if (e instanceof SdkClientException && e.getMessage().contains("Data read has a different length than the expected")) {
                return true;
            }
            if (e instanceof SdkClientException && e.getMessage().contains("Unable to execute HTTP request")) {
                return true;
            }
            if (e instanceof SdkClientException && e.getMessage().contains("Unable to find a region via the region provider chain")) {
                return true;
            }
            if (e instanceof InterruptedException) {
                Thread.interrupted();
                return false;
            }
            if (e instanceof AmazonClientException) {
                return AWSClientUtil.isClientExceptionRecoverable((AmazonClientException)((AmazonClientException)e));
            }
            return this.apply(e.getCause());
        }
    };

    public static <T> T retryS3Operation(RetryUtils.Task<T> f) throws Exception {
        return (T)RetryUtils.retry(f, S3RETRY, (int)10);
    }

    public static <T> T retryS3Operation(RetryUtils.Task<T> f, int maxRetries) throws Exception {
        return (T)RetryUtils.retry(f, S3RETRY, (int)maxRetries);
    }

    @Nullable
    public static String getS3ErrorCode(Throwable e) {
        if (e == null) {
            return null;
        }
        if (e instanceof AmazonS3Exception) {
            return ((AmazonS3Exception)e).getErrorCode();
        }
        return S3Utils.getS3ErrorCode(e.getCause());
    }

    static boolean isObjectInBucketIgnoringPermission(ServerSideEncryptingAmazonS3 s3Client, String bucketName, String objectKey) {
        try {
            return s3Client.doesObjectExist(bucketName, objectKey);
        }
        catch (AmazonS3Exception e) {
            if (e.getStatusCode() == 404) {
                return true;
            }
            throw e;
        }
    }

    public static Iterator<S3ObjectSummary> objectSummaryIterator(ServerSideEncryptingAmazonS3 s3Client, Iterable<URI> prefixes, int maxListingLength) {
        return new ObjectSummaryIterator(s3Client, prefixes, maxListingLength);
    }

    public static Iterator<S3ObjectSummary> objectSummaryIterator(ServerSideEncryptingAmazonS3 s3Client, Iterable<URI> prefixes, int maxListingLength, int maxRetries) {
        return new ObjectSummaryIterator(s3Client, prefixes, maxListingLength, maxRetries);
    }

    public static URI summaryToUri(S3ObjectSummary object) {
        return S3Utils.summaryToCloudObjectLocation(object).toUri(SCHEME);
    }

    public static CloudObjectLocation summaryToCloudObjectLocation(S3ObjectSummary object) {
        return new CloudObjectLocation(object.getBucketName(), object.getKey());
    }

    static String constructSegmentPath(String baseKey, String storageDir) {
        return JOINER.join((Object)(baseKey.isEmpty() ? null : baseKey), (Object)storageDir, new Object[0]) + "/index.zip";
    }

    static AccessControlList grantFullControlToBucketOwner(ServerSideEncryptingAmazonS3 s3Client, String bucket) {
        AccessControlList acl = s3Client.getBucketAcl(bucket);
        acl.grantAllPermissions(new Grant[]{new Grant((Grantee)new CanonicalGrantee(acl.getOwner().getId()), Permission.FullControl)});
        return acl;
    }

    public static String extractS3Key(URI uri) {
        return StringUtils.maybeRemoveLeadingSlash((String)uri.getPath());
    }

    public static URI checkURI(URI uri) {
        if (uri.getScheme().equalsIgnoreCase("s3_zip")) {
            uri = URI.create(SCHEME + uri.toString().substring("s3_zip".length()));
        }
        return CloudObjectLocation.validateUriScheme((String)SCHEME, (URI)uri);
    }

    public static ObjectMetadata getSingleObjectMetadata(ServerSideEncryptingAmazonS3 s3Client, String bucket, String key) {
        try {
            return (ObjectMetadata)S3Utils.retryS3Operation(() -> s3Client.getObjectMetadata(bucket, key));
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static void deleteObjectsInPath(ServerSideEncryptingAmazonS3 s3Client, int maxListingLength, String bucket, String prefix, Predicate<S3ObjectSummary> filter) throws Exception {
        S3Utils.deleteObjectsInPath(s3Client, maxListingLength, bucket, prefix, filter, 10);
    }

    public static void deleteObjectsInPath(ServerSideEncryptingAmazonS3 s3Client, int maxListingLength, String bucket, String prefix, Predicate<S3ObjectSummary> filter, int maxRetries) throws Exception {
        log.debug("Deleting directory at bucket: [%s], path: [%s]", new Object[]{bucket, prefix});
        ArrayList<DeleteObjectsRequest.KeyVersion> keysToDelete = new ArrayList<DeleteObjectsRequest.KeyVersion>(maxListingLength);
        ObjectSummaryIterator iterator = new ObjectSummaryIterator(s3Client, (Iterable<URI>)ImmutableList.of((Object)new CloudObjectLocation(bucket, prefix).toUri(SCHEME)), maxListingLength);
        while (iterator.hasNext()) {
            S3ObjectSummary nextObject = iterator.next();
            if (!filter.apply((Object)nextObject)) continue;
            keysToDelete.add(new DeleteObjectsRequest.KeyVersion(nextObject.getKey()));
            if (keysToDelete.size() != maxListingLength) continue;
            S3Utils.deleteBucketKeys(s3Client, bucket, keysToDelete, maxRetries);
            keysToDelete.clear();
        }
        if (keysToDelete.size() > 0) {
            S3Utils.deleteBucketKeys(s3Client, bucket, keysToDelete, maxRetries);
        }
    }

    public static void deleteBucketKeys(ServerSideEncryptingAmazonS3 s3Client, String bucket, List<DeleteObjectsRequest.KeyVersion> keysToDelete, int retries) throws Exception {
        if (keysToDelete != null && log.isDebugEnabled()) {
            List keys = keysToDelete.stream().map(DeleteObjectsRequest.KeyVersion::getKey).collect(Collectors.toList());
            log.debug("Deleting keys from bucket: [%s], keys: [%s]", new Object[]{bucket, keys});
        }
        DeleteObjectsRequest deleteRequest = new DeleteObjectsRequest(bucket).withKeys(keysToDelete);
        S3Utils.retryS3Operation(() -> {
            s3Client.deleteObjects(deleteRequest);
            return null;
        }, retries);
        log.info("Deleted %d files", new Object[]{keysToDelete.size()});
    }

    static void uploadFileIfPossible(ServerSideEncryptingAmazonS3 service, boolean disableAcl, String bucket, String key, File file) throws InterruptedException {
        PutObjectRequest putObjectRequest = new PutObjectRequest(bucket, key, file);
        if (!disableAcl) {
            putObjectRequest.setAccessControlList(S3Utils.grantFullControlToBucketOwner(service, bucket));
        }
        log.info("Pushing [%s] to bucket[%s] and key[%s].", new Object[]{file, bucket, key});
        service.upload(putObjectRequest);
    }

    @Nullable
    private static Protocol parseProtocol(@Nullable String protocol) {
        if (protocol == null) {
            return null;
        }
        if (protocol.equalsIgnoreCase("http")) {
            return Protocol.HTTP;
        }
        if (protocol.equalsIgnoreCase("https")) {
            return Protocol.HTTPS;
        }
        throw new IAE("Unknown protocol[%s]", new Object[]{protocol});
    }

    public static Protocol determineProtocol(AWSClientConfig clientConfig, AWSEndpointConfig endpointConfig) {
        Protocol protocolFromClientConfig = S3Utils.parseProtocol(clientConfig.getProtocol());
        String endpointUrl = endpointConfig.getUrl();
        if (org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)endpointUrl)) {
            URI uri = URIs.parse((String)endpointUrl, (String)protocolFromClientConfig.toString());
            Protocol protocol = S3Utils.parseProtocol(uri.getScheme());
            if (protocol != null && protocol != protocolFromClientConfig) {
                log.warn("[%s] protocol will be used for endpoint [%s]", new Object[]{protocol, endpointUrl});
            }
            return protocol;
        }
        return protocolFromClientConfig;
    }

    public static ClientConfiguration setProxyConfig(ClientConfiguration conf, AWSProxyConfig proxyConfig) {
        if (org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)proxyConfig.getHost())) {
            conf.setProxyHost(proxyConfig.getHost());
        }
        if (proxyConfig.getPort() != -1) {
            conf.setProxyPort(proxyConfig.getPort());
        }
        if (org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)proxyConfig.getUsername())) {
            conf.setProxyUsername(proxyConfig.getUsername());
        }
        if (org.apache.commons.lang3.StringUtils.isNotEmpty((CharSequence)proxyConfig.getPassword())) {
            conf.setProxyPassword(proxyConfig.getPassword());
        }
        return conf;
    }
}

