/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.hadoop.hdfs.tools.snapshot;


import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.SnapshotStatus;
import org.apache.hadoop.hdfs.tools.AdminHelper;
import org.apache.hadoop.util.Tool;
import org.apache.hadoop.util.ToolRunner;

/**
 * A tool used to list all snapshottable directories that are owned by the
 * current user. The tool returns all the snapshottable directories if the user
 * is a super user.
 */
@InterfaceAudience.Private
public class LsSnapshot extends Configured implements Tool {
  @Override
  public int run(String[] argv) throws Exception {
    String description = "hdfs lsSnapshot <snapshotDir>: \n" +
        "\tGet the list of snapshots for a snapshottable directory.\n";

    if(argv.length != 1) {
      System.err.println("Invalid no of arguments");
      System.err.println("Usage: \n" + description);
      return 1;
    }
    Path snapshotRoot = new Path(argv[0]);
    try {
      DistributedFileSystem dfs = AdminHelper.getDFS(getConf());
      SnapshotStatus[] stats = dfs.getSnapshotListing(snapshotRoot);
      SnapshotStatus.print(stats, System.out);
    } catch (Exception e) {
      String[] content = e.getLocalizedMessage().split("\n");
      System.err.println("lsSnapshot: " + content[0]);
      return 1;
    }
    return 0;
  }
  public static void main(String[] argv) throws Exception {
    int rc = ToolRunner.run(new LsSnapshot(), argv);
    System.exit(rc);
  }
}