/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pinot.plugin.filesystem;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.function.Consumer;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.FileUtil;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.RemoteIterator;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.pinot.shaded.com.google.common.base.Strings;
import org.apache.pinot.shaded.com.google.common.collect.ImmutableList;
import org.apache.pinot.spi.env.PinotConfiguration;
import org.apache.pinot.spi.filesystem.BasePinotFS;
import org.apache.pinot.spi.filesystem.FileMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HadoopPinotFS
extends BasePinotFS {
    private static final Logger LOGGER = LoggerFactory.getLogger(HadoopPinotFS.class);
    private static final String PRINCIPAL = "hadoop.kerberos.principle";
    private static final String KEYTAB = "hadoop.kerberos.keytab";
    private static final String HADOOP_CONF_PATH = "hadoop.conf.path";
    private static final String WRITE_CHECKSUM = "hadoop.write.checksum";
    private FileSystem _hadoopFS = null;
    private Configuration _hadoopConf;

    public void init(PinotConfiguration config) {
        try {
            this._hadoopConf = this.getConf(config.getProperty(HADOOP_CONF_PATH));
            this.authenticate(this._hadoopConf, config);
            this._hadoopFS = FileSystem.get((Configuration)this._hadoopConf);
            this._hadoopFS.setWriteChecksum(config.getProperty(WRITE_CHECKSUM, false));
            LOGGER.info("successfully initialized HadoopPinotFS");
        }
        catch (IOException e) {
            throw new RuntimeException("Could not initialize HadoopPinotFS", e);
        }
    }

    public boolean mkdir(URI uri) throws IOException {
        return this._hadoopFS.mkdirs(new Path(uri));
    }

    public boolean delete(URI segmentUri, boolean forceDelete) throws IOException {
        if (this.isDirectory(segmentUri) && this.listFiles(segmentUri, false).length > 0 && !forceDelete) {
            return false;
        }
        return this._hadoopFS.delete(new Path(segmentUri), true);
    }

    public boolean doMove(URI srcUri, URI dstUri) throws IOException {
        return this._hadoopFS.rename(new Path(srcUri), new Path(dstUri));
    }

    public boolean copyDir(URI srcUri, URI dstUri) throws IOException {
        Path source = new Path(srcUri);
        Path target = new Path(dstUri);
        RemoteIterator sourceFiles = this._hadoopFS.listStatusIterator(source);
        if (sourceFiles != null) {
            while (sourceFiles.hasNext()) {
                FileStatus sourceFile = (FileStatus)sourceFiles.next();
                Path sourceFilePath = sourceFile.getPath();
                if (sourceFile.isFile()) {
                    try {
                        FileUtil.copy((FileSystem)this._hadoopFS, (Path)sourceFilePath, (FileSystem)this._hadoopFS, (Path)new Path(target, sourceFilePath.getName()), (boolean)false, (Configuration)this._hadoopConf);
                    }
                    catch (FileNotFoundException e) {
                        LOGGER.warn("Not found file {}, skipping copying it...", (Object)sourceFilePath, (Object)e);
                    }
                    continue;
                }
                if (!sourceFile.isDirectory()) continue;
                try {
                    this.copyDir(sourceFilePath.toUri(), new Path(target, sourceFilePath.getName()).toUri());
                }
                catch (FileNotFoundException e) {
                    LOGGER.warn("Not found directory {}, skipping copying it...", (Object)sourceFilePath, (Object)e);
                }
            }
        }
        return true;
    }

    public boolean exists(URI fileUri) throws IOException {
        return this._hadoopFS.exists(new Path(fileUri));
    }

    public long length(URI fileUri) throws IOException {
        return this._hadoopFS.getFileStatus(new Path(fileUri)).getLen();
    }

    public String[] listFiles(URI fileUri, boolean recursive) throws IOException {
        ImmutableList.Builder builder = ImmutableList.builder();
        this.visitFiles(fileUri, recursive, f -> builder.add((Object)f.getPath().toString()));
        String[] listedFiles = (String[])builder.build().toArray((Object[])new String[0]);
        LOGGER.debug("Listed {} files from URI: {}, is recursive: {}", new Object[]{listedFiles.length, fileUri, recursive});
        return listedFiles;
    }

    public List<FileMetadata> listFilesWithMetadata(URI fileUri, boolean recursive) throws IOException {
        ImmutableList.Builder listBuilder = ImmutableList.builder();
        this.visitFiles(fileUri, recursive, f -> {
            FileMetadata.Builder fileBuilder = new FileMetadata.Builder().setFilePath(f.getPath().toString()).setLastModifiedTime(f.getModificationTime()).setLength(f.getLen()).setIsDirectory(f.isDirectory());
            listBuilder.add((Object)fileBuilder.build());
        });
        ImmutableList listedFiles = listBuilder.build();
        LOGGER.debug("Listed {} files from URI: {}, is recursive: {}", new Object[]{listedFiles.size(), fileUri, recursive});
        return listedFiles;
    }

    private void visitFiles(URI fileUri, boolean recursive, Consumer<FileStatus> visitor) throws IOException {
        Path path = new Path(fileUri);
        if (!this._hadoopFS.exists(path)) {
            throw new IllegalArgumentException("fileUri does not exist: " + String.valueOf(fileUri));
        }
        this.visitFileStatus(path, recursive, visitor);
    }

    private void visitFileStatus(Path path, boolean recursive, Consumer<FileStatus> visitor) throws IOException {
        FileStatus[] files;
        for (FileStatus file : files = this._hadoopFS.listStatus(path)) {
            visitor.accept(file);
            if (!file.isDirectory() || !recursive) continue;
            this.visitFileStatus(file.getPath(), true, visitor);
        }
    }

    public void copyToLocalFile(URI srcUri, File dstFile) throws Exception {
        LOGGER.debug("starting to fetch segment from hdfs");
        String dstFilePath = dstFile.getAbsolutePath();
        Path remoteFile = new Path(srcUri);
        Path localFile = new Path(dstFile.toURI());
        try {
            if (this._hadoopFS == null) {
                throw new RuntimeException("_hadoopFS client is not initialized when trying to copy files");
            }
            if (this._hadoopFS.isDirectory(remoteFile)) {
                throw new IllegalArgumentException(srcUri.toString() + " is a direactory");
            }
            long startMs = System.currentTimeMillis();
            this._hadoopFS.copyToLocalFile(remoteFile, localFile);
            LOGGER.debug("copied {} from hdfs to {} in local for size {}, take {} ms", new Object[]{srcUri, dstFilePath, dstFile.length(), System.currentTimeMillis() - startMs});
        }
        catch (IOException e) {
            LOGGER.warn("failed to fetch segment {} from hdfs to {}, might retry", new Object[]{srcUri, dstFile, e});
            throw e;
        }
    }

    public void copyFromLocalFile(File srcFile, URI dstUri) throws Exception {
        if (srcFile.isDirectory()) {
            throw new IllegalArgumentException(srcFile.getAbsolutePath() + " is a directory");
        }
        this._hadoopFS.copyFromLocalFile(new Path(srcFile.toURI()), new Path(dstUri));
    }

    public void copyFromLocalDir(File srcFile, URI dstUri) throws Exception {
        if (!srcFile.isDirectory()) {
            throw new IllegalArgumentException(srcFile.getAbsolutePath() + " is not a directory");
        }
        this._hadoopFS.copyFromLocalFile(new Path(srcFile.toURI()), new Path(dstUri));
    }

    public boolean isDirectory(URI uri) {
        try {
            return this._hadoopFS.getFileStatus(new Path(uri)).isDirectory();
        }
        catch (IOException e) {
            LOGGER.error("Could not get file status for {}", (Object)uri, (Object)e);
            throw new RuntimeException(e);
        }
    }

    public long lastModified(URI uri) {
        try {
            return this._hadoopFS.getFileStatus(new Path(uri)).getModificationTime();
        }
        catch (IOException e) {
            LOGGER.error("Could not get file status for {}", (Object)uri, (Object)e);
            throw new RuntimeException(e);
        }
    }

    public boolean touch(URI uri) throws IOException {
        Path path = new Path(uri);
        if (!this.exists(uri)) {
            FSDataOutputStream fos = this._hadoopFS.create(path);
            fos.close();
        } else {
            this._hadoopFS.setTimes(path, System.currentTimeMillis(), -1L);
        }
        return true;
    }

    public InputStream open(URI uri) throws IOException {
        Path path = new Path(uri);
        return this._hadoopFS.open(path);
    }

    private void authenticate(Configuration hadoopConf, PinotConfiguration configs) {
        String principal = configs.getProperty(PRINCIPAL);
        String keytab = configs.getProperty(KEYTAB);
        if (!Strings.isNullOrEmpty((String)principal) && !Strings.isNullOrEmpty((String)keytab)) {
            UserGroupInformation.setConfiguration((Configuration)hadoopConf);
            if (UserGroupInformation.isSecurityEnabled()) {
                try {
                    if (!UserGroupInformation.getCurrentUser().hasKerberosCredentials() || !UserGroupInformation.getCurrentUser().getUserName().equals(principal)) {
                        LOGGER.info("Trying to authenticate user {} with keytab {}..", (Object)principal, (Object)keytab);
                        UserGroupInformation.loginUserFromKeytab((String)principal, (String)keytab);
                    }
                }
                catch (IOException e) {
                    throw new RuntimeException(String.format("Failed to authenticate user principal [%s] with keytab [%s]", principal, keytab), e);
                }
            }
        }
    }

    private Configuration getConf(String hadoopConfPath) {
        Configuration hadoopConf = new Configuration();
        if (Strings.isNullOrEmpty((String)hadoopConfPath)) {
            LOGGER.warn("no hadoop conf path is provided, will rely on default config");
        } else {
            hadoopConf.addResource(new Path(hadoopConfPath, "core-site.xml"));
            hadoopConf.addResource(new Path(hadoopConfPath, "hdfs-site.xml"));
        }
        return hadoopConf;
    }

    public void close() throws IOException {
        this._hadoopFS.close();
        super.close();
    }
}

