## ----warning=FALSE, message=FALSE---------------------------------------------
library("xtable")

## ----echo=TRUE, eval=FALSE,  warning=FALSE, message=FALSE---------------------
#  library("devtools")
#  install_github("lahothorn/SiTuR")

## ----Call, echo=FALSE, warning=FALSE, message=FALSE---------------------------
library("tukeytrend")
library("multcomp")

## ----Exam1, echo=TRUE, results='hide', warning=FALSE, message=FALSE-----------
data(litter, package="multcomp")
dl <- litter
dl$dosen <- as.numeric(as.character(dl$dose))
fitw <- lm(weight ~ dosen, data=dl)
ttw <- tukeytrendfit(fitw, dose="dosen", scaling=c("ari", "ord", "arilog"))
exa1<-summary(glht(model=ttw$mmm, linfct=ttw$mlf))

## ----Exam1a, echo=FALSE, results='asis', message=FALSE, warning=FALSE---------
Exa1 <- data.frame(names(exa1$test$tstat),exa1$test$tstat, exa1$test$pvalues)

colnames(Exa1) <-c("Dose metameter", "Test statistics", "$p$-value")
print(xtable(Exa1, digits=4, caption="Adjusted p-values for standard Tukey Trend Test",
       label="tab:exa1"),include.rownames=FALSE, caption.placement = "top",size="\\small", 
      sanitize.text.function=function(x){x})

## ----Exam1c, echo=FALSE, fig.cap='Mean litter weights', fig.align='center',fig.pos="H", fig.width=4.5, fig.height=4.0----
boxplot(weight~dose, data=dl)
points(y=dl$weight, x=dl$dose, pch=16)

## ----Exam2, echo=TRUE, warning=FALSE, message=FALSE---------------------------
ttdf <- tukeytrendfit(fitw, dose="dosen", scaling=c("ari", "ord", "arilog"), ddf="residual")
exa1a <- summary(glht(model=ttdf$mmm, linfct=ttdf$mlf, df=min(ttdf$df)))

## ----Exam3, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
fitc <- lm(weight ~ dosen + gesttime+number, data=dl)
covars<-anova(fitc)
ttc<- tukeytrendfit(fitc, dose="dosen", scaling=c("ari", "ord", "arilog"), ddf="residual")
exa3 <- summary(glht(model=ttc$mmm, linfct=ttc$mlf))

## ----Exam3a, echo=FALSE, results='asis', message=FALSE, warning=FALSE---------
Exa3 <- data.frame(names(exa3$test$tstat),exa3$test$tstat, exa3$test$pvalues)
colnames(Exa3) <-c("Dose metameter", "Test statistics", "$p$-value")
print(xtable(Exa3, digits=4, caption="Adjusted p-values for finite Tukey Trend Test- adjusted against covariates",
       label="tab:exa3"),include.rownames=FALSE, caption.placement = "top",size="\\small", 
      sanitize.text.function=function(x){x})

## ----Exam4data, echo=FALSE, results='hide', warning=FALSE, message=FALSE------
liv <-
structure(list(Dose = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62.5, 62.5, 
62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 125, 125, 125, 
125, 125, 125, 125, 125, 125, 125, 250, 250, 250, 250, 250, 250, 
250, 250, 250, 250, 500, 500, 500, 500, 500, 500, 500, 500, 500, 
500, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000
), BodyWt = c(337.6, 319.2, 368.7, 373.3, 314.9, 306, 326.2, 
331.7, 319.7, 304.9, 321, 325.9, 342.9, 322.6, 330.6, 302.2, 
325.2, 326.7, 297.1, 329.8, 350.8, 305.5, 333.3, 348.2, 359.9, 
345.7, 338, 322.2, 351.2, 309.8, 338, 337.3, 338.8, 329.7, 349.2, 
298.4, 332.6, 323.3, 318.3, 330.6, 306.1, 320.3, 301.1, 320.7, 
326.8, 313.4, 324.8, 315.2, 320.1, 308.2, 278, 297.8, 320.9, 
313.4, 291.2, 293.8, 294.4, 280.9, 316.8, 292.2), LiverWt = c(10.73, 
9.89, 12.6, 13.43, 10.21, 10.64, 10.76, 11.34, 10.55, 8.79, 9.27, 
10.62, 11.53, 9.68, 10.53, 10.05, 10.09, 10.84, 8.87, 11.49, 
12.02, 9.42, 12.55, 11.11, 13.67, 10.94, 11.66, 11.03, 11.85, 
10.24, 10.7, 10.59, 9.89, 10.57, 10.84, 9.54, 11.35, 10.58, 11.09, 
9.93, 9.2, 9.52, 9.58, 9.62, 9.89, 8.18, 9.41, 9.05, 8.56, 8.96, 
7.92, 9.25, 9.38, 9.23, 9.02, 9.07, 9.1, 7.9, 9.48, 8.46)), .Names = c("Dose", 
"BodyWt", "LiverWt"), row.names = c(NA, 60L), class = "data.frame")

## ----Exam4, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
#data("liv", package="SiTuR")

liv$relLiv <-liv$LiverWt/liv$BodyWt
mod1<-lm(LiverWt~Dose, data=liv)
mod2<-lm(relLiv~Dose, data=liv)
mod3<-lm(LiverWt~Dose+BodyWt, data=liv)
tt1<- tukeytrendfit(mod1, dose="Dose", scaling=c("ari", "ord", "arilog"), ddf="residual")
tt2<- tukeytrendfit(mod2, dose="Dose", scaling=c("ari", "ord", "arilog"), ddf="residual")
tt3<- tukeytrendfit(mod3, dose="Dose", scaling=c("ari", "ord", "arilog"), ddf="residual")
cttC <- combtt(tt1,tt2, tt3)
Exa4 <- summary(glht(model=cttC$mmm, linfct=cttC$mlf))

## ----Exam5data, echo=FALSE, results='hide', warning=FALSE, message=FALSE------
clin <-
structure(list(Dose = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 62.5, 62.5, 
62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 62.5, 125, 125, 125, 
125, 125, 125, 125, 125, 125, 125, 250, 250, 250, 250, 250, 250, 
250, 250, 250, 250, 500, 500, 500, 500, 500, 500, 500, 500, 500, 
500, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000, 1000
), BUN = c(17.3, 15, 15.7, 16.5, 16.7, 15.2, 16.2, 16.6, 17.9, 
16.7, 18.4, 21.1, 17.1, 18.6, 18.4, 19.1, 22.9, 22.2, 19.8, 20.4, 
20, 18, 17.9, 17.6, 16.7, 16, 17.3, 17.2, 15.6, 15.5, 19.2, 21.1, 
20.7, 19.8, 20.4, 16.9, 16.6, 18.2, 16.8, 16.5, 21.9, 19.2, 22.3, 
18.6, 18.1, 21, 16.3, 20, 15.2, 17.8, 23.4, 21.9, 22.9, 21.2, 
24.3, 17.4, 17.4, 18.7, 20.5, 19), Creat = c(0.6, 0.6, 0.5, 0.5, 
0.5, 0.6, 0.6, 0.5, 0.6, 0.5, 0.5, 0.5, 0.4, 0.5, 0.5, 0.5, 0.6, 
0.5, 0.5, 0.6, 0.5, 0.6, 0.5, 0.5, 0.6, 0.5, 0.5, 0.6, 0.5, 0.5, 
0.5, 0.6, 0.5, 0.5, 0.6, 0.5, 0.5, 0.5, 0.6, 0.5, 0.5, 0.6, 0.6, 
0.6, 0.5, 0.5, 0.5, 0.5, 0.5, 0.6, 0.5, 0.6, 0.5, 0.5, 0.5, 0.5, 
0.5, 0.5, 0.6, 0.5), `TP ` = c(7.4, 6.9, 7.1, 7.1, 7.2, 6.6, 
7, 6.8, 6.8, 6.5, 6.3, 6.2, 6.5, 6.5, 7, 6.4, 6.5, 6.7, 6.5, 
6.4, 7, 7.3, 6.9, 6.8, 7, 7.3, 7, 7, 6.7, 6.3, 6.7, 6.7, 6.2, 
6.6, 6.6, 6.8, 7.1, 6.8, 6.4, 6.6, 6.5, 6.9, 6.3, 6.9, 6.4, 6.6, 
6.4, 7.4, 6.5, 7.4, 6.1, 6.4, 6.4, 6.8, 6.3, 5.7, 6.2, 6.1, 6.1, 
6.7), `ALB ` = c(5, 4.7, 4.9, 4.9, 5, 4.6, 4.8, 4.6, 4.6, 4.5, 
4.4, 4.3, 4.4, 4.3, 4.7, 4.4, 4.6, 4.7, 4.4, 4.5, 4.9, 5, 4.7, 
4.8, 4.8, 5, 4.8, 4.8, 4.6, 4.4, 4.6, 4.7, 4.5, 4.6, 4.6, 4.8, 
5, 4.7, 4.5, 4.5, 4.5, 4.8, 4.4, 4.8, 4.6, 4.7, 4.6, 5.1, 4.6, 
5.1, 4.3, 4.4, 4.5, 4.8, 4.4, 4, 4.3, 4.2, 4.2, 4.7), SerumGlucose = c(137, 
180, 164, 145, 126, 126, 124, 127, 120, 144, 138, 151, 136, 133, 
121, 141, 131, 125, 124, 141, 131, 140, 168, 131, 141, 125, 126, 
143, 137, 196, 125, 175, 188, 144, 157, 139, 150, 144, 158, 137, 
142, 158, 187, 119, 159, 160, 146, 119, 141, 126, 173, 184, 123, 
127, 168, 142, 125, 141, 146, 113), CreatKinase = c(202, 205, 
188, 155, 160, 229, 107, 101, 277, 343, 240, 276, 247, 164, 144, 
1135, 131, 189, 250, 330, 336, 428, 239, 265, 239, 284, 165, 
203, 224, 264, 384, 220, 206, 271, 241, 362, 295, 317, 429, 233, 
369, 366, 462, 409, 237, 419, 268, 344, 297, 420, 403, 336, 330, 
362, 497, 444, 401, 337, 838, 370), ALT = c(100, 56, 70, 54, 
64, 64, 61, 77, 57, 41, 192, 413, 171, 288, 225, 497, 590, 553, 
623, 821, 214, 343, 70, 158, 195, 261, 122, 230, 295, 287, 227, 
182, 215, 279, 468, 233, 284, 268, 192, 105, 189, 247, 333, 162, 
166, 132, 161, 330, 227, 511, 196, 268, 297, 249, 334, 182, 118, 
222, 322, 288), SDH = c(35.1, 18.7, 18.4, 19.6, 21.5, 27.2, 23.8, 
24.9, 16.4, 11, 42, 97.6, 35.6, 58.2, 49.4, 104, 142.8, 137.6, 
146, 198.4, 73.6, 114.4, 14.8, 40.4, 52.8, 83.6, 33.7, 58.7, 
104.4, 75.2, 78.4, 62, 67.2, 84.8, 176.4, 56.6, 100, 103.6, 50, 
26, 80.8, 88.8, 130.8, 43.2, 52.8, 33.2, 53.9, 119.6, 101.2, 
252, 76.8, 140, 122.4, 121.6, 140, 60.9, 29.6, 99.2, 133.6, 106
), ALP = c(159, 150, 155, 161, 167, 187, 155, 204, 170, 155, 
173, 193, 178, 213, 180, 200, 187, 225, 211, 204, 120, 119, 122, 
133, 123, 118, 137, 135, 115, 116, 105, 129, 104, 110, 128, 108, 
122, 114, 105, 101, 116, 111, 101, 109, 119, 122, 108, 104, 116, 
103, 104, 109, 132, 98, 101, 98, 116, 116, 101, 92), Nuc = c(41.1, 
40.7, 39.3, 43.2, 43.3, 50.1, 41.8, 44.7, 44.5, 45, 39.6, 41.8, 
35.7, 34.5, 35.7, 48.5, 41.1, 42.5, 44, 42.4, 38.8, 47.2, 39.1, 
43.8, 39.3, 41.2, 41.7, 43.7, 42.8, 40, 38.5, 42, 37, 40.2, 45.4, 
39.2, 43.7, 42.5, 40.2, 37.1, 39.9, 39.3, 40.2, 40, 38.3, 50.1, 
43.3, 44, 53.8, 55.5, 44.2, 47, 44, 45.2, 53.1, 48.5, 49.1, 55.4, 
51.6, 50.9), TBA = c(20.5, 39.7, 21.9, 20, 14.2, 14.1, 20.5, 
16.5, 12.3, 17, 21.5, 38.2, 29.6, 49.8, 55.4, 38.4, 58.5, 69.4, 
60.2, 83.3, 42.4, 37.9, 34.8, 28.8, 62.9, 51.4, 15.9, 37.5, 32.4, 
55.3, 36.7, 22.8, 53.1, 47.8, 46.7, 22.6, 36, 28.6, 33.2, 25.2, 
61.9, 33.5, 56.4, 67.7, 36.7, 32.9, 25.9, 35.8, 27.8, 73.9, 50, 
44.1, 51, 35.8, 40, 31.3, 21.5, 33.8, 51, 28.6), Cholesterol = c(102, 
100, 106, 95, 96, 80, 96, 88, 95, 93, 126, 127, 103, 112, 122, 
107, 117, 102, 98, 97, 97, 101, 99, 83, 89, 97, 97, 96, 91, 90, 
87, 93, 75, 84, 97, 83, 88, 79, 85, 95, 89, 93, 86, 83, 76, 79, 
73, 91, 74, 90, 83, 86, 84, 84, 81, 64, 75, 70, 75, 86), Triglyceride = c(236, 
102, 144, 130, 125, 93, 235, 67, 145, 108, 116, 172, 90, 154, 
95, 131, 135, 107, 86, 70, 70, 167, 67, 81, 90, 114, 108, 95, 
86, 104, 90, 84, 75, 86, 102, 81, 55, 95, 72, 67, 79, 66, 74, 
100, 69, 51, 37, 98, 86, 101, 49, 49, 85, 80, 29, 55, 59, 32, 
79, 70), Chloride = c(99, 101, 104, 103, 99, 101, 103, 102, 103, 
106, 98, 101, 101, 99, 100, 102, 101, 101, 101, 103, 94, 102, 
104, 101, 100, 102, 94, 103, 101, 103, 97, 102, 103, 102, 104, 
104, 103, 103, 109, 104, 101, 102, 107, 102, 104, 107, 103, 101, 
104, 102, 101, 103, 102, 100, 105, 106, 103, 104, 105, 105), 
    Sodium = c(146, 141, 144, 147, 146, 147, 150, 148, 149, 147, 
    145, 145, 145, 147, 148, 146, 147, 150, 147, 146, 139, 146, 
    147, 147, 146, 149, 138, 148, 147, 144, 144, 145, 145, 147, 
    150, 148, 148, 146, 150, 148, 145, 145, 147, 149, 150, 147, 
    149, 147, 149, 148, 144, 146, 147, 143, 145, 149, 147, 148, 
    144, 150), Calc = c(12.2, 11.4, 11.4, 11.3, 11.7, 11.4, 12.1, 
    11.4, 11.9, 11.7, 11.6, 11.1, 11, 11.5, 11.8, 11.2, 11.3, 
    11.8, 11.6, 11.7, 11.9, 11.6, 11.7, 11.6, 11.4, 11.9, 11.9, 
    12, 11.6, 12.2, 11.5, 11.5, 10.9, 11.3, 11.4, 11.3, 11.6, 
    11.4, 11.9, 11.8, 11.4, 11.4, 11.4, 11.8, 11.2, 11.5, 11.6, 
    12, 11.5, 12.2, 11.1, 11.5, 11.2, 11.6, 11.2, 11.2, 11.3, 
    11.2, 11.1, 11.6), PHOS = c(4.9, 5.6, 3.8, 2.9, 3.8, 5.2, 
    5.3, 5.5, 4.9, 5.6, 5.9, 3.7, 3.9, 5.6, 6.3, 5.6, 5.5, 7.7, 
    5.6, 7.1, 5.7, 3.1, 4.6, 5.4, 3.8, 4.6, 5.5, 5.1, 5.1, 7, 
    4.6, 3.9, 4.3, 3.8, 3.1, 4.1, 3.4, 2.6, 4.8, 6, 6.3, 3.3, 
    4.8, 4.4, 3.7, 5.3, 6.8, 5, 3.9, 4.4, 7.4, 5, 5.3, 5.6, 5.5, 
    4.8, 5.7, 4.9, 7.1, 3.7), Potassium = c(5, 3.7, 3.6, 4.4, 
    4.9, 5.1, 4.9, 5.5, 5.1, 4.9, 5.3, 4.6, 5, 5.3, 5.7, 5.2, 
    5.1, 5.4, 5.2, 4.9, 4.6, 4.9, 4.1, 5.2, 4.9, 5.3, 4.8, 5.6, 
    5.2, 5.1, 4.9, 4.8, 5, 5.1, 4.5, 5, 5.2, 4.8, 4.4, 5.5, 4.9, 
    4.4, 4.4, 5.2, 5, 4.7, 5.2, 5, 5.4, 5.5, 4.9, 4, 5.2, 5.2, 
    5.4, 5, 5.3, 5.5, 4.7, 5.4)), .Names = c("Dose", "BUN", "Creat", 
"TP ", "ALB ", "SerumGlucose", "CreatKinase", "ALT", "SDH", "ALP", 
"Nuc", "TBA", "Cholesterol", "Triglyceride", "Chloride", "Sodium", 
"Calc", "PHOS", "Potassium"), row.names = c(NA, 60L), class = "data.frame")

## ----Exam5, echo=FALSE, fig.cap="Boxplots of creatin kinase", message=FALSE, warning=FALSE, fig.pos="H", fig.height=3.5,fig.width=4.5, fig.align='center'----
#data("clin", package="SiTuR")

library("reshape2")
mclin <- na.omit(melt(clin, id="Dose"))
mclinA <-droplevels(mclin[mclin$variable=="CreatKinase", ])

boxplot(value ~ Dose, data=mclinA, ylab="Creatine Kinase")
points(y=mclinA$value, x=factor(mclinA$Dose), pch=16)


## ----Exam5a, echo=TRUE,results='hide', warning=FALSE, message=FALSE-----------
fitG<-lm(value~Dose, data=mclinA)
library("sandwich")
ttO <- tukeytrendfit(fitG, dose="Dose", scaling=c("ari", "ord", "arilog"), ddf="residual")
exa5a<-summary(glht(model=ttO$mmm, linfct=ttO$mlf))
exa5b<-summary(glht(model=ttO$mmm, linfct=ttO$mlf, vcov = sandwich))

## ----Examb, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
fitG<-lm(value~Dose, data=mclinA)
library("sandwich")
ttG <- tukeytrendfit(fitG, dose="Dose", scaling=c("ari", "ord", "arilog"), 
                     ddf="residual", d0shift=c(0.2,0.5,1, 2) )
exa5c<-summary(glht(model=ttG$mmm, linfct=ttG$mlf, vcov = sandwich))

## ----Exam6, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
ttw6 <- tukeytrendfit(fitw, dose="dosen", scaling=c("ari", "ord", "log", "arilog", "highvslow"))
Exa6 <- summary(glht(model=ttw6$mmm, linfct=ttw6$mlf, alternative="less"))

## ----Exam8, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
ttw8 <- tukeytrendfit(fitc, dose="dosen", ddf="residual",
                      scaling=c("ari", "ord", "log", "arilog", "treat"), 
                      ctype="Williams")
exa8<-summary(glht(ttw8$mmm, ttw8$mlf, alternative="less"))

## ----Exam8a, echo=FALSE, results='asis', message=FALSE, warning=FALSE---------

Exa8 <- data.frame(names(exa8$test$tstat),exa8$test$tstat, exa8$test$pvalues)
colnames(Exa8) <-c("Dose metameter", "Test statistics", "$p$-value")
print(xtable(Exa8, digits=4, caption="Tukey and Williams Trend Test",
       label="tab:exa8"),include.rownames=FALSE, caption.placement = "top",size="\\small", 
      sanitize.text.function=function(x){x})

## ----Exam9, echo=TRUE,results='hide', warning=FALSE, message=FALSE------------
ttw9 <- tukeytrendfit(fitw, dose="dosen", 
                      scaling=c("ari", "ord", "log", "arilog", "treat"), 
                      ctype="Dunnett")
Exa9<-summary(glht(ttw9$mmm, ttw9$mlf, alternative="less"))

## ----Exam10, echo=TRUE,results='hide', warning=FALSE, message=FALSE-----------
tt10 <- tukeytrendfit(fitw, dose="dosen", scaling=c("ari", "ord", "arilog", "treat"), 
                     ctype="UmbrellaWilliams", ddf="residual")
Exa10<-summary(glht(model=tt10$mmm, linfct=tt10$mlf, alternative="less"))

## ----Exam10a, echo=TRUE,results='hide', warning=FALSE, message=FALSE----------
dl$dos500<-dl$dosen; dl$dos50<-dl$dosen
dl$dos500[dl$dosen==500] <-NA
dl$dos50[dl$dosen==500] <-NA
dl$dos50[dl$dos50==50] <-NA
fitall<-lm(weight ~ dosen, data=dl)
fit500 <- lm(weight ~ dos500, data=dl)
fit50 <- lm(weight ~ dos50, data=dl)
tt50<- tukeytrendfit(fit50, dose="dos50", scaling=c("ari"), ddf="residual")
tt500<- tukeytrendfit(fit500, dose="dos500", scaling=c("ari", "ord", "arilog"), ddf="residual")
ttall<- tukeytrendfit(fitall, dose="dosen", scaling=c("ari", "ord", "arilog"), ddf="residual")
combi10 <- combtt(tt50,tt500,ttall)
comptt<- summary(glht(model=combi10$mmm, linfct=combi10$mlf, alternative="less"))

## ----pooledC, echo=TRUE,results='hide', warning=FALSE, message=FALSE----------
library("SimComp")
set.seed(170549)
d1 <- ermvnorm(n=73,mean=c(12.9, 22.0, 23.0, 24.4),sd=c(29.07,32.82, 30.99, 31.77))
d2 <- ermvnorm(n=61,mean=c(12.9, 22.0, 23.0, 24.4),sd=c(29.07,32.82, 30.99, 31.77))
d3 <- ermvnorm(n=75,mean=c(12.9, 22.0, 23.0, 24.4),sd=c(29.07,32.82, 30.99, 31.77))
d4 <- ermvnorm(n=77,mean=c(12.9, 22.0, 23.0, 24.4),sd=c(29.07,32.82, 30.99, 31.77))
dose <- as.numeric(rep(c(0, 1, 2, 4),c(73,61, 75, 77)))
IBDQ<-c(d1[,1], d2[,2], d3[,3], d4[,4])
rut<-data.frame(IBDQ,dose)
lmRU <-glm(IBDQ~dose, data=rut)
Cmat3 <--  c(-3, 1, 1, 1)
EXRU <- tukeytrendfit(lmRU, dose="dose", scaling=c("ari", "ord", "arilog", "treat"), ctype=Cmat3)
EXRRU <- summary(glht(model=EXRU$mmm, linfct=EXRU$mlf))

## ----poolC, echo=FALSE, results='asis', message=FALSE, warning=FALSE----------
Exrru <- data.frame(names(EXRRU$test$tstat),EXRRU$test$tstat, EXRRU$test$pvalues)
colnames(Exrru) <-c("Dose metameter", "Test statistics", "$p$-value")
print(xtable(Exrru, digits=4, caption="Tukey Trend Test and pooled dose vs. placebo test",
       label="tab:exarru"),include.rownames=FALSE, caption.placement = "top",size="\\small", 
      sanitize.text.function=function(x){x})

## ----weitere9a, echo=TRUE,results='asis', warning=FALSE, message=FALSE--------
verrier<-data.frame(
   dose = c(5,10, 20, 40, 60),
   events = c(28, 26,14,12,4),
   n = c(70, 59, 90,88, 77))
lmV <-glm(cbind(events,n-events)~dose, data=verrier, family= binomial(link="logit"))
EXV <- tukeytrendfit(lmV, dose="dose", scaling=c("ari", "ord", "log"))
EXVV <- summary(glht(model=EXV$mmm, linfct=EXV$mlf))

## ----Exam11, echo=TRUE,results='hide', warning=FALSE, message=FALSE-----------
fitLl <-lm(LiverWt~Dose, data=liv)
fitLb <-lm(BodyWt~Dose, data=liv)
ttLl <- tukeytrendfit(fitLl, dose="Dose", scaling=c("ari", "ord", "arilog"))
ttLb <- tukeytrendfit(fitLb, dose="Dose", scaling=c("ari", "ord", "arilog"))
cttL <- combtt(ttLl, ttLb)
EXA11<-summary(glht(model=cttL$mmm, linfct=cttL$mlf))

## ----Exam11a, echo=FALSE, results='asis', message=FALSE, warning=FALSE--------
exa11 <- data.frame(names(EXA11$test$tstat), EXA11$test$tstat, EXA11$test$pvalues)
colnames(exa11) <-c("Dose metameter", "Test statistics", "$p$-value")
print(xtable(exa11, digits=4, caption="Bivariate Tukey Trend Test",
       label="tab:ex11"),include.rownames=FALSE, caption.placement = "top",size="\\small", 
      sanitize.text.function=function(x){x})

## ----Exam15, echo=TRUE,results='hide', warning=FALSE, message=FALSE-----------
# data("liv", package="SiTuR")
livv<-liv
livv[3,2] <- livv[14,2] <- livv[19,2] <- livv[46,2] <- livv[53,2] <- livv[54,2]<- NA
livv[1,3] <- livv[11,3] <- livv[18,3] <- livv[42,3] <- livv[55,3]<- NA #missing MCAR
fitLlv <-lm(LiverWt~Dose, data=livv)
fitLbv <-lm(BodyWt~Dose, data=livv)
ttLlv <- tukeytrendfit(fitLlv, dose="Dose", scaling=c("ari", "ord", "arilog"))
ttLbv <- tukeytrendfit(fitLbv, dose="Dose", scaling=c("ari", "ord", "arilog"))
cttLv <- combtt(ttLlv, ttLbv)
Exa15<-summary(glht(model=cttLv$mmm, linfct=cttLv$mlf))

## ----mulBindata, echo=FALSE, results='hide', warning=FALSE, message=FALSE-----
miceF <-
structure(list(group = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 1,
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
1, 1, 1, 1, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 
2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3), 
    ID = c(230L, 236L, 208L, 231L, 244L, 229L, 234L, 210L, 201L, 
    205L, 228L, 207L, 246L, 209L, 219L, 202L, 217L, 235L, 227L, 
    203L, 212L, 213L, 222L, 232L, 247L, 248L, 204L, 206L, 211L, 
    214L, 215L, 216L, 218L, 220L, 221L, 223L, 224L, 225L, 226L, 
    233L, 237L, 238L, 239L, 240L, 241L, 242L, 243L, 245L, 249L, 
    250L, 251L, 263L, 268L, 279L, 267L, 289L, 280L, 275L, 277L, 
    296L, 276L, 253L, 284L, 272L, 298L, 261L, 285L, 278L, 291L, 
    290L, 264L, 281L, 292L, 295L, 270L, 297L, 256L, 255L, 265L, 
    299L, 252L, 283L, 254L, 266L, 274L, 257L, 258L, 259L, 260L, 
    262L, 269L, 271L, 273L, 282L, 286L, 287L, 288L, 293L, 294L, 
    300L, 344L, 335L, 342L, 314L, 312L, 303L, 308L, 315L, 318L, 
    320L, 313L, 334L, 339L, 302L, 307L, 323L, 310L, 305L, 306L, 
    338L, 331L, 309L, 319L, 321L, 337L, 326L, 348L, 322L, 330L, 
    346L, 332L, 304L, 301L, 311L, 347L, 349L, 316L, 317L, 324L, 
    325L, 327L, 328L, 329L, 333L, 336L, 340L, 341L, 343L, 345L, 
    350L, 396L, 397L, 386L, 354L, 385L, 352L, 356L, 384L, 369L, 
    361L, 381L, 351L, 390L, 377L, 376L, 367L, 353L, 363L, 370L, 
    394L, 366L, 362L, 383L, 365L, 364L, 357L, 392L, 382L, 388L, 
    389L, 393L, 371L, 358L, 398L, 355L, 374L, 379L, 373L, 399L, 
    380L, 391L, 395L, 378L, 360L, 368L, 400L, 359L, 375L, 372L, 
    387L), death = c(178, 564, 579, 610, 630, 632, 652, 663, 
    665, 665, 670, 673, 673, 695, 702, 706, 717, 718, 728, 731, 
    731, 731, 731, 731, 731, 731, 732, 732, 732, 732, 732, 732, 
    732, 732, 732, 732, 732, 732, 732, 732, 732, 732, 732, 732, 
    732, 732, 732, 732, 732, 732, 477, 493, 516, 519, 562, 562, 
    567, 578, 582, 596, 604, 605, 605, 617, 620, 627, 629, 632, 
    633, 649, 652, 658, 661, 664, 675, 675, 687, 719, 720, 720, 
    721, 726, 731, 731, 731, 732, 732, 732, 732, 732, 732, 732, 
    732, 732, 732, 732, 732, 732, 732, 732, 371, 465, 484, 489, 
    520, 521, 545, 582, 590, 597, 611, 619, 628, 633, 637, 641, 
    652, 658, 659, 659, 661, 690, 694, 694, 698, 704, 714, 717, 
    720, 722, 727, 728, 731, 731, 731, 731, 732, 732, 732, 732, 
    732, 732, 732, 732, 732, 732, 732, 732, 732, 732, 379, 450, 
    477, 505, 505, 518, 537, 548, 577, 588, 592, 596, 601, 603, 
    607, 615, 624, 626, 626, 627, 634, 652, 652, 654, 659, 662, 
    666, 676, 676, 676, 678, 679, 680, 681, 686, 686, 686, 687, 
    689, 692, 705, 711, 712, 717, 717, 718, 723, 730, 731, 731
    ), t01 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t02 = c(0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0), t03 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0), t04 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t05 = c(0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0), t06 = c(0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0), t07 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0), t08 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t09 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t10 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t11 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t12 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t13 = c(0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t14 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t15 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0), t16 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t17 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t18 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t19 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t20 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t21 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t22 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t23 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t24 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 1, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    1, 1, 0), t25 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1), t26 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 
    1, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 1, 
    0, 1, 0, 0, 0, 1, 0, 1, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 1, 1, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 1, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 
    0, 1, 0, 0, 1, 0, 0, 0, 0), t27 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0, 1, 1, 
    1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 0, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 1, 1, 0, 1, 
    1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 0, 
    0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 0, 1, 0, 1, 0, 1, 1, 0, 0, 
    1, 1, 1, 1, 1, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 1, 
    0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 0, 1, 0, 0, 0, 0, 0, 0, 1, 
    1, 1, 0), t28 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 
    0, 0, 0, 1, 1, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 
    1, 0, 1, 0, 1, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0), t29 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 1, 
    1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 1, 1, 1, 0, 
    0, 1, 0, 0, 1, 0, 1, 1, 0, 0, 1, 0, 0, 1, 1, 1, 1, 1, 0, 
    1, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 
    1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 
    1, 1, 1, 1, 1, 0, 0, 1, 1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 
    0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 1, 1, 1, 1, 1, 1, 1, 
    1, 0, 0, 1, 1, 1, 1, 1, 0, 1, 0, 1, 1, 1, 0, 1, 1, 1, 1, 
    1, 1, 0, 1, 1, 1, 1, 0, 1), t30 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t31 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t32 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t33 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t34 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t35 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t36 = c(0, 0, 0, 0, 0, 1, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t37 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t38 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t39 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t40 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t41 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 1, 0, 1), t42 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0, 1, 1, 0, 1, 1, 1, 
    1, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 1, 0, 1, 0, 0, 
    0, 1, 1, 0, 1, 0, 1, 1, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 
    1, 1, 0, 0, 0, 0, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0, 0, 1, 
    1, 1, 1, 0, 0, 1, 1, 1, 0, 0, 1, 0, 1, 1, 0, 1, 1, 1, 1, 
    0, 0, 1), t43 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t44 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t45 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t46 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t47 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0), t48 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t49 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t50 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t51 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t52 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t53 = c(0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t54 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t55 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t56 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t57 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t58 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t59 = c(0, 
    1, 1, 0, 0, 1, 0, 1, 0, 0, 1, 0, 1, 0, 1, 1, 1, 0, 0, 0, 
    0, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 1, 0, 0, 1, 0, 1, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 
    0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t60 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t61 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t62 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t63 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t64 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t65 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t66 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t67 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t68 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t69 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t70 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t71 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t72 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t73 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t74 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t75 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t76 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t77 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t78 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t79 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t80 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t81 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t82 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t83 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t84 = c(0, 0, 0, 0, 1, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t85 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t86 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0), t87 = c(0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0), t88 = c(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0), t89 = c(0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
    0, 0, 0, 0, 0, 0, 0, 0, 0)), .Names = c("group", "ID", "death", 
"t01", "t02", "t03", "t04", "t05", "t06", "t07", "t08", "t09", 
"t10", "t11", "t12", "t13", "t14", "t15", "t16", "t17", "t18", 
"t19", "t20", "t21", "t22", "t23", "t24", "t25", "t26", "t27", 
"t28", "t29", "t30", "t31", "t32", "t33", "t34", "t35", "t36", 
"t37", "t38", "t39", "t40", "t41", "t42", "t43", "t44", "t45", 
"t46", "t47", "t48", "t49", "t50", "t51", "t52", "t53", "t54", 
"t55", "t56", "t57", "t58", "t59", "t60", "t61", "t62", "t63", 
"t64", "t65", "t66", "t67", "t68", "t69", "t70", "t71", "t72", 
"t73", "t74", "t75", "t76", "t77", "t78", "t79", "t80", "t81", 
"t82", "t83", "t84", "t85", "t86", "t87", "t88", "t89"), row.names = c(NA, 
200L), class = "data.frame")

## ----mulBin, echo=TRUE,results='asis', warning=FALSE, message=FALSE-----------
# data("miceF", package="SiTuR")

miceF$dose <- miceF$group
      miceF$dose[miceF$dose==0] <-0
      miceF$dose[miceF$dose==1] <-37
      miceF$dose[miceF$dose==2] <-75
      miceF$dose[miceF$dose==3] <-150
cst <- colSums(miceF[, 4:92])
tt5 <-as.matrix(miceF[, names(cst[cst>5])])

## ----mulBinFig, echo=FALSE, fig.cap='Presence/absence of 10 tumor classes in 200 mice', fig.pos="H", fig.width=6.5, fig.height=4.5----
tt5o <- cbind(tt5, "dose"=miceF$group+2)
ys <- (1:ncol(tt5o))/ncol(tt5o)
xs <- (1:nrow(tt5o))/nrow(tt5o)
par(bty="o", mar=c(2,4,4,1))
image(z=tt5o,y=ys, x=xs,
      col=c("white", "black", "lightgrey", "lightblue", "blue", "darkblue"),
      xaxt="n", yaxt="n", bty="o",
      ylab="tumor", xlab="")
axis(side=2, at=ys, labels=colnames(tt5o),las=2)
axis(side=3, at=tapply(xs, miceF$dose, mean), labels=unique(miceF$dose),las=1, adj=0)
mtext(side=3, line=3, "dose group")

## ----mulBinMod, echo=TRUE, results='asis', warning=FALSE, message=FALSE-------
N24i <- glm(t24 ~ dose, data=miceF, family=binomial())
N26i <- glm(t26 ~ dose, data=miceF, family=binomial())
N27i <- glm(t27 ~ dose, data=miceF, family=binomial())
N28i <- glm(t28 ~ dose, data=miceF, family=binomial())
N29i <- glm(t29 ~ dose, data=miceF, family=binomial())
N36i <- glm(t36 ~ dose, data=miceF, family=binomial())
N41i <- glm(t41 ~ dose, data=miceF, family=binomial())
N42i <- glm(t42 ~ dose, data=miceF, family=binomial())
N59i <- glm(t59 ~ dose, data=miceF, family=binomial())
N71i <- glm(t71 ~ dose, data=miceF, family=binomial())

tu24i <- tukeytrendfit(N24i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu26i <- tukeytrendfit(N26i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu27i <- tukeytrendfit(N27i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu28i <- tukeytrendfit(N28i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu29i <- tukeytrendfit(N29i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu36i <- tukeytrendfit(N36i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu41i <- tukeytrendfit(N41i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu42i <- tukeytrendfit(N42i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu59i <- tukeytrendfit(N59i, dose="dose", scaling=c("ari", "ord", "arilog"))
tu71i <- tukeytrendfit(N71i, dose="dose", scaling=c("ari", "ord", "arilog"))

tt10 <- combtt(tu24i, tu26i, tu27i, tu28i, tu29i, tu36i, tu41i, tu42i, tu59i, tu71i)
stt10 <- summary(asglht(tt10))

## ----mulBin2, echo=FALSE, results='asis', warning=FALSE, message=FALSE--------
Tcar <- data.frame(names(stt10$test$tstat),stt10$test$tstat,stt10$test$pvalues)
colnames(Tcar)<-c("Model","Test stats", "p-value")
print(xtable(Tcar, digits=c(2,2,2,5), caption="Tukey trend test for bivariate binary", 
             label="tab:carc"), include.rownames=FALSE)

## ----Exam12, echo=TRUE,results='hide', warning=FALSE, message=FALSE-----------
fitw <- lm(weight ~ dosen + gesttime, data=dl) #lm-model
ttw <- tukeytrendfit(fitw, dose="dosen", scaling=c("ari", "ord", "arilog"))
fitqp <- glm(number~dosen + gesttime, data=dl, family=quasipoisson) #  glm-model
ttqp <- tukeytrendfit(fitqp, dose="dosen", scaling=c("ari", "ord", "arilog"))
cttqw <- combtt(ttqp, ttw) # combine both models
Exa12<-summary(glht(model=cttqw$mmm, linfct=cttqw$mlf))

## ----pupEGData,echo=FALSE, warning=FALSE, message=FALSE-----------------------
wuleon <-
structure(list(Litter = c(60L, 60L, 60L, 60L, 60L, 60L, 60L, 
61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 61L, 
61L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 68L, 
68L, 69L, 69L, 69L, 69L, 69L, 69L, 69L, 69L, 69L, 69L, 76L, 76L, 
76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 76L, 
76L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 77L, 
77L, 77L, 84L, 84L, 84L, 84L, 84L, 84L, 84L, 84L, 84L, 84L, 85L, 
85L, 85L, 85L, 85L, 85L, 85L, 85L, 85L, 85L, 85L, 85L, 93L, 93L, 
93L, 93L, 93L, 93L, 93L, 93L, 93L, 93L, 93L, 100L, 100L, 100L, 
100L, 100L, 100L, 100L, 100L, 101L, 101L, 101L, 101L, 101L, 101L, 
106L, 106L, 106L, 106L, 106L, 106L, 106L, 106L, 106L, 106L, 106L, 
106L, 106L, 106L, 106L, 107L, 107L, 107L, 107L, 107L, 107L, 107L, 
107L, 107L, 107L, 107L, 107L, 114L, 114L, 114L, 114L, 114L, 114L, 
114L, 114L, 114L, 114L, 114L, 114L, 115L, 115L, 115L, 115L, 115L, 
115L, 115L, 115L, 115L, 115L, 115L, 115L, 115L, 122L, 122L, 122L, 
122L, 122L, 122L, 122L, 122L, 122L, 122L, 123L, 123L, 123L, 123L, 
123L, 123L, 123L, 123L, 123L, 123L, 130L, 130L, 130L, 130L, 130L, 
130L, 130L, 130L, 130L, 130L, 130L, 131L, 131L, 131L, 131L, 131L, 
131L, 131L, 131L, 131L, 131L, 131L, 131L, 138L, 138L, 138L, 138L, 
138L, 138L, 138L, 138L, 138L, 138L, 138L, 138L, 138L, 139L, 139L, 
139L, 139L, 139L, 139L, 139L, 139L, 139L, 139L, 139L, 139L, 139L, 
139L, 146L, 146L, 146L, 146L, 146L, 146L, 146L, 146L, 146L, 146L, 
146L, 146L, 146L, 147L, 147L, 147L, 147L, 147L, 147L, 147L, 147L, 
147L, 147L, 147L, 147L, 147L, 153L, 153L, 153L, 153L, 153L, 153L, 
153L, 153L, 153L, 153L, 153L, 153L, 153L, 153L, 154L, 154L, 154L, 
154L, 154L, 154L, 154L, 154L, 154L, 154L, 154L, 154L, 154L, 154L, 
54L, 54L, 54L, 54L, 54L, 54L, 54L, 54L, 54L, 54L, 59L, 59L, 59L, 
59L, 59L, 59L, 59L, 59L, 59L, 59L, 59L, 59L, 59L, 59L, 62L, 62L, 
62L, 62L, 62L, 62L, 62L, 62L, 62L, 62L, 62L, 67L, 67L, 67L, 67L, 
67L, 67L, 67L, 67L, 67L, 67L, 67L, 67L, 70L, 70L, 70L, 70L, 70L, 
70L, 70L, 70L, 70L, 70L, 70L, 70L, 75L, 75L, 75L, 75L, 75L, 75L, 
75L, 75L, 75L, 75L, 75L, 75L, 75L, 78L, 78L, 78L, 78L, 78L, 78L, 
78L, 78L, 78L, 78L, 78L, 78L, 78L, 78L, 78L, 86L, 86L, 86L, 86L, 
86L, 86L, 86L, 86L, 86L, 86L, 86L, 91L, 91L, 91L, 91L, 91L, 91L, 
91L, 91L, 94L, 94L, 94L, 94L, 94L, 94L, 94L, 94L, 94L, 94L, 94L, 
99L, 99L, 99L, 99L, 99L, 99L, 99L, 99L, 99L, 99L, 102L, 102L, 
102L, 102L, 102L, 102L, 102L, 102L, 102L, 102L, 102L, 102L, 102L, 
102L, 108L, 108L, 108L, 108L, 108L, 108L, 108L, 108L, 108L, 108L, 
116L, 116L, 116L, 116L, 116L, 116L, 116L, 116L, 116L, 116L, 116L, 
116L, 116L, 116L, 121L, 121L, 121L, 121L, 121L, 121L, 121L, 121L, 
121L, 121L, 121L, 121L, 124L, 124L, 124L, 124L, 124L, 124L, 124L, 
124L, 124L, 124L, 129L, 129L, 129L, 129L, 129L, 129L, 129L, 129L, 
129L, 129L, 129L, 132L, 132L, 132L, 132L, 132L, 132L, 132L, 132L, 
132L, 132L, 132L, 132L, 132L, 137L, 137L, 137L, 137L, 137L, 137L, 
137L, 137L, 137L, 137L, 137L, 137L, 137L, 137L, 137L, 140L, 140L, 
140L, 140L, 140L, 140L, 140L, 140L, 140L, 140L, 140L, 140L, 140L, 
140L, 140L, 145L, 145L, 145L, 145L, 145L, 145L, 145L, 148L, 148L, 
148L, 148L, 148L, 148L, 148L, 148L, 148L, 148L, 148L, 148L, 152L, 
152L, 152L, 152L, 152L, 152L, 152L, 155L, 155L, 155L, 155L, 155L, 
155L, 155L, 155L, 155L, 55L, 55L, 55L, 55L, 55L, 55L, 55L, 55L, 
55L, 55L, 58L, 58L, 58L, 58L, 58L, 58L, 58L, 58L, 58L, 58L, 58L, 
63L, 63L, 63L, 63L, 63L, 63L, 63L, 63L, 63L, 63L, 63L, 63L, 66L, 
66L, 66L, 66L, 66L, 66L, 66L, 66L, 66L, 66L, 66L, 66L, 66L, 71L, 
71L, 71L, 71L, 71L, 71L, 71L, 71L, 71L, 74L, 74L, 74L, 74L, 74L, 
74L, 74L, 74L, 74L, 74L, 74L, 74L, 74L, 79L, 82L, 82L, 82L, 82L, 
82L, 82L, 82L, 82L, 82L, 82L, 82L, 87L, 87L, 87L, 87L, 87L, 87L, 
87L, 87L, 87L, 87L, 87L, 90L, 90L, 90L, 90L, 90L, 90L, 90L, 90L, 
90L, 90L, 90L, 90L, 90L, 90L, 95L, 95L, 95L, 95L, 95L, 95L, 95L, 
95L, 95L, 95L, 95L, 95L, 95L, 98L, 98L, 98L, 98L, 98L, 98L, 98L, 
98L, 98L, 98L, 98L, 98L, 98L, 98L, 98L, 103L, 103L, 103L, 103L, 
103L, 103L, 103L, 103L, 103L, 103L, 103L, 105L, 105L, 105L, 105L, 
105L, 105L, 105L, 105L, 105L, 105L, 105L, 105L, 112L, 112L, 112L, 
112L, 112L, 112L, 112L, 112L, 112L, 112L, 112L, 112L, 112L, 117L, 
117L, 117L, 117L, 117L, 117L, 117L, 117L, 117L, 117L, 117L, 117L, 
117L, 125L, 125L, 125L, 125L, 125L, 125L, 125L, 128L, 128L, 128L, 
128L, 128L, 128L, 128L, 128L, 136L, 136L, 136L, 141L, 141L, 141L, 
141L, 141L, 141L, 141L, 144L, 144L, 144L, 144L, 144L, 144L, 144L, 
144L, 144L, 144L, 149L, 149L, 149L, 149L, 149L, 149L, 149L, 149L, 
149L, 149L, 149L, 149L, 56L, 56L, 56L, 56L, 56L, 56L, 56L, 57L, 
57L, 57L, 57L, 57L, 57L, 57L, 57L, 57L, 57L, 64L, 64L, 64L, 64L, 
64L, 64L, 64L, 64L, 64L, 64L, 64L, 64L, 65L, 65L, 65L, 65L, 65L, 
65L, 65L, 65L, 65L, 65L, 65L, 72L, 72L, 72L, 72L, 72L, 72L, 72L, 
72L, 72L, 72L, 73L, 73L, 73L, 73L, 73L, 73L, 73L, 73L, 73L, 73L, 
80L, 80L, 80L, 80L, 81L, 81L, 81L, 81L, 81L, 81L, 81L, 89L, 89L, 
89L, 89L, 96L, 96L, 96L, 96L, 96L, 96L, 96L, 96L, 96L, 96L, 96L, 
96L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 97L, 
104L, 104L, 104L, 104L, 104L, 104L, 104L, 104L, 104L, 104L, 104L, 
104L, 104L, 104L, 110L, 110L, 110L, 110L, 110L, 110L, 110L, 110L, 
110L, 110L, 110L, 119L, 119L, 119L, 119L, 119L, 119L, 119L, 119L, 
119L, 119L, 119L, 119L, 126L, 126L, 126L, 126L, 126L, 126L, 126L, 
126L, 126L, 126L, 126L, 126L, 127L, 127L, 127L, 127L, 127L, 127L, 
127L, 127L, 127L, 127L, 127L, 127L, 134L, 134L, 134L, 134L, 134L, 
134L, 134L, 134L, 134L, 134L, 135L, 135L, 135L, 135L, 135L, 135L, 
135L, 135L, 135L, 142L, 142L, 142L, 142L, 142L, 142L, 142L, 142L, 
142L, 143L, 143L, 143L, 143L, 143L, 143L, 143L, 150L, 150L, 150L, 
150L, 150L, 150L, 150L, 150L, 150L, 150L, 150L, 150L, 151L, 151L, 
151L, 151L, 151L, 151L, 151L, 156L, 156L, 156L, 156L, 156L, 156L, 
156L, 156L, 156L, 156L, 156L, 156L), Dose = c(0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 750L, 
750L, 750L, 750L, 750L, 750L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 
1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 1500L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 
3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L, 3000L
), Weight = c(0.903, 0.828, 0.953, 0.954, 1.07, 1.065, 0.959, 
1.025, 1.006, 1.069, 0.825, 1.013, 0.994, 1.059, 1.041, 1.02, 
1.041, 0.911, 0.988, 0.947, 0.976, 1.058, 0.95, 0.86, 1.015, 
0.907, 0.856, 1.075, 0.958, 1.082, 0.916, 1.054, 0.921, 0.874, 
0.999, 0.95, 1.065, 0.944, 0.949, 1.044, 1.025, 0.987, 0.879, 
0.908, 1.071, 1.056, 1.084, 0.851, 0.925, 0.832, 0.979, 1.096, 
0.954, 0.875, 1.057, 1.245, 1.018, 1.123, 1.03, 0.968, 0.852, 
0.94, 0.905, 0.787, 0.918, 0.745, 0.907, 0.818, 0.896, 0.858, 
0.935, 0.669, 0.93, 0.89, 0.983, 0.974, 1.005, 0.897, 1.11, 1.035, 
0.997, 1.059, 1.067, 0.911, 1.122, 1.134, 1.117, 0.999, 1.2, 
1.154, 1.077, 0.98, 0.989, 1.161, 1.101, 1.153, 1.051, 1.082, 
1.062, 0.955, 1.095, 0.994, 1.093, 1.002, 1.038, 1.084, 1.026, 
0.975, 1.11, 1.12, 1.17, 1.13, 1.07, 1.1, 1.02, 1.19, 1.04, 0.821, 
1.07, 0.665, 1.12, 1.044, 0.933, 0.978, 0.981, 0.958, 1.005, 
1.053, 1.059, 0.917, 1.079, 1.077, 1.06, 1.031, 1.006, 1.007, 
0.96, 0.844, 0.833, 0.87, 0.887, 0.835, 0.716, 0.964, 0.933, 
0.753, 0.803, 0.742, 0.885, 0.903, 0.815, 0.937, 0.936, 0.984, 
0.835, 0.954, 0.89, 0.93, 0.924, 0.873, 1.045, 1.087, 1.021, 
0.999, 0.908, 1.027, 1.041, 0.968, 0.971, 1.029, 0.923, 0.944, 
0.953, 1.032, 1.045, 0.967, 1.065, 0.981, 0.964, 1.08, 1.1, 0.994, 
0.949, 1.08, 1.081, 1.012, 0.856, 0.915, 0.952, 0.983, 0.872, 
0.95, 0.939, 1.014, 0.907, 0.841, 0.892, 0.868, 0.837, 0.931, 
0.871, 0.901, 0.987, 0.986, 0.933, 0.935, 1.033, 1.081, 0.773, 
1.054, 0.946, 0.994, 0.724, 0.883, 1.047, 0.951, 0.806, 0.858, 
0.815, 0.914, 0.841, 0.866, 0.792, 0.773, 0.882, 0.887, 0.847, 
1.035, 0.918, 0.886, 0.923, 1.027, 0.991, 0.909, 1.061, 1.054, 
1.018, 1.029, 1.1, 0.932, 0.886, 1.024, 0.93, 1.025, 1.014, 0.954, 
1.081, 0.967, 0.912, 1.03, 0.871, 0.94, 0.996, 1.028, 0.794, 
0.879, 1.058, 1.1, 1.121, 1.093, 1.017, 1.058, 1.09, 1.052, 1.046, 
1.103, 1.026, 1.076, 1.095, 1.037, 0.987, 1.025, 0.967, 0.785, 
0.861, 0.893, 0.941, 0.762, 0.976, 1.019, 0.889, 0.938, 0.961, 
0.999, 0.88, 0.971, 0.924, 0.939, 0.966, 0.981, 0.974, 0.965, 
0.938, 0.903, 1.094, 0.867, 1.007, 0.768, 0.731, 0.716, 0.693, 
0.769, 0.731, 0.791, 0.765, 0.611, 0.745, 0.747, 0.755, 0.6, 
0.796, 0.719, 0.458, 0.599, 0.635, 0.775, 0.66, 0.736, 0.745, 
0.732, 0.653, 0.869, 0.834, 0.892, 0.542, 0.94, 0.814, 0.872, 
0.772, 0.847, 0.946, 0.742, 0.959, 0.931, 0.967, 0.851, 1.061, 
0.866, 0.838, 0.911, 0.885, 1.041, 0.94, 0.951, 0.836, 0.897, 
0.738, 0.793, 0.764, 0.759, 0.872, 0.802, 0.857, 0.875, 0.83, 
0.778, 0.86, 0.871, 0.794, 0.879, 0.674, 0.817, 0.913, 0.846, 
0.931, 0.953, 0.809, 0.918, 0.85, 0.94, 0.871, 0.834, 0.963, 
0.945, 0.966, 0.911, 0.934, 1.018, 0.948, 0.948, 0.947, 0.963, 
0.837, 0.855, 0.815, 0.961, 0.92, 0.963, 0.812, 0.96, 0.982, 
0.966, 0.874, 0.854, 0.9, 0.943, 0.898, 0.895, 0.938, 0.87, 0.878, 
0.849, 0.802, 1.014, 1.043, 0.967, 0.967, 0.936, 0.996, 0.909, 
0.962, 0.714, 0.979, 0.921, 1.01, 1.053, 1.006, 0.961, 1.043, 
0.984, 1.034, 1.031, 0.894, 1.004, 0.97, 0.952, 0.932, 0.954, 
1.03, 0.979, 0.89, 1.1, 0.965, 1.11, 0.965, 1.01, 1.08, 1.02, 
0.812, 0.813, 0.833, 0.739, 0.819, 0.874, 0.8, 0.728, 0.774, 
0.877, 0.564, 0.835, 0.722, 0.778, 0.81, 0.816, 0.838, 0.691, 
0.695, 0.903, 0.731, 0.811, 0.77, 0.791, 0.928, 0.879, 0.906, 
0.841, 0.906, 0.899, 0.888, 0.865, 0.914, 0.859, 0.927, 0.905, 
0.936, 0.93, 0.845, 0.757, 0.868, 0.759, 0.931, 0.92, 0.921, 
0.848, 1.016, 0.988, 0.9, 0.82, 0.949, 0.826, 0.828, 1.003, 0.924, 
0.879, 0.913, 0.835, 0.876, 0.894, 0.841, 0.853, 0.733, 0.86, 
0.814, 0.835, 0.995, 0.864, 0.87, 0.802, 0.875, 0.925, 0.813, 
0.819, 0.961, 0.957, 0.903, 0.895, 0.876, 0.943, 0.882, 0.831, 
0.911, 0.857, 0.912, 0.874, 0.867, 0.815, 0.832, 0.875, 0.876, 
0.82, 0.857, 0.861, 0.858, 0.878, 0.836, 0.828, 0.903, 0.874, 
0.957, 1.074, 0.993, 0.854, 1.043, 0.912, 0.952, 1.008, 1.048, 
0.911, 0.998, 0.851, 0.868, 0.842, 0.744, 1.037, 1.027, 1.015, 
1.033, 1.045, 1.026, 1.116, 0.794, 1.038, 1.105, 0.939, 0.809, 
0.914, 0.772, 0.837, 0.96, 0.913, 0.883, 0.892, 0.831, 0.673, 
0.75, 0.68, 0.501, 0.712, 0.662, 0.631, 0.647, 0.708, 0.657, 
0.67, 0.716, 0.749, 0.672, 0.676, 0.682, 0.735, 0.717, 0.727, 
0.679, 0.674, 0.673, 0.644, 0.605, 0.627, 0.457, 0.656, 0.665, 
0.748, 0.755, 0.743, 0.744, 0.645, 0.574, 0.619, 0.607, 0.637, 
0.573, 0.6, 0.623, 0.554, 0.658, 0.624, 0.582, 0.526, 0.467, 
0.722, 0.419, 0.704, 0.718, 0.695, 0.867, 0.8, 0.827, 0.756, 
0.677, 0.749, 0.686, 0.788, 0.719, 0.652, 0.771, 0.74, 0.57, 
0.778, 0.809, 0.658, 0.758, 1.154, 0.76, 0.786, 0.828, 0.791, 
0.644, 0.815, 0.73, 0.872, 0.943, 0.833, 0.809, 0.925, 0.866, 
0.789, 0.857, 0.904, 0.793, 0.8, 0.822, 0.789, 0.711, 0.703, 
0.654, 0.707, 0.674, 0.629, 0.772, 0.784, 0.861, 0.7, 0.763, 
0.834, 0.645, 0.716, 0.712, 0.714, 0.942, 0.871, 0.973, 0.893, 
0.788, 0.835, 0.917, 0.821, 0.87, 0.816, 0.848, 0.951, 0.874, 
0.881, 0.804, 0.795, 0.733, 0.656, 0.85, 0.737, 0.851, 0.883, 
0.814, 0.715, 0.728, 0.831, 0.792, 0.711, 0.895, 0.931, 0.916, 
0.848, 0.654, 0.613, 0.842, 0.669, 0.831, 0.788, 0.835, 0.877, 
0.954, 0.886, 0.895, 0.84, 0.723, 0.797, 0.836, 0.726, 0.727, 
0.837, 0.715, 0.81, 0.812, 0.782, 0.757, 0.749, 0.779, 0.686, 
0.76, 0.795, 0.785, 0.708, 0.66, 0.644, 0.785, 0.762, 0.744, 
0.709, 0.718, 0.785, 0.82, 0.674, 0.811, 0.845, 0.693, 0.743, 
0.723, 0.705, 0.607, 0.709, 0.715, 0.713, 0.765, 0.723, 0.82, 
0.909, 0.925, 0.884, 0.788, 0.822, 0.816, 0.789, 0.877, 1, 0.817, 
0.808, 0.879, 0.816, 0.939, 0.824, 0.922, 0.947, 0.792, 0.868, 
0.526, 0.816, 0.77, 0.76, 0.771, 0.854, 0.921, 0.84, 0.844, 0.957, 
0.929, 0.809, 0.883, 0.856, 0.755, 0.693, 0.914, 0.905, 0.86, 
0.828, 0.678, 0.765, 0.518, 0.629, 0.763, 0.658, 0.732, 0.55, 
0.633, 0.498, 0.633, 0.619, 0.521, 0.507, 0.696, 0.599, 0.605, 
0.718, 0.615, 0.634, 0.514, 0.716, 0.708, 0.668, 0.693, 0.627, 
0.623, 0.734, 0.599, 0.702, 0.678, 0.733, 0.656, 0.647, 0.719, 
0.644, 0.628, 0.653, 0.597, 0.531, 0.624, 0.827, 0.846, 0.778, 
0.764, 0.838, 0.841, 0.919, 0.852, 0.833, 0.665, 0.665, 0.563, 
0.649, 0.617, 0.617, 0.55, 0.621, 0.607, 0.599, 1.025, 0.49, 
1.004, 0.968, 0.687, 0.593, 0.631, 0.713, 0.747, 0.685, 0.791, 
0.66, 0.64, 0.756, 0.727, 0.619, 0.601, 0.665, 0.572, 0.472, 
0.561, 0.637, 0.627, 0.621, 0.637, 0.596, 0.591, 0.686, 0.685, 
0.464, 0.649, 0.595, 0.523, 0.64, 0.618, 0.566, 0.539, 0.606, 
0.585, 0.684, 0.748, 0.701, 0.484, 0.779, 0.734, 0.777, 0.893, 
0.812, 0.852, 0.77, 0.734, 0.733, 0.751, 0.686, 0.762, 0.798, 
0.804, 0.779, 0.731, 0.83, 0.644, 0.764, 0.75, 0.721, 0.669, 
0.716, 0.717, 0.629, 0.69, 0.755, 0.663, 0.618, 0.717, 0.69, 
0.675, 0.677, 0.727, 0.677, 0.623, 0.618, 0.641, 0.592, 0.768, 
0.816, 0.498, 0.644, 0.743, 0.662, 0.713, 0.411, 0.768, 0.665, 
0.711, 0.719, 0.71, 0.722, 0.603, 0.662, 0.649, 0.614, 0.402, 
0.507, 0.451, 0.517, 0.475, 0.576, 0.651, 0.618, 0.539, 0.705, 
0.687, 0.847, 0.817, 0.842, 0.699, 0.838, 0.801, 0.635, 0.761, 
0.696, 0.573, 0.656, 0.688, 0.689, 0.587, 0.608, 0.643, 0.618, 
1.198, 0.758, 0.929, 0.93, 1.005, 0.917, 0.91, 0.733, 0.731, 
0.81, 0.934, 0.785, 0.799, 0.742, 0.921, 0.957, 0.894, 0.87, 
0.791, 0.883, 0.9, 0.756, 0.776, 0.857, 0.769, 0.89, 0.811, 1.002, 
0.877, 0.79, 0.902, 0.864, 0.856, 0.797, 0.839, 0.867, 0.724, 
0.829), Malformation = c(0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 1L, 0L, 0L, 0L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
1L, 0L, 1L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 0L, 0L, 0L, 1L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 
0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 
1L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 
0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 1L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 
0L, 0L, 0L, 0L, 1L, 1L, 0L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 
0L, 1L, 1L, 1L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 1L, 0L, 
0L, 1L, 1L, 1L, 0L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 0L, 1L, 1L, 0L, 0L, 1L, 0L, 1L, 0L, 0L, 
1L, 0L, 1L, 1L, 0L, 0L, 1L, 1L, 0L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 
0L, 0L, 0L, 1L, 0L, 0L, 1L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 1L, 1L, 1L, 
1L, 0L, 1L, 1L, 1L, 1L, 0L, 0L, 1L, 0L, 1L, 1L, 0L, 0L, 1L, 0L, 
1L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 
0L, 0L, 1L, 0L, 0L, 0L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 1L, 1L, 0L, 1L, 1L, 
1L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 1L, 0L, 1L, 0L, 
1L, 1L, 1L, 1L, 0L, 1L, 1L, 1L, 1L, 1L, 0L, 0L, 0L, 0L, 1L, 0L, 
1L, 0L, 1L, 0L, 1L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 0L, 1L, 1L, 1L, 0L, 0L, 1L, 1L, 0L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
0L, 0L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 
1L, 1L, 1L, 1L, 0L, 1L, 1L, 1L, 0L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 
1L, 1L, 0L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 1L, 1L, 0L, 
0L, 0L, 0L, 1L, 1L, 1L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 0L, 1L, 0L, 
0L, 0L, 0L, 0L, 0L, 1L, 0L, 1L, 0L, 1L, 1L, 1L, 1L, 1L, 1L, 0L, 
1L, 1L, 1L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 1L, 1L, 
1L, 0L, 0L, 1L, 0L, 0L, 0L, 0L, 0L, 1L, 1L, 0L, 0L, 1L, 1L, 0L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 1L, 0L, 0L, 1L, 0L, 0L, 0L, 1L, 1L, 
0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L, 0L)), .Names = c("Litter", 
"Dose", "Weight", "Malformation"), class = "data.frame", row.names = c(NA, 
-1028L))


## ----pupEG0, echo=FALSE,results='asis'----------------------------------------
# data("wuleon", package="SiTuR")

wuleonTab<-xtable(wuleon, caption="Raw data of a fetal weight and malformation", label="tab:wu")
print(wuleonTab, include.rownames=FALSE, table.placement = "H",
      add.to.row=list(pos=list(18:1018, 1019),
      command=c("%", "$\\hdots$ & $\\hdots$ & $\\hdots$ & $\\hdots$ \\\\ \n")))

## ----pupEG, echo=FALSE, fig.cap='Ethylen glycol example', fig.pos="H", fig.width=6.5, fig.height=4.5, dev="png", dpi=300----
wuleon$Mal<- factor(wuleon$Malformation)
wuleon$Litter<- factor(wuleon$Litter)

boxplot(Weight~factor(Dose), data=wuleon)
 points(y=wuleon$Weight, x=factor(wuleon$Dose), col=c("blue", "black")[wuleon$Mal])

## ----pupEGM, echo=TRUE, results='asis'----------------------------------------
wuleon$dose<-as.factor(wuleon$Dose)
fitwu <- lm(Weight~Dose, data=wuleon) 
ttwu <- tukeytrendfit(fitwu, dose="Dose", scaling=c("ari", "ord", "arilog"))
fitwm <- glm(Malformation~Dose, family=binomial(),data=wuleon)
ttwm <- tukeytrendfit(fitwm, dose="Dose", scaling=c("ari", "ord", "arilog"))
cttwum <- combtt(ttwu, ttwm) # combine both models
Exawu<-summary(glht(model=cttwum$mmm, linfct=cttwum$mlf))

## ----pupEGM1, echo=FALSE, results='asis'--------------------------------------

Twu <- data.frame(names(Exawu$test$tstat),Exawu$test$tstat, Exawu$test$pvalues)
colnames(Twu)<-c("Model","Test stats", "p-value")
print(xtable(Twu, digits=c(2,2,2,7), caption="Tukey trend test for joint modeling of normal and binomial endpoint", label="tab:wuL"), include.rownames=FALSE)

## ----Exa16, echo=TRUE, results='asis', warning=FALSE, message=FALSE-----------
adverse <-c(rep("absent",11), rep("present",9), rep("absent",24), rep("present",19),
         rep("absent",21),rep("present",21),rep("absent",21),rep("present",21), 
         rep("absent",17),rep("present",24))
#dose <-c(rep("0", 20), rep("0.125", 43),rep("0.5", 42),rep("0.75", 42), rep(1, 41))
dose <- c(rep(0, 20), rep(0.125, 43),rep(0.5, 42),rep(0.75, 42), rep(1, 41))
webster <- data.frame(adverse=factor(adverse),dose=dose)
t(table(webster))

lmW <-glm(adverse~dose, data=webster, family= binomial(link="logit"))
EX16 <- tukeytrendfit(lmW, dose="dose", scaling=c("ari", "ord", "arilog"))
EXA16 <- summary(glht(model=EX16$mmm, linfct=EX16$mlf))

## ----multRepDATA, echo=FALSE, fig.cap='Multiple overdispersed proportions.', message=FALSE, fig.pos="H", warning=FALSE----
if(requireNamespace("CorrBin")){
data(dehp, package="CorrBin")
Dehp <- dehp[rep(1:nrow(dehp),times=dehp$Freq),]
Dehp$p1<-Dehp$NResp.1/Dehp$ClusterSize
Dehp$p2<-Dehp$NResp.2/Dehp$ClusterSize
Dehp$p3<-Dehp$NResp.3/Dehp$ClusterSize
Dehp$Dose<-as.numeric(paste(Dehp$Trt))

MmN <- glm(cbind(NResp.1, ClusterSize-NResp.1)~Dose, data=Dehp, family= quasibinomial(link="logit"))
DmN <- glm(cbind(NResp.2, ClusterSize-NResp.2)~Dose, data=Dehp, family= quasibinomial(link="logit"))
NmN <- glm(cbind(NResp.3, ClusterSize-NResp.3)~Dose, data=Dehp, family= quasibinomial(link="logit"))
tMmN <- tukeytrendfit(MmN, dose="Dose", scaling=c("ari", "ord", "arilog"))
tDmN <- tukeytrendfit(DmN, dose="Dose", scaling=c("ari", "ord", "arilog"))
tNmN <- tukeytrendfit(NmN, dose="Dose", scaling=c("ari", "ord", "arilog"))
ctMDN <- combtt(tMmN, tDmN, tNmN)

Exa18<-summary(glht(model=ctMDN$mmm, linfct=ctMDN$mlf))
Exa18

}

## ----Exa20data, echo=FALSE,results='hide', warning=FALSE, message=FALSE-------
mn <- structure(list(group = structure(c(1L, 1L, 1L, 1L, 1L, 2L, 2L, 
2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 
5L, 5L), .Label = c("Control", "D188", "D375", "D750", "Positive"
), class = "factor"), animal = c(1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 
11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 22, 23, 24), 
    MN = c(4, 2, 4, 2, 2, 3, 5, 7, 2, 0, 5, 6, 1, 4, 2, 2, 4, 
    1, 1, 0, 26, 28, 22, 37, 29)), .Names = c("group", "animal", 
"MN"), row.names = c(NA, 25L), class = "data.frame")

## ----Exa20, echo=TRUE,results='asis', warning=FALSE, message=FALSE------------
#data("mn", package="SiTuR")

Mn<-droplevels(mn[mn$group != "Positive", ])
Mn$dose<-c(rep(0,5), rep(188, 5), rep(375,5), rep(750,5))
fitMN <- glm(MN ~ dose, data=Mn, family=quasipoisson(link="log"))
Exa20 <- tukeytrendfit(fitMN, dose="dose", scaling=c("ari", "ord", "arilog", "treat"), 
                     ctype="UmbrellaWilliams")
EXA20<-summary(glht(model=Exa20$mmm, linfct=Exa20$mlf))

## ----OrderedDATA, echo=FALSE, results='hide', warning=FALSE, message=FALSE----
hyalin <-
structure(list(dose = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 
4L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 6L, 6L, 6L, 6L, 6L, 7L, 7L, 7L, 
7L, 7L, 7L, 8L, 8L, 8L, 8L, 8L, 8L), .Label = c("0", "5", "10", 
"15", "20", "30", "45", "90"), class = "factor"), droplets = c(1, 
1, 1, 1, 2, 2, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 3, 3, 3, 3, 3, 3, 
3, 4, 3, 3, 3, 3, 4, 4, 3, 3, 3, 3, 4, 4, 3, 3, 3, 4, 4, 4, 3, 
3, 3, 4, 4, 4)), .Names = c("dose", "droplets"), class = "data.frame", row.names = c(NA, 
48L))

## ----Ordered, echo=TRUE, results='asis', warning=FALSE, message=FALSE---------
#data("hyalin", package="SiTuR")

exP<-glm(droplets~dose, data=hyalin, family=quasipoisson(link = "log"))

ntab <- table(hyalin$dose)  # sample sizes needed for unbalanced designs
cmd <- contrMat(n=ntab, type="Dunnett")
cmuw <- contrMat(n=ntab, type="UmbrellaWilliams")
cmduw <- rbind(cmd, cmuw) # combine both matrices
aa<-summary(glht(exP, linfct=mcp(dose=cmduw),alternative="greater"))

DlevelO<-as.numeric(levels(hyalin$dose))
S0O<-log(DlevelO[2])-log(DlevelO[3]/DlevelO[2])*(DlevelO[2]-DlevelO[1])/(DlevelO[3]-DlevelO[2])
hyalin$DoseN<-as.numeric(as.character(hyalin$dose))
hyalin$DoseO<-as.numeric(hyalin$dose)
hyalin$DoseL<-log(hyalin$DoseN)
hyalin$DoseLL<-hyalin$DoseL
hyalin$DoseLL[hyalin$DoseN==DlevelO[1]] <-S0O

# umbrella needed?
lmN <-glm(droplets~DoseN, data=hyalin,family=quasipoisson(link = "log"))
lmO <-glm(droplets~DoseO, data=hyalin, family=quasipoisson(link = "log"))
lmLL <-glm(droplets~DoseLL, data=hyalin, family=quasipoisson(link = "log"))
ordC <- summary(glht(mmm(covariate=lmN, ordinal=lmO,  linlog=lmLL),
             mlf(covariate="DoseN=0", ordinal="DoseO=0", linlog="DoseLL=0")))

## ----Ordered1, echo=FALSE,results='asis', warning=FALSE, message=FALSE--------

Tord <- data.frame(names(ordC$test$tstat),ordC$test$tstat, ordC$test$pvalues)
colnames(Tord)<-c("Model","Test stats", "p-value")
print(xtable(Tord, digits=7, caption="Tukey trend test for ordered categorical data", 
             label="tab:order"), include.rownames=FALSE)

## ----weitere7, echo=TRUE, results='asis', warning=FALSE, message=FALSE--------
if(requireNamespace("HSAUR3")){
data(bp, package="HSAUR3")
bp$Dose<-10^(bp$logdose)
fitbp <-lm(bloodp~Dose+recovtime, data=bp)
ttbp <- tukeytrendfit(fitbp, dose="Dose", scaling=c("ari", "ord", "log"))
EXA31 <- summary(glht(model=ttbp$mmm, linfct=ttbp$mlf))
EXA31}

## ----Exam21a, echo=TRUE,results='hide', warning=FALSE, message=FALSE----------
if(requireNamespace("SASmixed")){
data(AvgDailyGain, package="SASmixed")
AVG<-AvgDailyGain

library("lme4")
mmNN <-lmer(adg~Trt+InitWt+(1|Block), data=AVG)
tt21 <- tukeytrendfit(mmNN, dose="Trt", scaling=c("ari", "ord", "arilog"),
                      ddf="residual", d0shift=c(0.1, 0.5, 1,  5) )
comptt21 <- glht(model=tt21$mmm, linfct=tt21$mlf)
Exa21a<-summary(comptt21)
Exa21a
}

## ----atladat, echo=FALSE,results='asis', warning=FALSE, message=FALSE---------
atla <-
structure(list(dose = c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L), centre = c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 6L, 6L, 6L, 6L, 
6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
6L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
8L), weight = c(10.2, 8.8, 10.4, 9, 8.9, 8.4, 10.7, 9.3, 9.1, 
8.7, 11.6, 9.9, 9.92, 9.5, 9.9, 9.9, 9.7, 10, 10.6, 8.2, 12.6, 
11.4, 11.9, 12.1, 10.5, 12.8, 11.8, 10.3, 12.4, 12.6, 10.6, 13.1, 
15.6, 12.6, 13.1, 13, 12.9, 14.9, 14.4, 12.5, 6.23, 5.99, 6.3, 
5.49, 5.15, 6.03, 6.72, 5.8, 6.12, 5.82, 7.19, 5.83, 6.75, 5.98, 
6.38, 6.36, 6.55, 6.82, 5.56, 7.13, 9.45, 7.53, 8.81, 8.06, 9.83, 
8.98, 7.653, 8.99, 8.82, 8.79, 10.27, 10.67, 10.73, 9.46, 8.94, 
10.9, 10.95, 10.93, 10.36, 9.96, 7.93, 9.14, 7.58, 9.14, 7.33, 
8.05, 7.25, 7.11, 6.86, 7.09, 7.31, 8.09, 7.65, 7.27, 7.14, 7.37, 
8.8, 7.87, 8.25, 7.22, 8.89, 8.46, 8.43, 9.37, 9.68, 10.27, 9.06, 
9.74, 8.37, 9.18, 12.48, 12.16, 11.39, 10.66, 11.98, 13.66, 11.23, 
12.98, 10.72, 11.23, 7.16, 5.9, 5.62, 7.02, 6.84, 6.12, 6.42, 
6.03, 6.29, 7.36, 6.99, 6.45, 7.73, 6.33, 7.15, 6.91, 6.09, 7.35, 
6.92, 10.02, 9.09, 9.15, 7.82, 8.94, 8.45, 6.84, 10.21, 11.62, 
8.51, 11.14, 11.77, 10.11, 10.2, 11.25, 9.92, 10.54, 10.68, 11.464, 
10.39, 6.7, 5.9, 5.7, 6.3, 6.9, 5.4, 5.7, 6.5, 5.9, 5.8, 6.5, 
6.5, 6.6, 5.7, 5.6, 5.8, 7.2, 6.8, 6.8, 6.2, 9, 8.7, 7.7, 8.3, 
8.6, 8, 7.8, 9.3, 8, 8.4, 10.7, 11.6, 8.5, 9.7, 11, 11.8, 11.9, 
11, 10.5, 11.3, 7.09, 7.99, 8.02, 6.75, 7.2, 6.88, 7.16, 8.05, 
7.77, 7.51, 8.11, 7.64, 6.59, 6.87, 8.02, 6.08, 7.3, 8.79, 8.1, 
7.86, 9.99, 9.4, 8.88, 8.22, 9.54, 9.01, 9.69, 8.55, 8.77, 7.77, 
12.55, 12.43, 9.67, 10.66, 10.94, 12.83, 11.33, 7.1, 10.12, 9.34, 
9.52, 8.75, 8.88, 9.15, 10.43, 9.63, 7.81, 8.69, 9.08, 11.23, 
8.19, 9.11, 8.07, 8.65, 8.08, 9.51, 8.59, 11.16, 12.82, 11.69, 
10.4, 11.29, 12.36, 11.91, 12.46, 11.21, 10.22, 16.56, 11.87, 
11.56, 13.17, 11.56, 14.06, 12.22, 11.36, 12.2, 13.2, 6.65, 5.68, 
6.8, 6.61, 5.34, 6.11, 5.87, 6.66, 6.25, 6.4, 6.67, 6.06, 6.71, 
6.83, 5.98, 5.61, 6.12, 6.96, 6.68, 7.43, 8.67, 8.01, 7.71, 7.94, 
7.38, 8.28, 8.68, 8.15, 6.82, 6.65, 8.33, 7.6, 9.08, 9.74, 7.66, 
8.51, 8.41, 9.36, 9.86, 8.75)), .Names = c("dose", "centre", 
"weight"), class = "data.frame", row.names = c(NA, -316L))
atla$Dose[atla$dose==4] <-100
atla$Dose[atla$dose==3] <-30
atla$Dose[atla$dose==2] <-3
atla$Dose[atla$dose==1] <-0
atla$DOSE<-as.factor(atla$Dose)

## ----weitere2a, echo=TRUE,results='asis', warning=FALSE, message=FALSE--------
library("lme4")
mixedA <- lmer(weight ~ Dose+ (1|centre), data=atla)

tt24 <- tukeytrendfit(mixedA, dose="Dose", scaling=c("ari", "ord", "arilog"),
                      ddf="KR")
comptt24 <- glht(model=tt24$mmm, linfct=tt24$mlf)
Exa24<-summary(comptt24)

## ----weitere2p, echo=FALSE,results='asis', warning=FALSE, message=FALSE-------

exa24 <- data.frame(names(Exa24$test$tstat),Exa24$test$tstat, Exa24$test$pvalues)
colnames(exa24)<-c("Comparison","Test stats", "p-value")
print(xtable(exa24, digits=c(2,2,2,8), caption="Multicentre trial: mixed model", 
             label="tab:exa24"), include.rownames=FALSE)

## ----ComBox1DATA, echo=FALSE, results='hide',message=FALSE, warning=FALSE-----
TIComet <-
structure(list(Sample = structure(c(1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L), .Label = c("1", 
"2"), class = "factor"), Treatment = structure(c(1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
1L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
2L, 2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
5L), .Label = c("VC", "Low", "medium", "High", "Superhigh"), class = "factor"), 
    Tail.intensity = c(0.297642070675257, 0.230695029306598, 
    0.68111972385501, 0.118253758653864, 0.639277332464636, 0.950483365472184, 
    0.824093296468855, 0.177447908441866, 0.10513737893206, 0, 
    0.550013983406358, 0.177780704209123, 0.0788848561466468, 
    0.272209074249721, 0.0300605644182657, 0.0161701313418918, 
    0.0370958339765582, 0.15383186393578, 0.0612552697548245, 
    0.0694313120033349, 0.227685272243425, 0.0815373166745593, 
    0.03336713380654, 0.208415899288961, 0.408047408861768, 0.407852342144013, 
    0.515668385561285, 0.374039207845613, 0.223646271789919, 
    0.102920167744512, 2.38163871999669, 0.00092113504892543, 
    0.0969721564861632, 0.16487856942019, 0.0399481548763721, 
    0.606373582207434, 0.00292543076968084, 0.0844076878967038, 
    0.984492416186708, 1.7374152966243, 5.36338404967878, 2.8661677141644, 
    0.00549602545758992, 0.0671594178024692, 1.00675094956888, 
    3.95278512852406, 0.41599924679112, 0.0620435440245579, 1.41796487524067, 
    0.542645707148076, 1.14223325996867, 0.00752843395398998, 
    0.0466438484084312, 0.313696315461328, 0.106241724772982, 
    0.20075422935429, 0.0419637776301275, 0.0440681231693794, 
    1.39739716017932, 0.0102118310507082, 0.241922720387076, 
    0.459821658298476, 0.526454207560739, 1.03242629163098, 17.9683057387103, 
    1.48805741862896, 0.00139176477319647, 0.14875032253223, 
    0.0202135689898585, 0.330483235661447, 0.44678845659507, 
    0.0895720766274668, 0, 0.0563964188274045, 0.00142249129435328, 
    0.0578208254092888, 0.412959322794895, 1.51964602535809, 
    6.51196361731754, 0.253855925447822, 0.061910099931156, 0.265560815699784, 
    0.0410782586469734, 0.14393955737088, 0.103773497201718, 
    3.01400403457486, 0.00427557086242687, 0.40236037840211, 
    0.453696675191289, 0.236723308988146, 0, 0.392092888482257, 
    0.140472535906151, 3.9951049378939, 0.0603877956725921, 0.936432163204474, 
    0.00837036398773988, 0.0614146611724089, 0.0226844019081903, 
    0.701245894178342, 0.20768739201439, 0.256866734486267, 0.970457689402619, 
    0.163653333671198, 0.0150639275425085, 0.350861923593696, 
    0.492849127441156, 0.106270897086349, 0.234076583630949, 
    0.0161805531567532, 0.505132267529108, 0.219951147571205, 
    0.091787130612767, 0.402706203985214, 0.0387151748666272, 
    0.54802153901143, 0.335504335201134, 0.575230161946957, 0.162187618543177, 
    1.65814532669206, 1.79235676910592, 0.0674598195858949, 0.00341689930056071, 
    0.491823678248555, 14.0260735861337, 3.12774153735629, 0.619507597629257, 
    0.31161316332419, 0.0327376432271891, 0.348339812453845, 
    1.62505236473785, 0.414036618571415, 0.381136190276292, 4.1609908280947, 
    0.414938230270289, 0.115418985597399, 0.0120800728916739, 
    0.30889779204032, 0.0552715818026702, 0.157685210447415, 
    0.235924450147453, 6.05017015466754, 0.13823425530101, 2.70449438435379, 
    0.00991432654372902, 0.250023347527538, 0.33160813901869, 
    0.040649215580002, 0.6979035353352, 6.77590762966261, 1.89509813009695, 
    0.0300948106916028, 0.0434731000919909, 0.402487821510172, 
    0.325511406385304, 0.355074754149198, 0.574822033575698, 
    0.157738013287756, 0.339176476040412, 0.389655960308134, 
    0.36138118387189, 0.986500622665006, 0.679087857971833, 4.72812009858839, 
    0.419353047394554, 1.88898936273652, 0.00285878811887658, 
    0.304809545100692, 0.384747840190782, 0.104809126456242, 
    0.134261368593503, 0.115929217264536, 1.33650438864513, 0.697375061253447, 
    1.06704427409143, 5.31506337762199, 0.485411526704166, 0.200986661794263, 
    0.058870003745107, 2.38508239375542, 2.01799423204365, 0.198608540681335, 
    2.88079705219648, 0.281799130895696, 0.666506737035704, 0.598471694545205, 
    0.446742823569841, 3.9578165557544, 0.773002919817324, 0.368252769760884, 
    2.47941395042783, 0.467605627589614, 0.331038710583788, 0.0925326666751268, 
    13.8010522294284, 1.06517357939704, 28.1760817179839, 0.532883728357746, 
    0.0970062633523795, 3.46938368592108, 0.640453592405121, 
    1.21182231303061, 0.24512978602145, 0.44997690477165, 2.07118192257766, 
    0.763466834535769, 0.0106152255277872, 3.17445912124938, 
    0.945717463579469, 7.33153228681691, 0.0341532230829911, 
    0.685758503605569, 0.00913806770245601, 0.613407815065634, 
    0.555395751691601, 3.3756732913998, 0.317928357269567, 7.58370959810452, 
    0.564255058765483, 0.573412429496335, 0.338130527323234, 
    4.70669272754609, 0.882796121593291, 0.0240445814828435, 
    0.736922677429155, 1.85005978029738, 0.0672884489318826, 
    0.530268209592819, 2.41702702799855, 1.66843313902137, 1.8692364196329, 
    0.00496056352001587, 0.970591287009009, 1.16613283884019, 
    0.586636142509658, 0.50121171427812, 0.965954106974871, 0.562353477536185, 
    0, 26.166877535625, 0.813571307103953, 1.69513997672374, 
    9.86844407716111, 10.3477037155959, 6.95627522114878, 0.676345106705436, 
    1.04656462122972, 0.0248986462714277, 3.07066276099912, 0.637595421643653, 
    1.44932831549076, 15.6383666238759, 1.37950066387233, 0.141209473355004, 
    0.0857542803576458, 0.150686762478201, 0.0611512474186154, 
    0.273212373648482, 0.101663032703638, 9.128803431794, 0.00212079059217293, 
    0.0346183471470155, 0.0369097722919856, 0.0487316888380526, 
    0.000679438602530682, 0.00269998881433205, 10.4340010272214, 
    0.547876527338717, 0.371723975296693, 0.00119296745684192, 
    2.08592712840231, 5.48595874250404, 0, 0.270783561707744, 
    0.470149525160465, 0.107998429113758, 0.144525430230795, 
    0.308008631846843, 0.19998607155831, 0.220880440572953, 0.438876616489807, 
    0.0409556867579316, 12.1016243805556, 0.262302153978297, 
    0.11455028580192, 0.562630871164439, 0.127002219900298, 0.00748987930059507, 
    0, 0.0878729964903801, 3.22724313289042, 0.052425661289301, 
    0.00245794230287687, 0.235416773420471, 0.343620496649319, 
    0.541989436798744, 5.1472718697722, 1.61125741727341, 0.43157462925604, 
    0.00188456325246624, 0.0142810219499307, 0.263102596879542, 
    0.497975914226189, 0.611241000437548, 0.0544820650640756, 
    0.247677460900735, 0.0196093930512824, 0.239749616328109, 
    1.15696131988658, 0.505548437036604, 0.50142929968665, 0.236608651840892, 
    0.355651565752501, 0.00064369615481665, 0.254175477756892, 
    0.141749171673754, 0.0655062276092938, 0.0777526363290462, 
    0.0845091659006124, 0.297122154369059, 0.0853957543116998, 
    0.00561581580124172, 1.57106461359409, 0.382893702648143, 
    0.910302728292015, 0.109164801328269, 0.388072648604607, 
    0.153476651187638, 0.189914394647867, 10.1763091903413, 0.359644956116033, 
    4.93144006337985, 0.269699266265575, 0.17032400638715, 2.36150384298241, 
    1.30183912357542, 0.283152806802048, 0.00627125507523714, 
    0.94247559009096, 1.0077675058664, 9.35632699831464, 0.374631057662184, 
    0.00625779830222101, 0.118140295995142, 0.290150832829797, 
    3.77407078886377, 0.164258663048009, 0.310208855271247, 0.107744187329126, 
    2.23098150534914, 1.63096565049684, 0.0643293663557414, 0.00201762830674868, 
    0.922240620817649, 0.386208346473247, 1.59026982383145, 1.34669619071117, 
    0.000385747776887343, 1.17243905359621, 0.223879648402494, 
    0.100286175191158, 0.0273891243288777, 0.312701545918268, 
    0.0238248109158015, 0.346926728255627, 0.149115037034219, 
    0.000678493946985875, 8.68484706361582, 1.12657781714631, 
    2.93077879309969, 1.09621268854459, 2.52054597873028, 1.46837852549372, 
    0.383727720156828, 0.130648126922651, 0.0147837283609137, 
    0.139781225491494, 4.96213630928657, 0.260105810749075, 0.194604555599978, 
    0.066067276543659, 0.458100638760046, 2.19785084935185, 13.1624465137496, 
    0.004966017031726, 2.31821563879081, 2.76194862528424, 0.0555118945149258, 
    0.174958633424193, 0.0954837392393129, 0.497925591349935, 
    0.0546339307403117, 0.0284520204095826, 0.108217559567214, 
    0.436491520556055, 0.0245676370883317, 1.26837565943045, 
    4.59500982443663, 3.40195887207176, 0.12123093529413, 0.405786190722179, 
    0.00554495624618784, 2.11420932220351, 0.155096903782712, 
    0.413298242302866, 3.40729156964346, 6.06076293138635, 0.558717615637046, 
    0.380251157487214, 0.93776500128736, 1.43615984711806, 0.0229890431882296, 
    0.69225438996772, 0.683367795600653, 0.0530611636964827, 
    9.02540245969412, 1.60315760323768, 0.331393873381236, 0.647929442867359, 
    0.673856610834908, 1.11882632776312, 0.42184368737475, 1.60774638028958, 
    0.298932384341637, 4.4527377572345, 0.156047159844992, 0.0906758035381483, 
    0.444260220595298, 0.361933353091305, 0.00396122401822163, 
    0.532239115999523, 2.65185312958186, 0.138979100599855, 0.255736831719068, 
    0.480277591635532, 0.433758271622341, 0.417151948358416, 
    0.193365068821728, 0.285736341780807, 0.000735992227922073, 
    0.125260281103592, 1.22528840193375, 0.265342741524638, 0, 
    0.174020586963779, 0.437986117632997, 1.66836586787923, 1.03458323611206, 
    0.0753527099616055, 3.13084284104669, 0.0125647378485242, 
    0.165720212854049, 0.00451936548108646, 0.182946623271148, 
    0.512943113675041, 0.500945923492425, 1.42551106145291, 0.0377016576401844, 
    2.43779379838902, 1.88535522688876, 11.0661348889815, 1.11766478067265, 
    0.201002067153042, 0.105900892570657, 0.31896387000207, 0.412458833537474, 
    0.00792585846085391, 0.220342604396929, 0.371306968757356, 
    0.185872842156015, 0.0811753799847529, 7.6564982376161, 0.000628627376407929, 
    0.510381280454812, 9.92630579297246, 1.45143062586503, 1.84554544401418, 
    0.956113064875897, 0.128576842333211, 11.6494235950753, 0.243911981028734, 
    0.0783059553229636, 0.561074626262013, 0.110269802998505, 
    0.161363458681886, 0.0188688291582182, 0.560504975174517, 
    0.0105603423240568, 0.390694672113743, 3.01683918045426, 
    4.13107740491451, 0.010715600632312, 4.80348754917245, 2.40121545316021, 
    0.250725429677578, 0.666546442266426, 0.0415733191676098, 
    17.9236932862827, 1.56682601172658, 0.274248755801012, 0.00320470878544208, 
    0.797924394471524, 0.0465613899944366, 0.0625099017061931, 
    0.105375803968245, 0.383547676022303, 0.0154457406301586, 
    0.354173235529168, 0.225175513807652, 0.139991874384682, 
    0.961633258274542, 0.186843463179916, 0.548126352042766, 
    0.303686247747412, 0.335957977758168, 1.6876030960538, 0.0267579858030169, 
    0.111828092476865, 0.00895698740304538, 0.373096098647428, 
    0.100989222981303, 0.0141926773189909, 0.0944376822400371, 
    2.5232235799566, 0.285128046705894, 0.438187322377363, 0.27142731761547, 
    0.0143894165841024, 0.330978552069624, 0.188330776837473, 
    0.0484984144749114, 0.777158955290526, 0.0730107644707212, 
    0.0276294568562818, 0.176034899598631, 0.0678889894651696, 
    0.236332559759808, 0.00880865343067292, 0.0222212852563482, 
    0.914038538753007, 2.3929619483486, 0.325073912981215, 0.735555589080243, 
    0.0358820218782027, 0.256518582418611, 0.148169355983301, 
    2.41166774170429, 0.290155886968481, 0.064675679961938, 0.0412287771544696, 
    0.404494424143706, 1.87686444453256, 0.790075109211889, 0.31476302063648, 
    0.202814481314671, 9.07891704740788, 0.405558176974964, 6.02171129585146, 
    0.7470578004522, 0.378280356749688, 6.29190865443467, 0.164223080789721, 
    1.94901006490271, 0.552464687014012, 0.88380067369599, 2.29673116873359, 
    2.72944998351427, 0.0102364298709577, 0.495968395663006, 
    0.872908557852702, 0.638368136436815, 2.7753482419879, 0.00860793248450572, 
    0.293216096195586, 0.297785305583735, 1.3985706977774, 1.44038613459857, 
    0.553127060489878, 2.28723137246153, 0.26276501122826, 4.20908601083288, 
    0.385449845358445, 0.284793858596936, 0.615487371867192, 
    1.34890577648642, 1.1033999215677, 4.98491196203088, 1.05096175105019, 
    2.2275659751905, 1.97484211996677, 1.7956481537622, 0.31212874675947, 
    0.84349494358274, 0.590867853557566, 7.13162428830277, 8.06810338216121, 
    0.716934852986223, 0.210130966889995, 1.96529300943089, 0.210107569572336, 
    0.0142910461739539, 0.0771659806005252, 0.48099693072291, 
    0.212590302151102, 0.0342763730324044, 10.1491606299715, 
    0.598955405419218, 0.284803776925347, 3.00807681962811, 0.389715951453057, 
    0.201967817217515, 0.755690029020496, 0.468688993794913, 
    0.535218102098702, 0.170516669848487, 7.50403071385953, 0.00414764161491131, 
    0.337165296413615, 54.9166663142448, 0.00553242475111992, 
    0.0432597371936849, 0.282885698682323, 0.996530530036254, 
    9.2821221807062, 9.61480013735993, 0.290147975467488, 0.197369134503812, 
    0.122609122118686, 0.740695396669931, 0.132731907073079, 
    0.00430748882269996, 0.281558474189549, 0.0233830192514196, 
    0.141658635031728, 0.00210901943250505, 0.0207049578021457, 
    0.0450324730361647, 1.70699432688715, 1.77627672931489, 0.211322796174937, 
    0.0335758254057079, 0.618603528343, 0.000709606115385502, 
    0.645911018031363, 2.30596881161499, 0.463998244546624, 0.64274566631843, 
    0.758482463442431, 5.24367558021521, 0.21326126511827, 0.262461134862029, 
    0.422040893481617, 0.710284511391953, 0.149158648565529, 
    9.35554423823229, 0.302646900969361, 0.00342708962029751, 
    0.0101885055463677, 0.868639450342517, 0.812439839010808, 
    0.389319401194108, 1.86770339389608, 0.352319292720854, 0.793283047685307, 
    0.0873859822603751, 2.87781059584759, 0.621345870351243, 
    0.91452514967744, 1.72839816046563, 3.51701623511746, 0.0192654533530087, 
    0.167432015557088, 5.79225279222413, 6.31390630035715, 0.753377751831307, 
    2.26709840541773, 2.38487903514936, 0.276839253085489, 6.81348415092101, 
    0.13272352544647, 0.248042030429093, 1.2342336603389, 0.183422784432457, 
    5.32537161463443, 23.3008654581318, 0.120166427028416, 0.329251689687352, 
    0.1103467321336, 0.369828011296565, 4.94502417442924, 0.297711279972563, 
    1.89666456116187, 2.22170002561677, 0.551327888893601, 0.33117716738091, 
    4.64970685034037, 10.2076919408735, 7.70673237869662, 3.92383826191913, 
    0.45204568665934, 0.950960533062714, 1.86130389019455, 0.784182126825488, 
    0.258392640221335, 0.0241191835780685, 4.3398349510796, 0.0558934736149926, 
    0.231941406619583, 1.13264226917098, 1.55689346497556, 0.40563326762771, 
    2.27775913340978, 1.36051352052616, 0.971096775233406, 1.25947079516023, 
    39.4933327978151, 1.36737180603018, 1.63495816660604, 0.0360105299756618, 
    0.168291062772072, 2.61478242247839, 1.15478110409487, 1.69109129467038, 
    0.0548818056459226, 1.71621072306865, 1.0667435995968, 0.61984595787612, 
    26.0817192526179, 1.04335550188047, 0.785352212879065, 2.77770394194732, 
    2.36601223774984, 0.45011603784619, 0.950957985717339, 2.62378708483808, 
    0.385774472801902, 2.75558542262581, 0.34435261707989, 0.507049174835172, 
    0.681360625376602, 0.793467966930303, 0.961732334925502, 
    0.405285250511776, 3.58889799828661, 17.2014755136181, 19.3004461549732, 
    0.886849697348886, 3.00297274147226, 8.7280845103992, 0.0539937117291356, 
    2.49441261849102, 0.503430067161889, 26.6904897644922, 0.163226840071624, 
    0.883572233754874, 0.114845892836821, 0.487766059533673, 
    0.485208275932772, 0.405633301938427, 0.759969337069515, 
    0.209861720204723, 0.049006466670144, 0.096180652355729, 
    0.485795024756682, 0.0184698479731838, 0.337654942313939, 
    0.590545125031114, 1.9670552385333, 0.343768450735126, 0.11735716503805, 
    0.65314506296197, 0.00536473225018727, 0.274416238563005, 
    1.2442926096592, 0.703460767703103, 3.7395287537012, 2.53309857603895, 
    1.98293876264525, 4.52433139000303, 10.5662535092584, 1.960993653854, 
    0.511058708733127, 0.638981327417268, 0.467759704085706, 
    0.0626381115252288, 1.66955374082323, 0.0333197990615834, 
    0.525377092657053, 0.968834195004733, 0.0462852809089016, 
    3.02225665949319, 0.644531178555621, 0.115892170471073, 2.42621727603232, 
    5.69307116151639, 0.351331519508822, 0.19374727697609, 0.146920983192447, 
    0.565104557336732, 3.09652363871733, 0.535147195597384, 0.943571449839004, 
    1.52766258846035, 0.248411487541866, 3.00320908641557, 1.95211239734837, 
    0.814749881997091, 0.969080925611191, 5.53363841986218, 3.68871404318568, 
    0.744252831213564, 4.21057302844971, 0.121192487236733, 0.0470453819916999, 
    1.51046980646321, 0.496311503147066, 0.819274927547436, 3.06132945095464, 
    0.594054983585323, 0.352097417230023, 3.78109314386619, 0.505287987173162, 
    0.357142218750503, 0.80221715910336, 4.04282974767287, 0.677076156562983, 
    7.41786135710561, 1.22598435767996, 4.37615506398976, 3.28377911661998, 
    1.70698812954453, 0.515847838680312, 12.5141460368225, 0.145057592781541, 
    1.61846032303371, 0.00785581681889035, 0.41403744156307, 
    2.72964581697873, 0.0991986460810989, 0.253205140585829, 
    0.844167459407599, 0.296152884977207, 0.421956857597547, 
    0.305118171275481, 0.605153187780733, 0.245509772847332, 
    1.68781577145395, 0.0947673314515073, 2.47286259000609, 1.75037795198, 
    2.23149118091397, 0.816627416187724, 0.0838914234658781, 
    7.16752222919474, 0.156446659761487, 0.942245610743178, 11.653093877551, 
    0.13487191823875, 0.47633896721139, 0.0245861044290364, 0.12243648607285, 
    0.443438420421076, 1.28237104810108, 0.29347365447097, 2.33215476895537, 
    0.00359381090472807, 0.0746556924036904, 1.98372230122963, 
    0.00861744136338061, 0.341121364609785, 0.198712961392725, 
    0.197713025633978, 0.118293739644291, 0.058129578536771, 
    1.9171353095019, 0.50407341376898, 0.741880839202842, 0.359077828098706, 
    0.2519421358483, 0.357191544377944, 0.600507074875058, 0.493490930878588, 
    0.543943080798536, 0.0574535813048895, 0.197810382377616, 
    0.188270791999738, 0.342092728729512, 0.575153075695306, 
    0.748913633894623, 0.174444894486556, 0.0628566262689647, 
    0.742541220640975, 0.00690084244190623, 0.848535859693079, 
    19.3583205698709, 0.686467560682999, 4.1845806871551, 0.165400303215351, 
    1.11272622717761, 0.957011387462492, 0.340785698272526, 3.40310493827963, 
    1.81243664080455, 1.98346421707966, 0.488365663790763, 3.59980586067045, 
    14.4963886225631, 1.66981659738993, 0.0467075825089445, 18.0173692014404, 
    0, 0.524167461942905, 0.665450472229892, 2.56856102750101, 
    5.50930645468269, 0.676226084652122, 1.52704071954362, 5.92407257233387, 
    0.156136379629847, 0.147728689520396, 0.908785692985584, 
    4.84241058592126, 4.06840290093503, 0.0088260178833383, 0.76434253210683, 
    6.27535740581998, 0.23653908214325, 0.857992917782238, 0.524011033697374, 
    1.10703377780945, 0.295978526578153, 3.94528598053248, 0.273044633151922, 
    23.9684913101894, 7.63057451451306, 1.91500011110958, 1.33996584281874, 
    0.122967879415101, 1.46986503469756, 3.32553585216067, 0.639968410889575, 
    0.528901805285324, 0.184596097322616, 10.985246801741, 1.4695917425513, 
    0.522394498609973, 1.78489618341543, 0.730196723763932, 1.49653041857084, 
    1.0100337931673, 7.1390764001633, 0.896906622888125, 1.18570105068115, 
    0.332201912298213, 3.55368765486254, 0.53429036315589, 0.924251032027589, 
    0.724253783890718, 0.970231041830087, 0.0256996919606366, 
    0.768055661033188, 22.1491497787914, 0.140384081461871, 0.315797272969396, 
    7.41607257156731, 6.94030950908558, 2.23280611061962, 0.39633896565423, 
    0.130758220981393, 0.25616550731444, 0.0978840410269734, 
    0.821000047020135, 0.892190833405001, 9.08706822028223, 0.857739284575853, 
    8.02388781925679, 1.29106875779369, 0.475523927438208, 0.0405708643748317, 
    0.629883848588008, 0.771940094398792, 12.505912150057, 0.0684913085963764, 
    0.752431868823204, 4.35190021369156, 6.88492817863998, 21.0663631776162, 
    1.42674031397008, 1.13418756185017, 0.671245587161454, 1.67558248112772, 
    0.360588138299164, 0.907333814353595, 0.541206767150264, 
    0.925382605936531, 8.97779186049751, 0.226491980383331, 2.15690438862273, 
    5.66838824075768, 4.07134816423704, 0.77126064294156, 0.405681928696042, 
    2.4788376156707, 1.46373535111193, 2.39300101950925, 3.69098611851704, 
    0.00231493023156425, 7.39969176277224, 0.308206314374395, 
    10.0020526775713, 3.16465135881055, 6.85976743148597, 6.54716438101645, 
    8.75255974117611, 0.171720206277565, 0.309125494302314, 5.18696677538242, 
    6.83513038521826, 5.9963528400388, 1.03319377667916, 6.01361316392152, 
    19.3819704246845, 3.16022591455555, 1.01003338884193, 6.11419689592905, 
    0.0131469677614439, 7.48437933522002, 0, 2.51323423124255, 
    0.550087503033864, 1.2751832944686, 11.3722573016813, 0.00642045741173018, 
    0.942972332502385, 2.32476651258069, 0.194890703186726, 2.70136271171611, 
    0.656930208183866, 2.45289391510044, 5.20396036471423, 0.667859582429984, 
    12.0318384510524, 0, 6.85563227696193, 2.04360978824093, 
    2.60178761146583, 3.28523346184444, 1.22075688433316, 0.136768454215657, 
    0.625207101042456, 0.191716791260844, 30.4637016914765, 10.6374172692263, 
    1.08642406613478, 15.0942668278505, 2.06245226376192, 1.13470043717136, 
    0.610895281774178, 0.403194040161843, 1.19538792518336, 10.6083128624822, 
    0.905191212367779, 6.3426025864945, 0.81406232431951, 1.22947670397787, 
    1.074676353707, 8.93639884123065, 0.970721279044938, 2.39955513931164, 
    7.64248033282809, 11.4332028150867, 0.162903738883947, 2.85362287961285, 
    0.898828514300399, 2.33317882225624, 0.289860715015719, 5.44178078592523, 
    0.251383239433061, 4.16570137685676, 2.16662278108714, 0.836890443278464, 
    2.55265495051819, 9.96789131700637, 0.721568500417068, 12.8553913631887, 
    4.61702264326493, 7.3482421069577, 0.639619177825433, 0.584283119245146, 
    0.110602364489365, 1.62430035145136, 0.688888071775261, 2.9831762733422, 
    0.667831810991478, 5.47492433324048, 1.38249443898402, 4.24539073951786, 
    9.95233118155427, 0.22421646291297, 0.422876614879285, 10.6709471043759, 
    0.215285005640605, 0.859219369103128, 14.6155943780845, 0.665625336883947, 
    2.33910207377189, 10.3088229925897, 0, 1.39177978213043, 
    0.0808767873251558, 7.28899340341533, 30.037294921704, 15.2168241544972, 
    4.59985360223818, 1.79933582634981, 0.249873102805245, 0.132920576906661, 
    1.28433284306222, 7.49290448969911, 3.51064403693143, 14.124732776672, 
    4.51858941261906, 10.23585143491, 8.75558112586806, 2.7174389115716, 
    1.09365826526723, 0.252756985872943, 0.0158132198517961, 
    0.305856125278669, 2.65814362682289, 5.67966488592196, 10.5235144200764, 
    0.57237797029477, 9.08354944769305, 0.0667678954113225, 0.331769533155433, 
    18.5348236099472, 12.0287704271966, 6.71698786011187, 2.76550701533345, 
    0.528414708567698, 0.252480037982654, 2.35347885293389, 1.31368320250604, 
    12.3639905008319, 2.07682704622309, 3.50390135944278, 0.783591912765115, 
    0.119732878648477, 5.35781498716269, 7.03709691356132, 1.06193967674574, 
    7.43717196585458, 0.108167937011043, 0.111299714230463, 29.4735188374271, 
    5.54777321525552, 3.62289070931031, 1.75374743394318, 10.6364815911493, 
    13.7434572904122, 0.428918284434058, 0.400856659402961, 1.20497270964606, 
    3.40230940577165, 8.01989091556476, 1.05460812633372, 0.116401935639735, 
    24.3527532475361, 0.110353204316366, 4.84871271704759, 2.01507968713172, 
    5.97697906535302, 23.432165602774, 10.0624080995836, 1.56541051282655, 
    3.52328818491964, 6.56961324041393, 28.1667605534107, 14.6827808206804, 
    0.777586292054737, 3.80317984736574, 1.28727653504674, 1.33066656634941, 
    11.3206246553366, 16.2576491261512, 3.77580361212446, 16.9202235897482, 
    6.64025444273457, 4.56158867124482, 3.22158665887616, 0.940793856529874, 
    3.3338569710627, 3.92707972712805, 8.54268825826104, 5.8188102965367, 
    0.858401373141399, 3.97241668556436, 2.55044996157754, 4.31348566688348, 
    9.48030314004095, 0.0700884761456405, 5.33570616463317, 3.43876218202818, 
    7.34093481138518, 1.40918580375783, 4.71142224915989, 14.111324629219, 
    13.1360746140002, 3.75804989112377, 1.58575484001885, 7.43744170121908, 
    8.80021687738276, 8.27079642850752, 5.87474273668818, 10.9049258231203, 
    15.1052971763185, 4.0910587881326, 2.40582838051793, 7.712284175106, 
    9.47069772484956, 22.2722268704038, 0.564899519212976, 13.2675610483785, 
    3.73395760872176, 13.4174051496102, 8.88233471758864, 7.76326095449236, 
    11.8771260694774, 3.26159524113151, 3.28333279409862, 11.5692511314149, 
    0.126827182694018, 6.45924461021637, 3.84530842319256, 7.9920664092148, 
    9.12488380716307, 25.8921274567867, 2.43378109781362, 8.98928628161606, 
    2.82479832419664, 2.46070945195241, 11.2313645251512, 0.966969677773555, 
    7.24228682561461, 5.43519567048148, 20.9956013083288, 0.0900407943143417, 
    7.12408710913174, 5.19361400597372, 3.60129416654043, 11.2575923817911, 
    0.143271188237143, 1.75793423173097, 0.00130371084997059, 
    21.35845777953, 0.140344544897775, 12.250067676111, 5.92381304911108, 
    9.41619128535016, 6.38687011459713, 0.785673853753924, 0.186619863589766, 
    11.8361645691066, 21.4844296240521, 2.39050351203842, 15.7335593059536, 
    12.6447622389893, 13.1058902877698, 7.63812507508541, 9.80038574318285, 
    6.73377544555546, 9.81106445333609, 0.820485872146923, 15.7432583220611, 
    0.146302736349448, 7.1761980262978, 10.1173215785283, 5.35180601752671, 
    0.0351384418194794, 1.22975212548166, 13.7049961256234, 2.65411068151948, 
    12.1607504159637, 0.00458551702659791, 0.105275853750991, 
    0.82398912674142, 6.08652515497322, 6.81174683768832, 9.69708647581069, 
    12.9300266640047, 1.39098943965802, 13.2367444837172, 13.0402485957888, 
    0.506018756645435, 22.5272763668106, 5.25975581642385, 0.0540831959927252, 
    13.8254640329111, 6.15552482606629, 10.1762635391327, 0.219318712085861, 
    1.19760479041916, 12.0506712151501, 0.884823634547653, 5.70623148268979, 
    1.4032200147825, 0.547589779974095, 26.9420872552645, 6.49679180138689, 
    0.268161492810115, 15.593628984422, 12.6007002813003, 6.12844789281198, 
    0.877065084510044, 4.55504488033933, 0.775870604224984, 5.51668414447558, 
    6.91054750109791, 10.5920274087647, 13.1019240877242, 5.50367974627325, 
    19.5278388628613, 16.8954720142869, 19.0296905978048, 24.5917587124946, 
    7.55084042332434, 7.29097128070308, 0.0311832052782509, 11.1898932190709, 
    0.0659641091666269, 5.3668308996038, 13.0518212782786, 16.8747417665963, 
    7.31034446345224, 6.0060004406638, 7.40703671751847, 11.6248199820728, 
    19.2811763937229, 16.3995982683387, 8.75392816015005, 2.01298334592764, 
    5.93203226618554, 0.0902603276711227, 36.508299974728, 7.56414076476174, 
    0.156415990789257, 11.4204835658123, 3.75600358179795, 21.7354657393382, 
    6.24499618700541, 10.8609417793091, 2.56050761628299, 15.9345114675382, 
    15.2384070110242, 0.0974632945253367, 5.59369444803968, 15.4707167455473, 
    0.0233260257762067, 2.40539304969994, 0.00978180202058006, 
    11.5981022799642, 14.6172525286388, 0.210092600308136, 15.6455825573473, 
    15.9543883625042, 8.99442557160778, 0.0816356822145534, 11.0693864381867, 
    0.477898612613639, 5.49243201115714, 4.29433124268523, 3.05195133544361, 
    12.3318567990465, 13.8719924266581, 13.1310562736367, 0.000756012186916453, 
    9.10341775886047, 12.4880278863928, 2.97262167677547, 7.5044815259905, 
    0.25077311559057, 18.2397279621836, 10.5741311978771, 8.154049366602, 
    13.8844071464222, 6.92724105220913, 8.17323219593871, 9.08373163303293, 
    1.71611026073636, 9.85714109844654, 1.8852912035271, 14.4746839225194, 
    7.18797710775604, 24.7533078390103, 2.73854692194278, 17.5459326730725, 
    13.6346490379909, 0.240856848237605, 0.0952404512170987, 
    7.91597421950217, 0.337853117565648, 0.144360513449408, 0.995081050420894, 
    0.0854762186165094, 1.27178723145005, 6.85048053825965, 3.05024230792433, 
    8.18470424154541, 0.283786945800493, 0.0678328807776555, 
    0.284100220188844, 4.33265177038251, 2.88502404080023, 8.74591366573142, 
    0.680223982986897, 0.358795931310201, 5.77159507836475, 0.405404855912527, 
    0.289874949725972, 0.110523456574272, 0.226049630384482, 
    1.11007281670311, 0.330552104652796, 10.8034234311253, 5.40163888032903, 
    8.82161614120541, 5.46302766777959, 0.314445809366223, 2.64576866039571, 
    8.01007825257633, 21.975756237165, 3.48463099539672, 1.61863899521651, 
    7.69285312957378, 5.16539505351016, 0.103793615793629, 5.72837103077776, 
    5.61812640784418, 1.39896659172485, 7.90868778151733, 5.96921918510681, 
    0.509368403739138, 45.997048868024, 13.5604099360277, 5.80907783819895, 
    1.86921346451679, 10.8988042925934, 2.95924226274068, 0.900269364311843, 
    22.3389530091216, 2.85595073974264, 0.0530765343308674, 8.58205610354657, 
    22.7900498233974, 1.16809346211703, 7.80820040624081, 6.15486850203621, 
    5.19319784651449, 6.91834168989414, 10.1870747824533, 2.84740274791303, 
    11.0042418701105, 11.2269061720949, 6.83081038493858, 0.64593448352225, 
    4.95321665890228, 5.83368201998808, 9.81242518851606, 0.713437263615734, 
    9.11922089214056, 0.519093281976177, 1.07537473510942, 0.978555457476539, 
    0.791689789269565, 1.52431553189449, 8.03814068563787, 1.96475802507701, 
    2.69101667656588, 3.75393786600411, 0.243341816942565, 9.98038741952447, 
    1.35213792802916, 4.12672168489175, 13.9726638530386, 14.4412471655595, 
    1.77222611757109, 1.35701769736528, 0.957620754485851, 4.44746344912405, 
    16.5944249458689, 0.984035752283209, 3.0405749961343, 5.15426795296602, 
    4.42035069050058, 6.72969593461222, 3.62330297023281, 0.121909074911536, 
    8.00983081980682, 2.12465115387335, 9.30722339660081, 0.482493122450196, 
    2.82605376783247, 4.48305044516215, 10.956323573361, 5.54037913821088, 
    7.78381831696245, 6.46464004911729, 12.2687927954119, 15.0873117307706, 
    6.7814188932024, 10.8333710931068, 17.2871537523353, 5.46611300776524, 
    9.98752327138518, 0.920408254399849, 7.01419003379682, 0.067047802426224, 
    21.929333134359, 14.8200289868167, 25.4622413563329, 0.142892839624999, 
    12.4014359968487, 0.915518153832393, 6.56012114026799, 8.24768622678853, 
    11.3052620286435, 11.4889421304829, 9.68876127320614, 12.8736232311497, 
    13.6608758972225, 10.6144857084558, 7.72790374826469, 8.1891196176231, 
    30.4695282481856, 11.7178870193319, 0.844519201945175, 0.31952580195258, 
    2.81527557089202, 8.67900953652977, 0.83946797734113, 8.95916175341839, 
    13.9443485537947, 20.0376101357494, 18.2355071159494, 8.93106883706378, 
    1.32218223608845, 14.0200236825783, 8.86726476405764, 0.849731663685152, 
    10.8726210047641, 0.0990703425298892, 0.249430553510648, 
    6.41786458071886, 6.84584290024896, 12.9091841030992, 20.6509226373577, 
    10.6459334569039, 7.98697014679202, 0.706899358748799, 1.4689842929918, 
    8.1793267420176, 19.8556313447774, 5.75521809773778, 3.10850670032635, 
    1.75305865200035, 10.7620332478595, 3.76975800044926, 17.7341180814348, 
    1.09067112629972, 4.28103097445173, 8.3486595854087, 4.05447616667708, 
    9.24113682295054, 3.54121012395657, 0.560609610583465, 30.454312476174, 
    5.270836046966, 5.35181968550011, 4.65516146690439, 3.60347166206272, 
    16.9111193804545, 8.94215441097234, 0.222170462078727, 0.183258316505111, 
    2.07740528663135, 1.10254418238763, 2.69560996598706, 14.7642019757728, 
    0.0184854621012749, 0.0204917523801481, 9.7631551774235, 
    1.13196203561698, 1.82782150097298, 4.22247162522931, 8.14153211954655, 
    7.74235073555139, 8.81151469857455, 5.42123436384623, 0.266080152695879, 
    9.77230902164747, 5.85188684196714, 10.3201617586423, 0.569750548430726, 
    0.595154396230858, 6.57958940942661, 1.88135843254868, 0.304713517994513, 
    0.399448174249058, 0.517582797531528, 0.014688655584076, 
    2.98984789443327, 2.62245164909555, 3.64879866310833, 13.4554417799688, 
    2.70728696487971, 0.124796760897264, 5.52938176900309, 6.55938007220105, 
    1.50906167787197, 0.674950111410393, 11.0099390377429, 9.30632796809143, 
    8.02949502009436, 4.01752449436667, 15.5195988802906, 10.386904423659, 
    5.43975543013511, 5.67428944085512, 7.07036559685042, 8.71789413644557, 
    12.038914896537, 1.68563901894992, 0.163045944122369, 3.83662191756783, 
    6.86057619789834, 0.562222544799062, 4.0909186713565, 2.62566056024862, 
    0.154535974973931, 0.0924949195662628, 5.82548470177718, 
    6.7123311840157, 0.0635248794393417, 0.479232320864471, 5.12126167760218, 
    7.08909777704471, 0.0369119506308186, 0.158992835483738, 
    1.23182780860382, 6.42036856534138, 2.95823878339106, 0.075537188018928, 
    3.67293328269401, 3.59922772261005, 9.69880834968633, 4.08660367250635, 
    6.13160758481881, 2.18299411061218, 9.28919217539033, 0.888149705644817, 
    8.72779901720774, 6.16184701018781, 1.27897719510644, 0.947542690301092, 
    3.07074581248315, 17.6407817882282, 0.636126759538262, 5.0896930243243, 
    2.83252695914576, 2.05058039346483, 3.82396999531826, 4.01223545869424, 
    6.90963428596646, 11.9025826226047, 9.69620715312343, 0.682148368137943, 
    1.89295402462958, 0.475104591241269, 6.16055626871013, 3.35357832825797, 
    4.059747707873, 0.994350162767228, 1.85865891323977, 0.00212783161191756, 
    9.96758426795504, 3.25701031579137, 7.95122253259172, 1.13341362501158, 
    3.13060861190959, 3.96307003637042, 19.8976319069381, 2.31799907032123, 
    3.31608983366172, 20.7990030690096, 0.126432650024768, 0.00855331184234536, 
    13.3311511344369, 1.34634924738716, 6.55282649020562, 15.14028535992, 
    0.596226626933819, 35.5003738860036, 0.668205816740724, 22.6603032137811, 
    0.547625841625635, 7.24664029473042, 34.4009107667747, 0.41353486460569, 
    4.73134483475369, 26.1368949287785, 13.4050845208074, 0.463461864004393, 
    22.525305653207, 0.0645049096526845, 30.8297978995244, 1.04488159636116, 
    0.387618556850377, 0.682168167318782, 11.9086148212024, 15.9182259182259, 
    0.0371525753780915, 26.0737726175514, 1.04352554627126, 1.23007454809478, 
    19.6469098158263, 18.9995704051248, 23.5313784193354, 12.6551280651882, 
    25.6039574163939, 1.46207486349582, 29.2220558510447, 22.2452115106049, 
    14.1208048962282, 0.0818802991906132, 21.1814282429615, 0.0560563807321175, 
    20.4582031725538, 32.0106363240126, 25.8403719348753, 0.726373988759709, 
    30.2443470442401, 26.3212645767585, 3.07475537420241, 2.20192566845417, 
    39.597420481139, 25.4468123958579, 0.0104553027238635, 45.1128429073982, 
    31.8936546058026, 7.41608297008868, 0.619380499697498, 19.6551612520366, 
    21.0461545837937, 0.163293012632266, 34.1195860734375, 0.239708736141335, 
    10.2117855636374, 0.0848349967183494, 32.6338296844311, 4.46370204566379, 
    0.567330066946108, 0.309368739632444, 17.5519928933952, 29.4786764675872, 
    0.517087656213157, 0.0155060052543206, 0.195136145333681, 
    0.0958774128676835, 1.49799784907987, 0.0447686679686468, 
    0.0868656119797672, 32.0837671602235, 29.0365755960109, 0.471441581991711, 
    31.8782588803256, 20.6039954058791, 26.1911716606003, 0.107806509282681, 
    24.6316108543046, 0.0535481385394613, 3.0325255499401, 15.4517705618299, 
    0.0350224858298325, 4.6728371147206, 29.3959402488235, 0.0218515912759177, 
    20.8652659972259, 41.2151261034781, 5.07972936328944, 0.113844558021385, 
    0.324926177195898, 0.0308790013386125, 0.0175416571362729, 
    20.3504684181794, 0.955158954299058, 0.16648745032938, 23.6711492839015, 
    0.0619308109588624, 23.1383586156046, 6.55876999689943, 5.13153308568739, 
    9.06558773107431, 3.67322111326282, 6.1291271139796, 25.7554780283676, 
    14.2413108908406, 0.00776152997667063, 14.3833278974981, 
    6.48851687743528, 8.02907747357586, 15.551212013521, 10.4249525380175, 
    11.4648685421168, 0.0858786993610504, 8.42897247657952, 11.7625046930672, 
    12.0816919661761, 0.0480106066509463, 8.26495551620254, 17.6661395314626, 
    13.2420290725285, 0.496684485379686, 8.75331862879424, 29.9912556574893, 
    26.5715627685727, 0.0160159313188515, 4.57094404517999, 0.148203960514349, 
    8.8219837626242, 15.1822811889738, 4.19491043528649, 16.9225432077747, 
    7.4306696085425, 12.4071891518851, 22.5449655718404, 4.27411560487962, 
    13.5102692705305, 7.33913027120828, 14.8485504489257, 0.00260617729172571, 
    0.299630386460742, 1.0091516824296, 19.6177757310615, 0.186793141189347, 
    3.58273465480389, 11.3904111255899, 1.91104227580369, 2.15045193485372, 
    1.00857045729036, 22.6139036919679, 11.9428577791139, 5.98256622546234, 
    3.5577770677835, 9.62557404542345, 7.28466148345463, 12.6177242509567, 
    15.878918492538, 8.7089253332802, 14.4365268142494, 18.9295870953706, 
    8.51934959317602, 10.889382676823, 18.7553526504541, 0.071293907382351, 
    7.24744210011321, 8.94254599822203, 13.2249433621764, 12.0029192470462, 
    4.97201144420948, 18.3975153636247, 0.689202431807587, 16.7164317068871, 
    8.17392273853383, 0.981863301385034, 9.07503566939115, 35.2565128369257, 
    14.2706431581142, 2.79972718047677, 0.522639095261681, 2.64036564659146, 
    11.4998934842813, 6.92568729507257, 21.4805945295369, 7.52027402542902, 
    19.3620628108487, 16.1863712682747, 6.97013262525089, 0.0196512491140142, 
    5.34255946213112, 0.0519416189961852, 11.9688676328854, 0.185348417826442, 
    9.22871770275819, 18.567442737616, 6.16029168248379, 0.47872134673258, 
    13.9858622999914, 10.1257587501852, 18.4081361853084, 30.9174428113799, 
    37.6987358667582, 1.51096219020655, 37.0714547403558, 0.0631614406422817, 
    2.57429664895762, 47.4916708102115, 35.5259344505847, 43.8098772925621, 
    54.1885566664204, 46.9598780304328, 0.0590852217433616, 30.5922909744948, 
    1.2595686159509, 33.8032299131728, 0.309401869568771, 42.5062558490093, 
    1.19898890459474, 36.7060100015997, 4.79155256886573, 29.8604181890345, 
    27.3188575854899, 0.291707759592858, 32.1455111336475, 0.745221980608942, 
    0.0924414137533615, 0.49324247813547, 29.8679772301382, 56.7680735085902, 
    51.3234353410777, 24.9163710725962, 2.35006278159391, 34.8289388501785, 
    37.9570961147125, 0.844573017244618, 19.9559461780129, 3.6604201464612, 
    48.3471056896121, 33.0126858475961, 21.9441841536931, 38.8015787759233, 
    27.2492054411789, 41.0567022147628, 28.5134726719077, 43.171360082759, 
    39.7586544364974, 47.3072288686301, 37.2646248022275, 54.0589820833994, 
    1.38078607542237, 5.01662334407379, 36.3040056472445, 0.598953226806704, 
    0.506641175286778, 34.5835914262362, 0.103184927567013, 3.03444033958465, 
    0.311450174229733, 45.0191046579822, 2.68912036485273, 31.0903302860177, 
    39.8622757600145, 27.9549308669462, 45.4367537278351, 0.0844158701835945, 
    1.94727805616604, 35.0176775052472, 0.0127291242362525, 0.0432041164315525, 
    48.1175823560185, 51.3523869651328, 0, 0.604869363642173, 
    0.00429344571615804, 47.4452181393348, 0.0462873323975023, 
    20.1719333785985, 25.8157273331546, 0.141952975918075, 19.8568161618304, 
    14.3665762436722, 0.128568213209493, 35.3401366414604, 4.69402242024292, 
    32.0154288132767, 4.99908395630292, 0.275285289979678, 30.2352189907379, 
    31.4185823762774, 0.402140393686415, 50.3416092732711, 37.068772832202, 
    0.145914215145933, 39.7531284629807, 0.0591493345699861, 
    38.6108231140152, 0.282960740912528, 40.1960954710767, 42.3220824436651, 
    1.17203748161783, 12.6864817984613, 6.50345011438385, 6.42764675200623, 
    11.514851897232, 7.85892201581778, 19.3740570870311, 16.0742050648961, 
    7.69772431887396, 0.644324446261649, 5.8929636318267, 6.80704255583102, 
    9.65277598589429, 15.6595660584138, 18.3230291386697, 7.00590145796913, 
    8.82818327470461, 16.0687559775513, 24.9242677284618, 8.31785610171245, 
    32.1805817173826, 25.8697867733709, 12.7150588314216, 20.7651493664932, 
    18.0554478730553, 21.4644721045527, 0.503463387421984, 14.4342327579022, 
    15.8504018314382, 11.7250813717628, 20.5081096230163, 10.2135596451325, 
    3.02505328796601, 5.82907162879997, 25.5860423211281, 0.544995865341978, 
    17.2944017118732, 18.7849254555494, 8.52000138059573, 7.08105918495564, 
    6.07812086467522, 8.29027994713445, 9.11151411425547, 11.2791454198664, 
    2.33889383308842, 0.383052256572868, 12.6580328984268, 3.45235405356441, 
    7.53626914164849, 11.3658648339061, 11.0648355500166, 16.0923490186408, 
    4.61876233338836, 1.06877314282231, 6.09033238366572, 24.7913956466069, 
    27.4282846047303, 10.503828158498, 15.5379757855758, 6.72898335928559, 
    13.5672839062234, 1.74194371278579, 7.67665556788753, 12.1913301704955, 
    16.0070136393562, 9.94085449722289, 9.07443830079058, 22.2948828477504, 
    11.8256956903671, 6.93930514262899, 5.74851127284937, 8.9140078097432, 
    2.97604265390834, 14.3679500152691, 0.655215076434036, 0.31197833058669, 
    9.68063502577668, 8.25791016112305, 9.87727123413451, 1.20188533447447, 
    0.245383043110858, 0.603978736541778, 10.7683500965308, 26.6362496844933, 
    7.29815604310523, 0.00232326644113111, 11.5299033247717, 
    17.8594215965046, 2.00535154434501, 29.8145840281313, 0.0860884366011957, 
    4.55861921623607, 18.362737728865, 12.2555476638587, 10.8573002741448, 
    2.90891930848066, 11.1880322973122, 4.93806348492795, 28.0175979544258, 
    2.97355804348115, 0.323694470962964, 8.62268402436497, 0.67888058816554, 
    31.0493376459536, 27.9880088966251, 26.006680475186, 21.9514705177474, 
    0.736160972978013, 0.457111906246923, 31.8250813052121, 31.4677455608002, 
    20.9158362385463, 32.9688149200738, 28.100075003835, 0.613839285714286, 
    0.182042569456366, 42.4842470343099, 21.4493704042185, 28.4415856871927, 
    0.557516695770822, 0.0296347916045533, 22.2664418420983, 
    33.3287983080233, 10.8897706419535, 0.227813555980252, 30.7770121886993, 
    21.9407981675623, 12.6490174287753, 19.2223391477194, 28.0588026633956, 
    21.6065490761312, 26.0120034423877, 35.6236814666448, 10.6349942570036, 
    11.9538108809774, 12.4486082567135, 31.4552706351587, 21.4397486370015, 
    0.563375822108034, 12.9590273851226, 16.2149349344839, 0.212809169337819, 
    27.1114754746054, 42.2641652448183, 0.98403390887322, 0.337231192185904, 
    0.129888075169269, 2.46906896387, 29.9820252174333, 47.791988993971, 
    22.8557512904035, 21.3800033168467, 10.6129253702069, 42.1086644611243, 
    22.2633819364508, 2.28409392174144, 0.0519587406032618, 0.83650254239432, 
    0.325752794348189, 25.0859166043377, 23.4660791799986, 0.0510674365630132, 
    17.0280296038941, 16.7514991937267, 36.3032082342206, 37.8916024733151, 
    3.70337217984377, 50.3557221486635, 9.46915227221539, 19.2299040191962, 
    28.0230648831872, 15.1488055447112, 0.518733184399272, 31.1029477276933, 
    21.6506824784114, 3.26831224223624, 23.4118084652496, 37.3620095618343, 
    20.0089587954349, 23.1776095545435, 16.962787502487, 0.144159058827293, 
    23.9565966708147, 0.150684080889202, 13.5435576573484, 9.79296583988455, 
    11.7322406656353, 0.501146769946693, 28.1188836842407, 1.29214133694302, 
    0.0215162418809806, 12.1299130452035, 17.5501652061641, 22.4337005552758, 
    27.0072441384065, 26.9193426174107, 23.1183943818769, 1.44705819201341, 
    28.4501553387475, 27.3379908224768, 1.42070665400306, 0.123403295653518, 
    26.7671086146545, 29.618386538689, 22.9321692386654, 30.7214799723019, 
    25.7563382235633, 24.416440196159, 1.26141191720209, 32.9948282935408, 
    30.2813409261151, 29.2829270993953, 1.98164338628304, 14.194965913378, 
    31.6620806434543, 18.3911505335681, 1.40781985526593, 20.8353768647812, 
    31.4911757952172, 18.8152728265763, 27.9563140359248, 9.591138683576, 
    14.3602091388663, 38.7077713950292, 30.6973196756792, 22.9731480315794, 
    22.903891411714, 13.4692668188476, 27.4521621214404, 42.0408028221077, 
    26.8084493993947, 24.3125429271157, 0.149107940457008, 0.52087355589039, 
    28.2753738353946, 19.0915823393873, 0.115695867565156, 17.8393840986207, 
    0.0865520021583418, 0.579132902989516, 30.3670174982199, 
    2.22780224506733, 15.031257259921, 18.3962425024281, 28.6128853501327, 
    38.7729646697389, 26.0108562311078, 24.21077380265, 33.1897441750281, 
    33.5155707682459, 34.524680319106, 33.8111990365359, 18.2175192394749, 
    0.00922290442435901, 33.4003746100316, 23.9063706381937, 
    6.06679229480737, 26.3282075878526, 0.631944530493536, 5.21988162391492, 
    27.6527421862057, 32.8297105294471, 32.7070072334264, 22.2375184989346, 
    0.541052873321617, 0.862521204045138, 28.4522293638049, 23.9352620956482, 
    3.18171796501908, 0.646835302942732, 27.5641264245755, 43.7594985495537, 
    0.573370142280739, 1.4097765902758, 29.0857624252255, 0.25345184232262, 
    0.101185017040441, 0.193562312442993, 46.5786304289901, 1.36556998063373, 
    26.0926400295772, 30.0781088267115, 39.8733066921169, 23.2552037220363, 
    22.7203145590057, 0.270517461258035, 0.123172765715291, 30.3830311782978, 
    28.8004405487705, 15.4792263440198, 29.6901548833176, 27.7950530758559, 
    33.2211373117813, 39.4698262243286, 21.1405430587451, 0.248992798686076, 
    11.9978569411229, 25.1269622827636, 11.4688322870852, 49.0741649065556, 
    0.54983303784113, 9.89376896982682, 22.1535747773351, 42.4374162047506, 
    27.8453128324768, 32.2851175088502, 0.369242465349968, 29.9020858252815, 
    2.63461482752568, 34.836978775156, 0.000514472982451327, 
    55.1924422535525, 19.5358605877149, 17.9866573894485, 15.223725521198, 
    26.965492272089, 19.6903782541118, 11.9211106711411, 0.612741268930268, 
    27.1898180125529, 35.5097206852778, 1.77193247377577, 0.504385382311975, 
    28.6971653628062, 26.664221913241, 12.1450712146379, 0.912594914531611, 
    0.653199869609181, 0.141763000546702, 16.8514574098593, 31.0020560744536, 
    15.0048251851578, 0.236337788705031, 25.463149932042, 15.5543656950676, 
    11.6684564385097, 0.116523060411102, 10.0373925790112, 0.0609461328721424, 
    18.6511205477858, 13.9456689558235, 10.1376156398998, 0.965544836944703, 
    18.0970143555673, 17.6477294828546, 20.7509507268327, 0.201450512091306, 
    1.30496605882839, 0.841277337334242, 12.5124992550113, 0.0157769279577453, 
    28.0553049535279, 14.6762955356058, 13.4253763885912, 0.161735519058771, 
    22.7813238705249, 30.368258374712, 0.269847126517018, 16.367339287795, 
    0.150137115018308, 24.3276179623684, 24.4885601696715, 27.6464296567992, 
    1.0469975471641, 1.01079820286268, 28.1324288108207, 22.5724971415389, 
    23.0554202876835, 23.1744274839626, 12.0838300966779, 1.13107686121801, 
    24.4020932487445, 20.5124060673155, 30.1076670286129, 12.7798198290664, 
    14.40932028785, 13.5024274167045, 0.0277692070348658, 19.0107160402546, 
    24.0478167057796, 21.3221483945267, 24.9818648594036, 16.480321286237, 
    22.7015912559196, 22.5002139112116, 0.252271375732347, 20.7395178700976, 
    0.538498500488791, 20.6671300518552, 23.6169084594844, 2.63899636799087, 
    18.4070655678561, 16.1115882588857, 16.2958819828008, 38.19714120466, 
    26.0603850355533, 22.5697183184047, 0.00625717310798728, 
    16.1676406041064, 19.284294686413, 0.24867408217595, 9.63349526116813, 
    14.8991560112408, 4.18956962025316, 4.64792522635724, 30.9439145077233, 
    0.430769552219055, 6.17394028483987, 16.5506714788995, 18.6727569483487, 
    24.4247719462618, 20.5779296098189, 14.3595494956204, 24.2705913742231, 
    0.106024545797333, 13.1140824845271, 0.430882594770679, 28.7244368742962, 
    38.2374186372448, 0.410174295617661, 8.37417525492121, 21.5099524281897, 
    17.2798281465432, 24.8173817644958, 39.6104052427444, 15.0194965966418, 
    36.3850034318478, 0.404474272930649, 23.7110477255083, 31.739548932501, 
    1.45908651199874, 8.56903709903817, 22.0172494333812, 24.0345378564405, 
    40.0642170919195, 0.206770801127512, 18.316136046607, 7.49510996939692, 
    31.1630109875349, 3.06062775564597, 0.207012124995893, 14.0870910415894, 
    14.7449881736974, 28.0439670320353, 0.368056498709765, 16.8063810391978, 
    14.1906134141726, 34.0316618302127, 29.6978911774721, 18.0459972023136, 
    26.5638291499431, 19.2902262992565, 19.9274924288676, 16.5945161888806, 
    22.4296298755554, 24.0226320097004, 19.9590910591604, 11.5553981987564, 
    14.7655968248349, 26.0271327956708, 23.0297987979528, 0.0791749005870253, 
    9.59945449293056, 17.3992530401782, 29.2681921217183, 34.0784844466705, 
    3.03199124962163, 26.4425445612779, 19.1755058292971, 34.0585223170347, 
    18.5877881511331, 33.7048246861399, 21.5378449498758, 27.0668516445871, 
    32.4853769253982, 2.18496266846837, 35.013510532701, 16.2426862253416, 
    18.0727901195228, 31.1930702515295, 40.3243341265531, 26.2984523961311, 
    15.0505149708781, 0.549997962970508, 29.906694956415, 28.4710464894609, 
    16.3379687010528, 0.450165131918189, 0.435008607280712, 4.77946945655204, 
    2.74079233783195, 25.3769033584711, 27.5998894923631, 26.3501743297771, 
    28.0590113962846, 0.609563000402559, 26.9082759547506, 28.1101808236538, 
    10.7224069810409, 29.6692391143548, 20.5417613370745), Cell = structure(c(1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 
    12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 
    24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 
    36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 
    48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 
    60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 
    72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 
    84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 
    96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 5L, 6L, 7L, 8L, 
    9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 18L, 19L, 20L, 
    21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 30L, 31L, 32L, 
    33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 42L, 43L, 44L, 
    45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 55L, 56L, 
    57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 68L, 
    69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
    81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 
    93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 2L, 3L, 4L, 
    5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 16L, 17L, 
    18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 29L, 
    30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
    42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 
    54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 
    66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 
    78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 
    90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 99L, 100L, 1L, 
    2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 
    15L, 16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 
    27L, 28L, 29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 
    39L, 40L, 41L, 42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 
    51L, 52L, 53L, 54L, 55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 
    63L, 64L, 65L, 66L, 67L, 68L, 69L, 70L, 71L, 72L, 73L, 74L, 
    75L, 76L, 77L, 78L, 79L, 80L, 81L, 82L, 83L, 84L, 85L, 86L, 
    87L, 88L, 89L, 90L, 91L, 92L, 93L, 94L, 95L, 96L, 97L, 98L, 
    99L, 100L), .Label = c("1", "2", "3", "4", "5", "6", "7", 
    "8", "9", "10", "11", "12", "13", "14", "15", "16", "17", 
    "18", "19", "20", "21", "22", "23", "24", "25", "26", "27", 
    "28", "29", "30", "31", "32", "33", "34", "35", "36", "37", 
    "38", "39", "40", "41", "42", "43", "44", "45", "46", "47", 
    "48", "49", "50", "51", "52", "53", "54", "55", "56", "57", 
    "58", "59", "60", "61", "62", "63", "64", "65", "66", "67", 
    "68", "69", "70", "71", "72", "73", "74", "75", "76", "77", 
    "78", "79", "80", "81", "82", "83", "84", "85", "86", "87", 
    "88", "89", "90", "91", "92", "93", "94", "95", "96", "97", 
    "98", "99", "100"), class = "factor"), Animal_no = structure(c(1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 
    1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 1L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 2L, 
    2L, 2L, 2L, 2L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 
    3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 3L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 
    4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 4L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 5L, 
    5L, 5L, 5L, 5L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 
    6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 6L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 
    7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 7L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 8L, 
    8L, 8L, 8L, 8L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 
    9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 9L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 10L, 
    10L, 10L, 10L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 11L, 
    11L, 11L, 11L, 11L, 11L, 11L, 11L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 
    12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 12L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 13L, 
    13L, 13L, 13L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 14L, 
    14L, 14L, 14L, 14L, 14L, 14L, 14L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 
    15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 15L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 16L, 
    16L, 16L, 16L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 17L, 
    17L, 17L, 17L, 17L, 17L, 17L, 17L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 
    18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 18L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 19L, 
    19L, 19L, 19L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 20L, 
    20L, 20L, 20L, 20L, 20L, 20L, 20L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 
    21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 21L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 22L, 
    22L, 22L, 22L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 23L, 
    23L, 23L, 23L, 23L, 23L, 23L, 23L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 
    24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 24L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 25L, 
    25L, 25L, 25L), .Label = c("1", "2", "3", "4", "5", "6", 
    "7", "8", "9", "10", "11", "12", "13", "14", "15", "16", 
    "17", "18", "19", "20", "21", "22", "23", "24", "25"), class = "factor")), .Names = c("Sample", 
"Treatment", "Tail.intensity", "Cell", "Animal_no"), class = "data.frame", row.names = c(1L, 
2L, 3L, 4L, 5L, 6L, 7L, 8L, 9L, 10L, 11L, 12L, 13L, 14L, 15L, 
16L, 17L, 18L, 19L, 20L, 21L, 22L, 23L, 24L, 25L, 26L, 27L, 28L, 
29L, 30L, 31L, 32L, 33L, 34L, 35L, 36L, 37L, 38L, 39L, 40L, 41L, 
42L, 43L, 44L, 45L, 46L, 47L, 48L, 49L, 50L, 51L, 52L, 53L, 54L, 
55L, 56L, 57L, 58L, 59L, 60L, 61L, 62L, 63L, 64L, 65L, 66L, 67L, 
68L, 69L, 70L, 71L, 72L, 73L, 74L, 75L, 76L, 77L, 78L, 79L, 80L, 
81L, 82L, 83L, 84L, 85L, 86L, 87L, 88L, 89L, 90L, 91L, 92L, 93L, 
94L, 95L, 96L, 97L, 98L, 99L, 100L, 151L, 152L, 153L, 154L, 155L, 
156L, 157L, 158L, 159L, 160L, 161L, 162L, 163L, 164L, 165L, 166L, 
167L, 168L, 169L, 170L, 171L, 172L, 173L, 174L, 175L, 176L, 177L, 
178L, 179L, 180L, 181L, 182L, 183L, 184L, 185L, 186L, 187L, 188L, 
189L, 190L, 191L, 192L, 193L, 194L, 195L, 196L, 197L, 198L, 199L, 
200L, 201L, 202L, 203L, 204L, 205L, 206L, 207L, 208L, 209L, 210L, 
211L, 212L, 213L, 214L, 215L, 216L, 217L, 218L, 219L, 220L, 221L, 
222L, 223L, 224L, 225L, 226L, 227L, 228L, 229L, 230L, 231L, 232L, 
233L, 234L, 235L, 236L, 237L, 238L, 239L, 240L, 241L, 242L, 243L, 
244L, 245L, 246L, 247L, 248L, 249L, 250L, 301L, 302L, 303L, 304L, 
305L, 306L, 307L, 308L, 309L, 310L, 311L, 312L, 313L, 314L, 315L, 
316L, 317L, 318L, 319L, 320L, 321L, 322L, 323L, 324L, 325L, 326L, 
327L, 328L, 329L, 330L, 331L, 332L, 333L, 334L, 335L, 336L, 337L, 
338L, 339L, 340L, 341L, 342L, 343L, 344L, 345L, 346L, 347L, 348L, 
349L, 350L, 351L, 352L, 353L, 354L, 355L, 356L, 357L, 358L, 359L, 
360L, 361L, 362L, 363L, 364L, 365L, 366L, 367L, 368L, 369L, 370L, 
371L, 372L, 373L, 374L, 375L, 376L, 377L, 378L, 379L, 380L, 381L, 
382L, 383L, 384L, 385L, 386L, 387L, 388L, 389L, 390L, 391L, 392L, 
393L, 394L, 395L, 396L, 397L, 398L, 399L, 400L, 451L, 452L, 453L, 
454L, 455L, 456L, 457L, 458L, 459L, 460L, 461L, 462L, 463L, 464L, 
465L, 466L, 467L, 468L, 469L, 470L, 471L, 472L, 473L, 474L, 475L, 
476L, 477L, 478L, 479L, 480L, 481L, 482L, 483L, 484L, 485L, 486L, 
487L, 488L, 489L, 490L, 491L, 492L, 493L, 494L, 495L, 496L, 497L, 
498L, 499L, 500L, 501L, 502L, 503L, 504L, 505L, 506L, 507L, 508L, 
509L, 510L, 511L, 512L, 513L, 514L, 515L, 516L, 517L, 518L, 519L, 
520L, 521L, 522L, 523L, 524L, 525L, 526L, 527L, 528L, 529L, 530L, 
531L, 532L, 533L, 534L, 535L, 536L, 537L, 538L, 539L, 540L, 541L, 
542L, 543L, 544L, 545L, 546L, 547L, 548L, 549L, 550L, 601L, 602L, 
603L, 604L, 605L, 606L, 607L, 608L, 609L, 610L, 611L, 612L, 613L, 
614L, 615L, 616L, 617L, 618L, 619L, 620L, 621L, 622L, 623L, 624L, 
625L, 626L, 627L, 628L, 629L, 630L, 631L, 632L, 633L, 634L, 635L, 
636L, 637L, 638L, 639L, 640L, 641L, 642L, 643L, 644L, 645L, 646L, 
647L, 648L, 649L, 650L, 651L, 652L, 653L, 654L, 655L, 656L, 657L, 
658L, 659L, 660L, 661L, 662L, 663L, 664L, 665L, 666L, 667L, 668L, 
669L, 670L, 671L, 672L, 673L, 674L, 675L, 676L, 677L, 678L, 679L, 
680L, 681L, 682L, 683L, 684L, 685L, 686L, 687L, 688L, 689L, 690L, 
691L, 692L, 693L, 694L, 695L, 696L, 697L, 698L, 699L, 700L, 751L, 
752L, 753L, 754L, 755L, 756L, 757L, 758L, 759L, 760L, 761L, 762L, 
763L, 764L, 765L, 766L, 767L, 768L, 769L, 770L, 771L, 772L, 773L, 
774L, 775L, 776L, 777L, 778L, 779L, 780L, 781L, 782L, 783L, 784L, 
785L, 786L, 787L, 788L, 789L, 790L, 791L, 792L, 793L, 794L, 795L, 
796L, 797L, 798L, 799L, 800L, 801L, 802L, 803L, 804L, 805L, 806L, 
807L, 808L, 809L, 810L, 811L, 812L, 813L, 814L, 815L, 816L, 817L, 
818L, 819L, 820L, 821L, 822L, 823L, 824L, 825L, 826L, 827L, 828L, 
829L, 830L, 831L, 832L, 833L, 834L, 835L, 836L, 837L, 838L, 839L, 
840L, 841L, 842L, 843L, 844L, 845L, 846L, 847L, 848L, 849L, 850L, 
901L, 902L, 903L, 904L, 905L, 906L, 907L, 908L, 909L, 910L, 911L, 
912L, 913L, 914L, 915L, 916L, 917L, 918L, 919L, 920L, 921L, 922L, 
923L, 924L, 925L, 926L, 927L, 928L, 929L, 930L, 931L, 932L, 933L, 
934L, 935L, 936L, 937L, 938L, 939L, 940L, 941L, 942L, 943L, 944L, 
945L, 946L, 947L, 948L, 949L, 950L, 951L, 952L, 953L, 954L, 955L, 
956L, 957L, 958L, 959L, 960L, 961L, 962L, 963L, 964L, 965L, 966L, 
967L, 968L, 969L, 970L, 971L, 972L, 973L, 974L, 975L, 976L, 977L, 
978L, 979L, 980L, 981L, 982L, 983L, 984L, 985L, 986L, 987L, 988L, 
989L, 990L, 991L, 992L, 993L, 994L, 995L, 996L, 997L, 998L, 999L, 
1000L, 1051L, 1052L, 1053L, 1054L, 1055L, 1056L, 1057L, 1058L, 
1059L, 1060L, 1061L, 1062L, 1063L, 1064L, 1065L, 1066L, 1067L, 
1068L, 1069L, 1070L, 1071L, 1072L, 1073L, 1074L, 1075L, 1076L, 
1077L, 1078L, 1079L, 1080L, 1081L, 1082L, 1083L, 1084L, 1085L, 
1086L, 1087L, 1088L, 1089L, 1090L, 1091L, 1092L, 1093L, 1094L, 
1095L, 1096L, 1097L, 1098L, 1099L, 1100L, 1101L, 1102L, 1103L, 
1104L, 1105L, 1106L, 1107L, 1108L, 1109L, 1110L, 1111L, 1112L, 
1113L, 1114L, 1115L, 1116L, 1117L, 1118L, 1119L, 1120L, 1121L, 
1122L, 1123L, 1124L, 1125L, 1126L, 1127L, 1128L, 1129L, 1130L, 
1131L, 1132L, 1133L, 1134L, 1135L, 1136L, 1137L, 1138L, 1139L, 
1140L, 1141L, 1142L, 1143L, 1144L, 1145L, 1146L, 1147L, 1148L, 
1149L, 1150L, 1201L, 1202L, 1203L, 1204L, 1205L, 1206L, 1207L, 
1208L, 1209L, 1210L, 1211L, 1212L, 1213L, 1214L, 1215L, 1216L, 
1217L, 1218L, 1219L, 1220L, 1221L, 1222L, 1223L, 1224L, 1225L, 
1226L, 1227L, 1228L, 1229L, 1230L, 1231L, 1232L, 1233L, 1234L, 
1235L, 1236L, 1237L, 1238L, 1239L, 1240L, 1241L, 1242L, 1243L, 
1244L, 1245L, 1246L, 1247L, 1248L, 1249L, 1250L, 1251L, 1252L, 
1253L, 1254L, 1255L, 1256L, 1257L, 1258L, 1259L, 1260L, 1261L, 
1262L, 1263L, 1264L, 1265L, 1266L, 1267L, 1268L, 1269L, 1270L, 
1271L, 1272L, 1273L, 1274L, 1275L, 1276L, 1277L, 1278L, 1279L, 
1280L, 1281L, 1282L, 1283L, 1284L, 1285L, 1286L, 1287L, 1288L, 
1289L, 1290L, 1291L, 1292L, 1293L, 1294L, 1295L, 1296L, 1297L, 
1298L, 1299L, 1300L, 1351L, 1352L, 1353L, 1354L, 1355L, 1356L, 
1357L, 1358L, 1359L, 1360L, 1361L, 1362L, 1363L, 1364L, 1365L, 
1366L, 1367L, 1368L, 1369L, 1370L, 1371L, 1372L, 1373L, 1374L, 
1375L, 1376L, 1377L, 1378L, 1379L, 1380L, 1381L, 1382L, 1383L, 
1384L, 1385L, 1386L, 1387L, 1388L, 1389L, 1390L, 1391L, 1392L, 
1393L, 1394L, 1395L, 1396L, 1397L, 1398L, 1399L, 1400L, 1401L, 
1402L, 1403L, 1404L, 1405L, 1406L, 1407L, 1408L, 1409L, 1410L, 
1411L, 1412L, 1413L, 1414L, 1415L, 1416L, 1417L, 1418L, 1419L, 
1420L, 1421L, 1422L, 1423L, 1424L, 1425L, 1426L, 1427L, 1428L, 
1429L, 1430L, 1431L, 1432L, 1433L, 1434L, 1435L, 1436L, 1437L, 
1438L, 1439L, 1440L, 1441L, 1442L, 1443L, 1444L, 1445L, 1446L, 
1447L, 1448L, 1449L, 1450L, 1501L, 1502L, 1503L, 1504L, 1505L, 
1506L, 1507L, 1508L, 1509L, 1510L, 1511L, 1512L, 1513L, 1514L, 
1515L, 1516L, 1517L, 1518L, 1519L, 1520L, 1521L, 1522L, 1523L, 
1524L, 1525L, 1526L, 1527L, 1528L, 1529L, 1530L, 1531L, 1532L, 
1533L, 1534L, 1535L, 1536L, 1537L, 1538L, 1539L, 1540L, 1541L, 
1542L, 1543L, 1544L, 1545L, 1546L, 1547L, 1548L, 1549L, 1550L, 
1551L, 1552L, 1553L, 1554L, 1555L, 1556L, 1557L, 1558L, 1559L, 
1560L, 1561L, 1562L, 1563L, 1564L, 1565L, 1566L, 1567L, 1568L, 
1569L, 1570L, 1571L, 1572L, 1573L, 1574L, 1575L, 1576L, 1577L, 
1578L, 1579L, 1580L, 1581L, 1582L, 1583L, 1584L, 1585L, 1586L, 
1587L, 1588L, 1589L, 1590L, 1591L, 1592L, 1593L, 1594L, 1595L, 
1596L, 1597L, 1598L, 1599L, 1600L, 1651L, 1652L, 1653L, 1654L, 
1655L, 1656L, 1657L, 1658L, 1659L, 1660L, 1661L, 1662L, 1663L, 
1664L, 1665L, 1666L, 1667L, 1668L, 1669L, 1670L, 1671L, 1672L, 
1673L, 1674L, 1675L, 1676L, 1677L, 1678L, 1679L, 1680L, 1681L, 
1682L, 1683L, 1684L, 1685L, 1686L, 1687L, 1688L, 1689L, 1690L, 
1691L, 1692L, 1693L, 1694L, 1695L, 1696L, 1697L, 1698L, 1699L, 
1700L, 1701L, 1702L, 1703L, 1704L, 1705L, 1706L, 1707L, 1708L, 
1709L, 1710L, 1711L, 1712L, 1713L, 1714L, 1715L, 1716L, 1717L, 
1718L, 1719L, 1720L, 1721L, 1722L, 1723L, 1724L, 1725L, 1726L, 
1727L, 1728L, 1729L, 1730L, 1731L, 1732L, 1733L, 1734L, 1735L, 
1736L, 1737L, 1738L, 1739L, 1740L, 1741L, 1742L, 1743L, 1744L, 
1745L, 1746L, 1747L, 1748L, 1749L, 1750L, 1801L, 1802L, 1803L, 
1804L, 1805L, 1806L, 1807L, 1808L, 1809L, 1810L, 1811L, 1812L, 
1813L, 1814L, 1815L, 1816L, 1817L, 1818L, 1819L, 1820L, 1821L, 
1822L, 1823L, 1824L, 1825L, 1826L, 1827L, 1828L, 1829L, 1830L, 
1831L, 1832L, 1833L, 1834L, 1835L, 1836L, 1837L, 1838L, 1839L, 
1840L, 1841L, 1842L, 1843L, 1844L, 1845L, 1846L, 1847L, 1848L, 
1849L, 1850L, 1851L, 1852L, 1853L, 1854L, 1855L, 1856L, 1857L, 
1858L, 1859L, 1860L, 1861L, 1862L, 1863L, 1864L, 1865L, 1866L, 
1867L, 1868L, 1869L, 1870L, 1871L, 1872L, 1873L, 1874L, 1875L, 
1876L, 1877L, 1878L, 1879L, 1880L, 1881L, 1882L, 1883L, 1884L, 
1885L, 1886L, 1887L, 1888L, 1889L, 1890L, 1891L, 1892L, 1893L, 
1894L, 1895L, 1896L, 1897L, 1898L, 1899L, 1900L, 1951L, 1952L, 
1953L, 1954L, 1955L, 1956L, 1957L, 1958L, 1959L, 1960L, 1961L, 
1962L, 1963L, 1964L, 1965L, 1966L, 1967L, 1968L, 1969L, 1970L, 
1971L, 1972L, 1973L, 1974L, 1975L, 1976L, 1977L, 1978L, 1979L, 
1980L, 1981L, 1982L, 1983L, 1984L, 1985L, 1986L, 1987L, 1988L, 
1989L, 1990L, 1991L, 1992L, 1993L, 1994L, 1995L, 1996L, 1997L, 
1998L, 1999L, 2000L, 2001L, 2002L, 2003L, 2004L, 2005L, 2006L, 
2007L, 2008L, 2009L, 2010L, 2011L, 2012L, 2013L, 2014L, 2015L, 
2016L, 2017L, 2018L, 2019L, 2020L, 2021L, 2022L, 2023L, 2024L, 
2025L, 2026L, 2027L, 2028L, 2029L, 2030L, 2031L, 2032L, 2033L, 
2034L, 2035L, 2036L, 2037L, 2038L, 2039L, 2040L, 2041L, 2042L, 
2043L, 2044L, 2045L, 2046L, 2047L, 2048L, 2049L, 2050L, 2101L, 
2102L, 2103L, 2104L, 2105L, 2106L, 2107L, 2108L, 2109L, 2110L, 
2111L, 2112L, 2113L, 2114L, 2115L, 2116L, 2117L, 2118L, 2119L, 
2120L, 2121L, 2122L, 2123L, 2124L, 2125L, 2126L, 2127L, 2128L, 
2129L, 2130L, 2131L, 2132L, 2133L, 2134L, 2135L, 2136L, 2137L, 
2138L, 2139L, 2140L, 2141L, 2142L, 2143L, 2144L, 2145L, 2146L, 
2147L, 2148L, 2149L, 2150L, 2151L, 2152L, 2153L, 2154L, 2155L, 
2156L, 2157L, 2158L, 2159L, 2160L, 2161L, 2162L, 2163L, 2164L, 
2165L, 2166L, 2167L, 2168L, 2169L, 2170L, 2171L, 2172L, 2173L, 
2174L, 2175L, 2176L, 2177L, 2178L, 2179L, 2180L, 2181L, 2182L, 
2183L, 2184L, 2185L, 2186L, 2187L, 2188L, 2189L, 2190L, 2191L, 
2192L, 2193L, 2194L, 2195L, 2196L, 2197L, 2198L, 2199L, 2200L, 
2251L, 2252L, 2253L, 2254L, 2255L, 2256L, 2257L, 2258L, 2259L, 
2260L, 2261L, 2262L, 2263L, 2264L, 2265L, 2266L, 2267L, 2268L, 
2269L, 2270L, 2271L, 2272L, 2273L, 2274L, 2275L, 2276L, 2277L, 
2278L, 2279L, 2280L, 2281L, 2282L, 2283L, 2284L, 2285L, 2286L, 
2287L, 2288L, 2289L, 2290L, 2291L, 2292L, 2293L, 2294L, 2295L, 
2296L, 2297L, 2298L, 2299L, 2300L, 2301L, 2302L, 2303L, 2304L, 
2305L, 2306L, 2307L, 2308L, 2309L, 2310L, 2311L, 2312L, 2313L, 
2314L, 2315L, 2316L, 2317L, 2318L, 2319L, 2320L, 2321L, 2322L, 
2323L, 2324L, 2325L, 2326L, 2327L, 2328L, 2329L, 2330L, 2331L, 
2332L, 2333L, 2334L, 2335L, 2336L, 2337L, 2338L, 2339L, 2340L, 
2341L, 2342L, 2343L, 2344L, 2345L, 2346L, 2347L, 2348L, 2349L, 
2350L, 2401L, 2402L, 2403L, 2404L, 2405L, 2406L, 2407L, 2408L, 
2409L, 2410L, 2411L, 2412L, 2413L, 2414L, 2415L, 2416L, 2417L, 
2418L, 2419L, 2420L, 2421L, 2422L, 2423L, 2424L, 2425L, 2426L, 
2427L, 2428L, 2429L, 2430L, 2431L, 2432L, 2433L, 2434L, 2435L, 
2436L, 2437L, 2438L, 2439L, 2440L, 2441L, 2442L, 2443L, 2444L, 
2445L, 2446L, 2447L, 2448L, 2449L, 2450L, 2451L, 2452L, 2453L, 
2454L, 2455L, 2456L, 2457L, 2458L, 2459L, 2460L, 2461L, 2462L, 
2463L, 2464L, 2465L, 2466L, 2467L, 2468L, 2469L, 2470L, 2471L, 
2472L, 2473L, 2474L, 2475L, 2476L, 2477L, 2478L, 2479L, 2480L, 
2481L, 2482L, 2483L, 2484L, 2485L, 2486L, 2487L, 2488L, 2489L, 
2490L, 2491L, 2492L, 2493L, 2494L, 2495L, 2496L, 2497L, 2498L, 
2499L, 2500L, 2551L, 2552L, 2553L, 2554L, 2555L, 2556L, 2557L, 
2558L, 2559L, 2560L, 2561L, 2562L, 2563L, 2564L, 2565L, 2566L, 
2567L, 2568L, 2569L, 2570L, 2571L, 2572L, 2573L, 2574L, 2575L, 
2576L, 2577L, 2578L, 2579L, 2580L, 2581L, 2582L, 2583L, 2584L, 
2585L, 2586L, 2587L, 2588L, 2589L, 2590L, 2591L, 2592L, 2593L, 
2594L, 2595L, 2596L, 2597L, 2598L, 2599L, 2600L, 2601L, 2602L, 
2603L, 2604L, 2605L, 2606L, 2607L, 2608L, 2609L, 2610L, 2611L, 
2612L, 2613L, 2614L, 2615L, 2616L, 2617L, 2618L, 2619L, 2620L, 
2621L, 2622L, 2623L, 2624L, 2625L, 2626L, 2627L, 2628L, 2629L, 
2630L, 2631L, 2632L, 2633L, 2634L, 2635L, 2636L, 2637L, 2638L, 
2639L, 2640L, 2641L, 2642L, 2643L, 2644L, 2645L, 2646L, 2647L, 
2648L, 2649L, 2650L, 2701L, 2702L, 2703L, 2704L, 2705L, 2706L, 
2707L, 2708L, 2709L, 2710L, 2711L, 2712L, 2713L, 2714L, 2715L, 
2716L, 2717L, 2718L, 2719L, 2720L, 2721L, 2722L, 2723L, 2724L, 
2725L, 2726L, 2727L, 2728L, 2729L, 2730L, 2731L, 2732L, 2733L, 
2734L, 2735L, 2736L, 2737L, 2738L, 2739L, 2740L, 2741L, 2742L, 
2743L, 2744L, 2745L, 2746L, 2747L, 2748L, 2749L, 2750L, 2751L, 
2752L, 2753L, 2754L, 2755L, 2756L, 2757L, 2758L, 2759L, 2760L, 
2761L, 2762L, 2763L, 2764L, 2765L, 2766L, 2767L, 2768L, 2769L, 
2770L, 2771L, 2772L, 2773L, 2774L, 2775L, 2776L, 2777L, 2778L, 
2779L, 2780L, 2781L, 2782L, 2783L, 2784L, 2785L, 2786L, 2787L, 
2788L, 2789L, 2790L, 2791L, 2792L, 2793L, 2794L, 2795L, 2796L, 
2797L, 2798L, 2799L, 2800L, 2851L, 2852L, 2853L, 2854L, 2855L, 
2856L, 2857L, 2858L, 2859L, 2860L, 2861L, 2862L, 2863L, 2864L, 
2865L, 2866L, 2867L, 2868L, 2869L, 2870L, 2871L, 2872L, 2873L, 
2874L, 2875L, 2876L, 2877L, 2878L, 2879L, 2880L, 2881L, 2882L, 
2883L, 2884L, 2885L, 2886L, 2887L, 2888L, 2889L, 2890L, 2891L, 
2892L, 2893L, 2894L, 2895L, 2896L, 2897L, 2898L, 2899L, 2900L, 
2901L, 2902L, 2903L, 2904L, 2905L, 2906L, 2907L, 2908L, 2909L, 
2910L, 2911L, 2912L, 2913L, 2914L, 2915L, 2916L, 2917L, 2918L, 
2919L, 2920L, 2921L, 2922L, 2923L, 2924L, 2925L, 2926L, 2927L, 
2928L, 2929L, 2930L, 2931L, 2932L, 2933L, 2934L, 2935L, 2936L, 
2937L, 2938L, 2939L, 2940L, 2941L, 2942L, 2943L, 2944L, 2945L, 
2946L, 2947L, 2948L, 2949L, 2950L, 3001L, 3002L, 3003L, 3004L, 
3005L, 3006L, 3007L, 3008L, 3009L, 3010L, 3011L, 3012L, 3013L, 
3014L, 3015L, 3016L, 3017L, 3018L, 3019L, 3020L, 3021L, 3022L, 
3023L, 3024L, 3025L, 3026L, 3027L, 3028L, 3029L, 3030L, 3031L, 
3032L, 3033L, 3034L, 3035L, 3036L, 3037L, 3038L, 3039L, 3040L, 
3041L, 3042L, 3043L, 3044L, 3045L, 3046L, 3047L, 3048L, 3049L, 
3050L, 3051L, 3052L, 3053L, 3054L, 3055L, 3056L, 3057L, 3058L, 
3059L, 3060L, 3061L, 3062L, 3063L, 3064L, 3065L, 3066L, 3067L, 
3068L, 3069L, 3070L, 3071L, 3072L, 3073L, 3074L, 3075L, 3076L, 
3077L, 3078L, 3079L, 3080L, 3081L, 3082L, 3083L, 3084L, 3085L, 
3086L, 3087L, 3088L, 3089L, 3090L, 3091L, 3092L, 3093L, 3094L, 
3095L, 3096L, 3097L, 3098L, 3099L, 3100L, 3151L, 3152L, 3153L, 
3154L, 3155L, 3156L, 3157L, 3158L, 3159L, 3160L, 3161L, 3162L, 
3163L, 3164L, 3165L, 3166L, 3167L, 3168L, 3169L, 3170L, 3171L, 
3172L, 3173L, 3174L, 3175L, 3176L, 3177L, 3178L, 3179L, 3180L, 
3181L, 3182L, 3183L, 3184L, 3185L, 3186L, 3187L, 3188L, 3189L, 
3190L, 3191L, 3192L, 3193L, 3194L, 3195L, 3196L, 3197L, 3198L, 
3199L, 3200L, 3201L, 3202L, 3203L, 3204L, 3205L, 3206L, 3207L, 
3208L, 3209L, 3210L, 3211L, 3212L, 3213L, 3214L, 3215L, 3216L, 
3217L, 3218L, 3219L, 3220L, 3221L, 3222L, 3223L, 3224L, 3225L, 
3226L, 3227L, 3228L, 3229L, 3230L, 3231L, 3232L, 3233L, 3234L, 
3235L, 3236L, 3237L, 3238L, 3239L, 3240L, 3241L, 3242L, 3243L, 
3244L, 3245L, 3246L, 3247L, 3248L, 3249L, 3250L, 3301L, 3302L, 
3303L, 3304L, 3305L, 3306L, 3307L, 3308L, 3309L, 3310L, 3311L, 
3312L, 3313L, 3314L, 3315L, 3316L, 3317L, 3318L, 3319L, 3320L, 
3321L, 3322L, 3323L, 3324L, 3325L, 3326L, 3327L, 3328L, 3329L, 
3330L, 3331L, 3332L, 3333L, 3334L, 3335L, 3336L, 3337L, 3338L, 
3339L, 3340L, 3341L, 3342L, 3343L, 3344L, 3345L, 3346L, 3347L, 
3348L, 3349L, 3350L, 3351L, 3352L, 3353L, 3354L, 3355L, 3356L, 
3357L, 3358L, 3359L, 3360L, 3361L, 3362L, 3363L, 3364L, 3365L, 
3366L, 3367L, 3368L, 3369L, 3370L, 3371L, 3372L, 3373L, 3374L, 
3375L, 3376L, 3377L, 3378L, 3379L, 3380L, 3381L, 3382L, 3383L, 
3384L, 3385L, 3386L, 3387L, 3388L, 3389L, 3390L, 3391L, 3392L, 
3393L, 3394L, 3395L, 3396L, 3397L, 3398L, 3399L, 3400L, 3451L, 
3452L, 3453L, 3454L, 3455L, 3456L, 3457L, 3458L, 3459L, 3460L, 
3461L, 3462L, 3463L, 3464L, 3465L, 3466L, 3467L, 3468L, 3469L, 
3470L, 3471L, 3472L, 3473L, 3474L, 3475L, 3476L, 3477L, 3478L, 
3479L, 3480L, 3481L, 3482L, 3483L, 3484L, 3485L, 3486L, 3487L, 
3488L, 3489L, 3490L, 3491L, 3492L, 3493L, 3494L, 3495L, 3496L, 
3497L, 3498L, 3499L, 3500L, 3501L, 3502L, 3503L, 3504L, 3505L, 
3506L, 3507L, 3508L, 3509L, 3510L, 3511L, 3512L, 3513L, 3514L, 
3515L, 3516L, 3517L, 3518L, 3519L, 3520L, 3521L, 3522L, 3523L, 
3524L, 3525L, 3526L, 3527L, 3528L, 3529L, 3530L, 3531L, 3532L, 
3533L, 3534L, 3535L, 3536L, 3537L, 3538L, 3539L, 3540L, 3541L, 
3542L, 3543L, 3544L, 3545L, 3546L, 3547L, 3548L, 3549L, 3550L, 
3601L, 3602L, 3603L, 3604L, 3605L, 3606L, 3607L, 3608L, 3609L, 
3610L, 3611L, 3612L, 3613L, 3614L, 3615L, 3616L, 3617L, 3618L, 
3619L, 3620L, 3621L, 3622L, 3623L, 3624L, 3625L, 3626L, 3627L, 
3628L, 3629L, 3630L, 3631L, 3632L, 3633L, 3634L, 3635L, 3636L, 
3637L, 3638L, 3639L, 3640L, 3641L, 3642L, 3643L, 3644L, 3645L, 
3646L, 3647L, 3648L, 3649L, 3650L, 3651L, 3652L, 3653L, 3654L, 
3655L, 3656L, 3657L, 3658L, 3659L, 3660L, 3661L, 3662L, 3663L, 
3664L, 3665L, 3666L, 3667L, 3668L, 3669L, 3670L, 3671L, 3672L, 
3673L, 3674L, 3675L, 3676L, 3677L, 3678L, 3679L, 3680L, 3681L, 
3682L, 3683L, 3684L, 3685L, 3686L, 3687L, 3688L, 3689L, 3690L, 
3691L, 3692L, 3693L, 3694L, 3695L, 3696L, 3697L, 3698L, 3699L, 
3700L))


## ----ComBox1, echo=FALSE, result="hide", fig.cap="Boxplots for tail intensities in Comet assay.", message=FALSE, warning=FALSE, fig.pos="H", fig.height=4.5, fig.width=6.5, fig.align='center', dev="png",dpi=300----
  boxplot(Tail.intensity ~ Treatment, data=TIComet, ylab="Tail Intensity", xlab="Dose")
  points(y=TIComet$Tail.intensity, x=TIComet$Treatment, col=as.integer(TIComet$Animal_no))

## ----techn, echo=TRUE,results='asis', warning=FALSE, message=FALSE------------

TIComet$Dose[TIComet$Treatment=="VC"] <-0
TIComet$Dose[TIComet$Treatment=="Low"] <-25
TIComet$Dose[TIComet$Treatment=="medium"] <-75
TIComet$Dose[TIComet$Treatment=="High"] <-150
TIComet$Dose[TIComet$Treatment=="Superhigh"] <-500

TIComet$logTI <-log(TIComet$Tail.intensity+0.001)
mixL <- lmer(logTI ~ Dose+(1|Animal_no/Sample), data=TIComet)
EXB <- tukeytrendfit(mixL, dose="Dose", scaling=c("ari", "ord", "arilog"))
EXBB <- summary(glht(model=EXB$mmm, linfct=EXB$mlf))

## ----FENU, echo=FALSE, results='hide', warning=FALSE, message=FALSE-----------
fenu <-
structure(list(subject = c(1L, 1L, 1L, 2L, 2L, 2L, 3L, 3L, 3L, 
4L, 4L, 4L, 5L, 5L, 5L, 6L, 6L, 6L, 7L, 7L, 7L, 8L, 8L, 8L, 9L, 
9L, 9L, 10L, 10L, 10L, 11L, 11L, 11L, 12L, 12L, 12L, 13L, 13L, 
13L, 14L, 14L, 14L, 15L, 15L, 15L, 16L, 16L, 16L, 17L, 17L, 17L, 
18L, 18L, 18L), period = c(1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 
1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 
2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 
3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L, 1L, 2L, 3L), trt = c(4L, 
8L, 0L, 4L, 0L, 8L, 8L, 4L, 0L, 0L, 4L, 8L, 8L, 0L, 4L, 0L, 8L, 
4L, 0L, 4L, 8L, 0L, 8L, 4L, 8L, 4L, 0L, 4L, 0L, 8L, 4L, 8L, 0L, 
8L, 0L, 4L, 8L, 4L, 0L, 4L, 8L, 0L, 8L, 0L, 4L, 0L, 4L, 8L, 4L, 
0L, 8L, 0L, 8L, 4L), Ad_Lib_Lunch = c(1189.756, 576.962, 641.998, 
1554.898, 1297.948, 1245.796, 1261.746, 1169.432, 1126.988, 1078.072, 
1071.87, 825.067, 620.38, 1134.193, 1394.231, 727.288, 728.063, 
986.057, 957.849, 1136.109, 957.413, 2156.761, 1656.231, 1769.438, 
1206.818, 1204.818, 1385.801, 1011.428, 1345.279, 1173.188, 1173.106, 
1066.703, 1137.53, 1745.437, 1136.524, 1630.337, 1035.507, 1568.084, 
1786.647, 1327.06, 1219.183, 1276.848, 2068.996, 2616.345, 2161.583, 
1363.187, 1379.031, 1676.081, 1398.99, 794.473, 526.105, 1157.478, 
1240.915, 1207.228)), .Names = c("subject", "period", "trt", 
"Ad_Lib_Lunch"), class = "data.frame", row.names = c(NA, -54L
))

## ----fenu2, echo=TRUE, warning=FALSE, message=FALSE---------------------------
library("lme4")
modFE <- lmer(Ad_Lib_Lunch ~ trt + period + (1| subject), REML = TRUE, data=fenu)
ttfe <- tukeytrendfit(modFE, dose="trt", scaling=c("ari", "ord", "arilog"), ddf="KR")
compfe <- summary(glht(model=ttfe$mmm, linfct=ttfe$mlf))
compfe

## ----fenucorr, echo=TRUE, warning=FALSE, message=FALSE------------------------
round(cov2cor(vcov(compfe)), digits=2)

