#' Caculate the estimators of beta on the A-opt and D-opt
#'
#' @encoding UTF-8
#' @param K is the number of subsets
#' @param nk is the length of subsets
#' @param alpha is the significance level
#' @param X is the observation matrix
#' @param y is the response vector
#'
#' @return A list containing:
#' \item{betaA}{The estimator of beta on the A-opt.}
#' \item{betaD}{The estimator of beta on the D-opt.}
#' @export
#'
#' @references
#' Guo, G., Song, H. & Zhu, L. The COR criterion for optimal subset selection in distributed estimation. \emph{Statistics and Computing}, 34, 163 (2024). \doi{10.1007/s11222-024-10471-z}
#'
#' Guo, G., Sun, Y., Qian, G., & Wang, Q. (2022). LIC criterion for optimal subset selection in distributed interval estimation. \emph{Journal of Applied Statistics}, 50(9), 1900-1920. \doi{10.1080/02664763.2022.2053949}.
#'
#' Chang, D., Guo, G. (2024). LIC: An R package for optimal subset selection for distributed data. \emph{SoftwareX}, 28, 101909.
#'
#' Jing, G., & Guo, G. (2025). TLIC: An R package for the LIC for T distribution regression analysis. \emph{SoftwareX}, 30, 102132.
#'
#' Chang, D., & Guo, G. (2025). Research on Distributed Redundant Data Estimation Based on LIC. \emph{IAENG International Journal of Applied Mathematics}, 55(1), 1-6.
#'
#' Gao, H., & Guo, G. (2025). LIC for Distributed Skewed Regression. \emph{IAENG International Journal of Applied Mathematics}, 55(9), 2925-2930.
#'
#' Zhang, C., & Guo, G. (2025). The optimal subset estimation of distributed redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 270–277.
#'
#' Jing, G., & Guo, G. (2025). Student LIC for distributed estimation. \emph{IAENG International Journal of Applied Mathematics}, 55(3), 575–581.
#'
#' Liu, Q., & Guo, G. (2025). Distributed estimation of redundant data. \emph{IAENG International Journal of Applied Mathematics}, 55(2), 332–337.
#'
#' @examples
#'  p=6;n=1000;K=2;nk=200;alpha=0.05;sigma=1
#'  e=rnorm(n,0,sigma); beta=c(sort(c(runif(p,0,1))));
#'  data=c(rnorm(n*p,5,10));X=matrix(data, ncol=p);
#'  y=X%*%beta+e;
#'  beta_AD(K=K,nk=nk,alpha=alpha,X=X,y=y)
beta_AD=function(K=K,nk=nk,alpha=alpha,X=X,y=y){
  n=nrow(X);p=ncol(X)
  M=W=c(rep(1,K));
  mr=matrix(rep(0,K*nk),ncol=nk)
  R=matrix(rep(0,n*nk), ncol=n);Io=matrix(rep(0,nk*K), ncol=nk);
  for (i in 1:K){
    mr[i,]=sample(1:n,nk,replace=T);
    r=matrix(c(1:nk,mr[i,]),ncol=nk,byrow=T);
    R[t(r)]=1
    Io[i,]=r[2,]
    X1=R%*%X;y1=R%*%y;
    ux=solve(crossprod(X1))
    W[i]= sum(diag(t(ux)%*% ux))
    M[i]=  det(X1%*%t(X1))
  }
  XD= X[Io[which.max(M),],];yD= y[Io[which.max(M),]]
  betaD=solve(crossprod(XD))%*%t(XD)%*% yD
  XA= X[Io[which.min(W),],];yA= y[Io[which.min(W),]]
  betaA=solve(crossprod(XA))%*%t(XA)%*% yA
  return(list(betaA=betaA,betaD=betaD))
}
