#' Calculate n0
#'
#' @importFrom dplyr group_by count
#'
#' @param long_result_table table generated by create_table_ep_15 function
#'
#' @return The n0 number which refers to Number of Results per Run
calculate_n0 <- function(long_result_table){
  n0_table <-
    long_result_table |>
    group_by(rep) |>
    count()


  N <- sum(n0_table$n)

  SN2 <- sum(n0_table$n ^ 2)


  return((N - (SN2/N)) / (length(n0_table$rep) - 1))
}

#' Calculate ANOVA Results and Imprecision Estimates
#' @importFrom  stats aov sd
#'
#' @param ep_15_table table generated from create_table_ep_15()
#'
#' @return Named list with ANOVA Results and Imprecision Estimates
#' @export
#'
#' @examples calculate_aov_infos(create_table_ep_15(CLSIEP15::ferritin_long, data_type = 'long'))
calculate_aov_infos <- function(ep_15_table){
  return_object <- list()

  aov_results <- summary(aov(value ~ name, data = ep_15_table))

  aov_table <- data.frame(
    source_of_variation = c('between_run', 'within_run'),
    ss = aov_results[[1]]$`Sum Sq`,
    df = aov_results[[1]]$Df,
    ms = aov_results[[1]]$`Mean Sq`
  )

  return_object[['N']] <- length(ep_15_table$value)
  return_object[['k']] <- length(unique(ep_15_table$rep))

  return_object[['mean']] <- mean(ep_15_table$value)
  return_object[['sd']] <- sd(ep_15_table$value)


  return_object[['aov_table']] <- aov_table


  vw = aov_table$ms[2]

  n0 <- calculate_n0(ep_15_table)
  return_object[['n0']] = n0

  vb <- abs((aov_table$ms[1] - aov_table$ms[2])/n0)

  return_object[['Vbetween']] = vb
  return_object[['Vwithin']] = vw

  sR <- sqrt(vw)
  sB <- sqrt(vb)
  sWL <- sqrt(vw + vb)

  return_object[['SR']] = sR
  return_object[['SWL']] = sWL

  cvR <- sR/mean(ep_15_table$value)*100
  cvB <- sB/mean(ep_15_table$value)*100
  cvWL <- sWL/mean(ep_15_table$value)*100

  return_object[['CVR']] = cvR
  return_object[['CVWL']] = cvWL

  return(return_object)
}
