% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dual_logistic.R
\name{dual_logistic}
\alias{dual_logistic}
\title{Dual-Logistic Model for RR Interval Dynamics (Castillo-Aguilar et al.)}
\usage{
dual_logistic(t, params)
}
\arguments{
\item{t}{A numeric vector of time points.}

\item{params}{A named numeric vector of parameters, which must include:
\describe{
\item{alpha}{The baseline RRi level.}
\item{beta}{The amplitude parameter for the drop phase.}
\item{lambda}{The rate parameter controlling the steepness of the drop.}
\item{tau}{The time center of the drop phase.}
\item{c}{A scaling factor for the recovery phase.}
\item{phi}{The rate parameter controlling the steepness of the recovery.}
\item{delta}{The time offset for the recovery phase relative to \code{tau}.}
}}
}
\value{
A numeric vector containing the modeled RRi values at the times specified by \code{t}.
}
\description{
This function implements a dual-logistic model to capture the dynamic behavior of RR intervals (RRi)
during exercise and recovery, as described in Castillo-Aguilar et al. (2025). The model is designed to
account for the rapid drop and subsequent recovery of RRi values by combining two logistic functions.
This formulation allows for a robust characterization of the non-linear fluctuations in RRi signals,
which is critical for accurate cardiovascular monitoring and analysis.
}
\details{
The model is defined as:

\deqn{
RRi(t) = \alpha + \frac{\beta}{1 + e^{\lambda (t - \tau)}} + \frac{-c \cdot \beta}{1 + e^{\phi (t - \tau - \delta)}}
}

where:

\describe{
\item{\eqn{\alpha}}{is the baseline RRi level.}
\item{\eqn{\beta}}{controls the amplitude of the drop phase.}
\item{\eqn{\lambda}}{controls the steepness of the drop phase.}
\item{\eqn{\tau}}{defines the time at which the drop is centered.}
\item{\eqn{c}}{scales the amplitude of the recovery phase relative to \eqn{\beta}.}
\item{\eqn{\phi}}{controls the steepness of the recovery phase.}
\item{\eqn{\delta}}{shifts the recovery phase in time relative to the drop phase.}
}

This dual-logistic model is defined following the approach described in Castillo-Aguilar et al.
(2025), and is specifically tailored for RRi signal analysis in contexts where exercise-induced changes
and recovery dynamics are of interest. The model combines two logistic functions, one representing the
drop in RRi and one representing the recovery, allowing for an accurate fit even in the presence of
non-linear fluctuations. Attribution to Castillo-Aguilar et al. (2025) is provided to recognize the original
methodology that inspired this implementation.
}
\examples{
# Define example parameters based on Castillo-Aguilar et al. (2025)
params <- list(alpha = 1000, beta = -380, lambda = -3, tau = 6,
            c = 0.85, phi = -2, delta = 3)

# Simulate a time vector
t <- seq(0, 20, length.out = 150)

# Compute the dual-logistic model values
RRi_model <- dual_logistic(t, params)

# Plot the resulting model
library(ggplot2)

ggplot() +
geom_line(aes(t, RRi_model), linewidth = 1, col = "purple") +
  labs(x = "Time (min)", y = "RRi (ms)",
       title = "Dual-Logistic RRi Model",
       caption = "Castillo-Aguilar et al. (2025)") +
  theme_minimal()

}
\references{
Castillo-Aguilar, et al. (2025). \emph{Enhancing Cardiovascular Monitoring: A Non-linear Model for Characterizing RR Interval Fluctuations in Exercise and Recovery}. \strong{Scientific Reports}, 15(1), 8628.
}
