% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/getParents.R
\name{getParents}
\alias{getParents}
\title{Estimate the connectivity matrix of a causal graph}
\usage{
getParents(
  X,
  environment = NULL,
  interventions = NULL,
  parentsOf = 1:ncol(X),
  method = c("arges", "backShift", "bivariateANM", "bivariateCAM", "CAM", "fci",
    "fciplus", "ges", "gies", "hiddenICP", "ICP", "LINGAM", "mmhc", "rankArges",
    "rankFci", "rankGes", "rankGies", "rankPc", "rfci", "pc", "regression")[12],
  alpha = 0.1,
  mode = c("raw", "parental", "ancestral")[1],
  variableSelMat = NULL,
  excludeTargetInterventions = TRUE,
  onlyObservationalData = FALSE,
  indexObservationalData = 1,
  returnAsList = FALSE,
  sparse = FALSE,
  directed = FALSE,
  pointConf = FALSE,
  setOptions = list(),
  assumeNoSelectionVars = TRUE,
  verbose = FALSE,
  ...
)
}
\arguments{
\item{X}{A \eqn{(n} x \eqn{p)}-data matrix with n observations of  \eqn{p} variables.}

\item{environment}{An optional vector of length \eqn{n}, where the entry for 
observation \eqn{i} is an index for the environment in which observation \eqn{i} took 
place (Simplest case: entries \code{1} for observational data and entries
 \code{2} for interventional data of unspecified type. Encoding for observational
 data can be changed with \code{indexObservationalData}). Is required for 
 methods \code{ICP}, \code{hiddenICP} and \code{backShift}.}

\item{interventions}{A optional list of length \eqn{n}. The entry for observation
\eqn{i} is a numeric vector that specifies the variables on which interventions 
happened for observation \eqn{i} (a scalar if an intervention happened on just 
one variable and \code{integer(0)} if no intervention occured for this 
observation). Is used for methods \code{gies}, \code{rankGies} and \code{CAM} and will 
generate the vector \code{environment} if the latter is set to \code{NULL}.
(However, this might generate too many different environments for some data 
sets, so a hand-picked vector \code{environment} is preferable). Is also used 
for \code{ICP} and \code{hiddenICP} to exclude interventions on the target 
variable of interest.}

\item{parentsOf}{The variables for which we would like to estimate the 
parents. Default are all variables. Currently only used with \code{mode = "raw"}. 
Speeds up computation for methods \code{bivariateANM},
\code{bivariateCAM}, \code{ICP}, \code{hiddenICP} and \code{regression}; 
for other methods only affects output. Also see \code{variableSelMat} for possibly
speeding up computational time by restricting the set of potential parents
for a variable.}

\item{method}{A string that specfies the method to use. The methods 
\code{pc} (PC-algorithm), \code{LINGAM} (LINGAM), \code{arges} (Adaptively 
restricted greedy equivalence search), \code{ges} 
(Greedy equivalence search), \code{gies} (Greedy interventional equivalence 
search),  \code{fci} (Fast causal inference), \code{fciplus}  
and \code{rfci} (Really fast causal inference) are imported from the 
package "pcalg" and are documented there in more detail, including the 
additional options that can be supplied via \code{setOptions}. 
The "rank versions" of arges, fci, ges, gies and pc are based on [1]. The method 
\code{CAM} (Causal additive models) is documented in the package "CAM" and 
the methods \code{ICP} (Invariant causal prediction), \code{hiddenICP} 
(Invariant causal prediction with hidden variables) are from the package 
"InvariantCausalPrediction". The method \code{backShift} comes from the 
package "backShift". The method \code{mmhc} comes from the 
package "bnlearn". Finally, the methods \code{bivariateANM} and 
\code{bivariateCAM} are for now implemented internally but will hopefully 
be part of another package at some point in the near future.}

\item{alpha}{The level at which tests are done. This leads to confidence 
intervals for \code{ICP} and \code{hiddenICP} and is used internally for 
\code{pc}, \code{rankPc}, \code{mmhc}, \code{fci}, \code{rankFci}, \code{fciplus}
 and \code{rfci}. For all other methods \code{alpha} is not used.}

\item{mode}{Determines output type - can be "raw" or one of the queries "isParent", 
"isMaybeParent", "isNoParent", "isAncestor","isMaybeAncestor", "isNoAncestor".
If "raw", \code{getParents()} returns the connectivity matrix computed by the
specified method in sparse matrix format if \code{sparse} is set to \code{TRUE}; 
else in dense matrix format (or as list if \code{returnAsList = TRUE}). 
The options \code{directed} and  \code{pointConf} will be ignored for 
all modes except for "raw" if set to \code{TRUE}. The different mode types
are explained in the help for \code{\link{getRanking}}.}

\item{variableSelMat}{An optional logical matrix of dimension  \eqn{(p} x \eqn{p)}. An 
entry \code{TRUE} for entry \eqn{(i,j)} says that variable \eqn{i} should be considered 
as a potential parent for variable \eqn{j} and vice versa for \code{FALSE}. If the 
default value of \code{NULL} is used, all variables will be considered, but 
this can be very slow, especially for methods \code{pc}, \code{ges}, 
\code{gies}, \code{rfci} and \code{CAM}. Ignored for methods \code{backShift}, 
\code{fciplus}, \code{LINGAM} and \code{CAM}.}

\item{excludeTargetInterventions}{When looking for parents of variable \eqn{k} 
in \eqn{1,...,p}, set to \code{TRUE} if observations where an intervention on 
variable \eqn{k} occured should be excluded. Default is \code{TRUE}. Used
in  \code{ICP} and \code{hiddenICP}.}

\item{onlyObservationalData}{If set to \code{TRUE}, only observational data 
is used. It will take the index in \code{environment} specified by 
\code{indexObservationalData}. If \code{environment} is \code{NULL}, all 
observations are used. Default is \code{FALSE}.}

\item{indexObservationalData}{Index in \code{environment} that encodes 
observational data. Default is \code{1}.}

\item{returnAsList}{If set to \code{TRUE}, will return a list, where entry 
\eqn{k} is a list containing the estimated parents of variable \eqn{k}. 
Default is \code{FALSE}.}

\item{sparse}{If set to \code{TRUE} and \code{returnAsList} is \code{FALSE},
output matrix will be in sparse matrix format.}

\item{directed}{If \code{TRUE}, an edge will be returned if and only if an 
edge has been detected to be directed. I.e. entry will be set to 0 for entry 
\eqn{(j,k)} if both \eqn{j -> k} and \eqn{k -> j} are estimated 
(\code{ICP}, \code{hiddenICP}, \code{regression}), if \eqn{j -- k} is undirected 
(in case of CPDAGs) or if the edge type is not of type \eqn{i --> j} in case of 
PAGs. If \code{assumeNoSelectionVars = TRUE} the edge type \eqn{i --o j} is also 
considered 'directed' for methods returning PAGs.  Default is \code{FALSE}. 
Only supported in mode "raw".}

\item{pointConf}{If \code{TRUE}, numerical estimates will be returned if 
possible. For methods \code{ICP} and \code{hiddenICP}, these are the values 
in the individual confidence intervals (at chosen level \code{alpha}) that 
are closest to 0; for other methods these are point estimates. Some methods 
do not return numerical point estimates; for these the output will remain 
binary 0/1 (no-edge/edge). Default is \code{FALSE}. Only supported in mode "raw".}

\item{setOptions}{A list that can take method-specific options; see the 
individual documentations of the methods for more options and their 
possible values.}

\item{assumeNoSelectionVars}{Set to \code{TRUE} is you want to assume the absence 
of selection variables.}

\item{verbose}{If \code{TRUE}, detailed output is provided.}

\item{...}{Parameters to be passed to underlying method's function.}
}
\value{
If option \code{returnAsList} is \code{FALSE}, a sparse matrix, 
where a 0 entry in position (j,k) corresponds to an estimate of "no edge" 
\code{j} -> \code{k}, while an entry 1 corresponds to an 
estimated egde. If option \code{pointConf} is \code{TRUE}, the 1 entries 
will be replaced by numerical values that are either point estimates of the 
causal coefficients or confidence bounds (see above). 
If option \code{returnAsList} is \code{TRUE}, a list will be returned. 
The k-th entry in the list is the numeric vector with the indices of the 
estimated parents of node \code{k}.
}
\description{
Estimates the connectivity matrix of a directed causal graph, 
using various possible methods. Supported methods at the moment are ARGES,
backShift, bivariateANM, bivariateCAM, CAM, FCI, FCI+, GES, GIES, hiddenICP, 
ICP, LINGAM, MMHC, rankARGES, rankFci, rankGES, rankGIES, rankPC, 
regression, RFCI and PC.
}
\examples{
## load the backShift package for data generation and plotting functionality
if(require(backShift) & require(pcalg)){
  # Simulate data with connectivity matrix A with assumptions
  # 1) hidden variables present
  # 2) precise location of interventions is assumed unknown
  # 3) different environments can be distinguished
  
  ## simulate data
  myseed <- 1
  
  # sample size n
  n <- 10000
  
  # p=3 predictor variables and connectivity matrix A
  p <- 3
  labels <- c("1", "2", "3")
  A <- diag(p)*0
  A[1,2] <- 0.8
  A[2,3] <- 0.8
  A[3,1] <- -0.4
  
  # divide data in 10 different environments
  G <- 10
  
  # simulate
  simResult <- backShift::simulateInterventions(n, p, A, G, intervMultiplier = 3,
               noiseMult = 1, nonGauss = TRUE, hiddenVars = TRUE,
               knownInterventions = FALSE, fracVarInt = NULL, simulateObs = TRUE,
               seed = myseed)
  X <- simResult$X
  environment <- simResult$environment
  
  ## apply all  methods given in vector 'methods'
  ## (using all data pooled for pc/LINGAM/rfci/ges -- can be changed with option
  ## 'onlyObservationalData=TRUE')
  
  methods <- c("backShift", "LINGAM") #c("pc", "rfci", "ges")
  
  # select whether you want to run stability selection
  stability <- FALSE
  
  # arrange graphical output into a rectangular grid
  sq <- ceiling(sqrt(length(methods)+1))
  par(mfrow=c(ceiling((length(methods)+1)/sq),sq))
  
  ## plot and print true graph
  cat("\n true graph is  ------  \n" )
  print(A)
  plotGraphEdgeAttr(A, plotStabSelec = FALSE, labels = labels, thres.point = 0,
   main = "TRUE GRAPH")
  
  ## loop over all methods and compute and print/plot estimate
  for (method in methods){
    cat("\n result for method", method,"  ------  \n" )
  
    if(!stability){
      # Option 1): use this estimator as a point estimate
      Ahat <- getParents(X, environment, method=method, alpha=0.1, pointConf = TRUE)
    }else{
      # Option 2): use a stability selection based estimator
      # with expected number of false positives bounded by EV=2
      Ahat <- getParentsStable(X, environment, EV=2, method=method, alpha=0.1)
    }
  
   # print and plot estimate (point estimate thresholded if numerical estimates
   # are returned)
    print(Ahat)
    if(!stability)
      plotGraphEdgeAttr(Ahat, plotStabSelec = FALSE, labels = labels,
       thres.point = 0.05,
       main=paste("POINT ESTIMATE FOR METHOD\n", toupper(method)))
    else
      plotGraphEdgeAttr(Ahat, plotStabSelec = TRUE, labels = labels,
       thres.point = 0, main = paste("STABILITY SELECTION
       ESTIMATE\n FOR METHOD", toupper(method)))
   }
}else{
    cat("\nThe packages 'backShift' and 'pcalg' are needed for the examples to
work. Please install them.")
}
 

 
}
\references{
\enumerate{
\item Naftali Harris and Mathias Drton: PC Algorithm for Nonparanormal 
Graphical Models. J. Mach. Learn. Res. 14(1) 2013.
}
}
\seealso{
\code{\link{getParentsStable}} for stability selection-based 
estimation of the causal graph.
}
\author{
Christina Heinze-Deml \email{heinzedeml@stat.math.ethz.ch}, 
 Nicolai Meinshausen \email{meinshausen@stat.math.ethz.ch}
}
\concept{Causality}
\concept{Graph estimation}
