% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/observedInformation.R
\name{varEst}
\alias{varEst}
\alias{summary.mixVarEst}
\alias{print.summary.mixVarEst}
\title{Estimated asymptotic variance matrix for MLE}
\usage{
varEst(mixture, mle, npars, method = "Richardson", ...)

\method{summary}{mixVarEst}(object, transform = FALSE, ...)

\method{print}{summary.mixVarEst}(
  x,
  digits = max(3, getOption("digits") - 3),
  scientific = FALSE,
  print.gap = 3L,
  ...
)
}
\arguments{
\item{mixture}{A \code{\link{DNAmixture}}.}

\item{mle}{A \code{mixpar}, typically obtained by \code{mixML}.}

\item{npars}{A list of integers specifying the number of each of the four
parameters \code{rho}, \code{eta}, \code{xi} and \code{phi}. Allowed values are
\describe{
\item{\code{0}}{The parameter is fixed, but might differ across mixtures.}
\item{\code{1}}{The parameter is equal across mixtures.}
\item{\code{N}}{There is one parameter for each of the \code{N} mixtures in the model.}
}}

\item{method}{Method for numeric differentiation used in \code{\link[numDeriv]{hessian}}}

\item{...}{Arguments to be passed on to other methods.}

\item{object}{An object of class \code{mixVarEst}, typically obtained by a call to \code{\link{varEst}}.}

\item{transform}{Should the parameterisation \eqn{(\mu, \sigma)} be used? Defaults to \code{FALSE}.}

\item{x}{An object of class \code{"summary.mixVarEst"}.}

\item{digits}{Number of significant digits to print}

\item{scientific}{Should scientific notation be used?}

\item{print.gap}{Distance between columns in the printing of the summary.}
}
\value{
The mle and the estimated covariance matrix in different parametrisations
\item{\code{cov} and \code{mle}}{\eqn{\rho, \eta, \xi, \phi}}
\item{\code{cov.trans} and \code{mle.trans}}{\eqn{\mu, \sigma, \xi, \phi}}
\item{\code{cov.res}}{A non-singular covariance matrix for a
reparametrisation of \eqn{\rho, \eta, \xi, \phi}, collapsing the
parameters according to the specified constrains and removing one
contributor from \eqn{\phi}.}

An integer suffix is used to indicate which mixture the parameter
is associated with. In the restricted covariance matrix, all fixed
parameters are left out. If parameters are equal accross mixtures,
the suffix for this parameter will be \code{.1}. If parameters are
unconstrained and there are \code{N} mixtures in the model,
suffixes are \code{.1, \ldots, .N}
}
\description{
Provided that the user specifies the MLE as well as the
constraints used in the maximisation, this function computes an
estimate of the variance of the MLE based on the observed
information. The observed information is obtained by numerically
deriving the Hessian of the log-likelihood function.

\emph{IMPORTANT: This is the \pkg{DNAmixturesLite} package, which
is intended as a service to enable users to try \pkg{DNAmixtures}
without purchasing a commercial licence for Hugin. When at all
possible, we strongly recommend the use of \pkg{DNAmixtures}
rather than this lite-version. See
\url{https://dnamixtures.r-forge.r-project.org/} for details on
both packages.}

\emph{While the lite-version seeks to provide the full functionality of
\pkg{DNAmixtures}, note that computations are much less efficient
and that there are some differences in available functionality. Be
aware that the present documentation is copied from
\pkg{DNAmixtures} and thus may not accurately describe the
implementation of this lite-version.}
}
\details{
As the user can apply highly customized constraints to the model
parameters when maximising with \code{\link{mixML}}, it is a
complicated matter to write a generic function for computing the
asymptotic variance matrix. We have thus restricted attention to
the case where each of the (multi-dimensional) parameters \code{rho},
\code{eta}, \code{xi} and \code{phi} can be either

\itemize{
\item fixed at known values
\item unknown, but common across traces
\item unconstrained
}
}
\examples{

data(MC18, USCaucasian)
mixHp <- DNAmixture(list(MC18), k = 3, K = c("K1", "K2", "K3"), C = list(50),
                    database = USCaucasian)
p <- mixpar(rho = list(30), eta = list(34), xi = list(0.08),
            phi = list(c(K1 = 0.71, K3 = 0.1, K2 = 0.19)))
mlHp <- mixML(mixHp, pars = p)
## Find the estimated covariance matrix of the MLE
V.Hp <- varEst(mixHp, mlHp$mle, npars = list(rho=1,eta=1,xi=1,phi=1))
V.Hp$cov ## using (rho, eta)
V.Hp$cov.trans ## using (mu, sigma)
## The summary is a table containing the MLE and their standard errors
summary(V.Hp)

\donttest{
data(MC18, USCaucasian)
mixmult <- DNAmixture(list(MC18), C = list(50), k = 3, K = c("K1", "K2"), database = USCaucasian)
startpar <- mixpar(rho = list(30), eta = list(28), xi = list(0.08),
                   phi = list(c(U1 = 0.2, K1 = 0.7, K2 = 0.1)))
ml.mult <- mixML(mixmult, startpar)
Vmult <- varEst(mixmult, ml.mult$mle, list(rho=1,eta=1,xi=1,phi=1))
summary(Vmult)
}

\donttest{
## Be aware that the following two advanced examples are computationally demanding and
## typically have a runtime of several minutes with the lite-version of DNAmixtures.

data(MC15, MC18, USCaucasian)
mix <- DNAmixture(list(MC15, MC18), C = list(50, 38), k = 3, K = c("K1", "K2"),
database = USCaucasian)
startpar <- mixpar(rho = list(30, 30), eta = list(28, 35), xi = list(0.08, 0.1),
                   phi = list(c(U1 = 0.05, K1 = 0.7, K2 = 0.25),
                                c(K1 = 0.7, K2 = 0.1, U1 = 0.2)))
eqxis <- function(x){ diff(unlist(x[,"xi"])) }
## Here we set stutter equal for all traces
ml.diff <- mixML(mix, startpar, eqxis, val = 0, phi.eq = FALSE)
V.diff <- varEst(mix, ml.diff$mle, list(rho=2,eta=2,xi=1,phi=2))
summary(V.diff)

## Fixing stutter to 0.07
xival <- function(x){unlist(x[,"xi"])}
ml.eq <- mixML(mix, startpar, xival, val = c(0.07, 0.07), phi.eq = FALSE)
V.eq <- varEst(mix, ml.eq$mle, list(rho=2,eta=2,xi=0,phi=2))
summary(V.eq)
}
}
