#' @title Estimated Lambda
#' @description Estimate upper and lower bound of new tuning region of regularization parameter Lambda.
#'
#' @param cv.fit An object of either class "cv.glmnet" from glmnet::cv.glmnet() or class "cv.ncvreg" from ncvreg::cv.ncvreg(), which is a list generated by a cross-validation fit.
#' @param pihat eatimated proprtion from HCTR::est.prop().
#' @param p Total number of variables, except for covariates.
#' @param cov.num Number of covariates in model, default is 0. Covariate matrix, W, is assumed on the left side of variable matrix, X. The column index of covariates are before those of variables.
#'
#' @return A list of (1) lambda.max, upper bound of new tuning region; (2) lambda.min, lower bound of new tuning region.
#'
#' @export
#' @import glmnet
#' @import ncvreg
#'
#' @examples
#' set.seed(10)
#' X <- matrix(rnorm(20000), nrow = 100)
#' beta <- rep(0, 200)
#' beta[1:100] <- 5
#' Y <- MASS::mvrnorm(n = 1, mu = X%*%beta, Sigma = diag(100))
#' fit <- glmnet::cv.glmnet(x = X, y = Y)
#' pihat <- 0.01
#' result <- est.lambda(cv.fit = fit, pihat = pihat, p = ncol(X))
est.lambda <- function(cv.fit, pihat, p, cov.num = 0) {
  upper <- ceiling(pihat*p)
  lower <- floor(pihat*p)
  if (cov.num > 0) {
    upper <- (upper + cov.num)
    lower <- (lower + cov.num)
  }
  if (class(cv.fit) == "cv.glmnet") {
    lambda.max <- min(cv.fit$lambda[which(cv.fit$nzero == max(cv.fit$nzero[which(cv.fit$nzero <= lower)]))])
    lambda.min <- max(cv.fit$lambda[which(cv.fit$nzero == min(cv.fit$nzero[which(cv.fit$nzero >= upper)]))])
  } else if (class(cv.fit) == "cv.ncvreg") {
    nzero <- (colSums(cv.fit$fit$beta != 0) - 1)

    lambda.max <- min(cv.fit$lambda[which(nzero == max(nzero[which(nzero <= lower)]))])
    lambda.min <- max(cv.fit$lambda[which(nzero == min(nzero[which(nzero >= upper)]))])
  } else {
    stop("Invalid fitted model!")
  }

  lambda.max <- as.numeric(lambda.max)
  lambda.min <- as.numeric(lambda.min)
  result <- list("lambda.max" = lambda.max, "lambda.min" = lambda.min)
  return (result)
}
