#' @title Compute classification error probabilities for attributes
#'
#' @description
#' Function to compute classification error probabilities (CEP) for attributes (for single time-point situations).Details can be found in Iaconangelo (2017).
#'
#' @references Uses of classification error probabilities in the three-step approach to estimating cognitive diagnosis models (Doctoral dissertation). https://rucore.libraries.rutgers.edu/rutgers-lib/55495/PDF/1/play/
#'
#'
#' @param GDINA.obj a G-DINA model object returned from \code{GDINA} R package.
#'
#'
#' @return a list with elements
#' \describe{
#' \item{SL.k}{the sample-level correction weights}
#' \item{PDL.k}{the posterior distribution level correction weights}
#' }
#' @import GDINA
#' @examples
#' if(requireNamespace("GDINA")){
#' library(GDINA)
#' dat <- sim10GDINA$simdat
#' Q <- matrix(c(1,0,0,
#'               0,1,0,
#'               0,0,1,
#'               1,0,1,
#'               0,1,1,
#'               1,1,0,
#'               0,0,1,
#'               1,0,0,
#'               1,1,1,
#'               1,0,1),byrow = TRUE,ncol = 3)
#' fit.object <- GDINA(dat = dat, Q = Q, model = "GDINA", att.dist = "independent",verbose = FALSE)
#' CEP <- CEP_1t(fit.object)
#' }
#' @export

CEP_1t <- function(GDINA.obj){

  Q = extract(GDINA.obj, "Q")
  K = ncol(Q)
  J = nrow(Q)
  N = extract(GDINA.obj, "nobs")
  patt <- attributepattern(K)

  #============Attribute level correction weights================
  SL.k=list()
  PDL.k=list()

  mp <- personparm(GDINA.obj, what = "mp") # Person Marginal posterior
  eap=personparm(GDINA.obj, what = "EAP")

  for (k in 1:K){
    SL_k=list()
    PDL_k=list()
    Den=colSums(cbind(1-mp[,k],mp[,k]))
    tmp=matrix(0,2,2)
    for (i in 1:N){
      P_ik=c(1-mp[i,k],mp[i,k])
      P_iq=t(c(1-eap[i,k],eap[i,k]))
      PDL_k[[i]]=P_ik%*%P_iq*N/Den
      tmp=tmp+P_ik%*%P_iq
    }
    SL.k[[k]]=tmp/Den
    PDL.k[[k]]=PDL_k
  }

  return(list(SL.k=SL.k, PDL.k=PDL.k))
}
