% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mcpmod_simulation.R
\name{mcpmod_simulation}
\alias{mcpmod_simulation}
\title{Simulation to obtain operating characteristics for MCP-Mod design}
\usage{
mcpmod_simulation(doses, parm, sample.size, model.true,
  models.candidate, selModel = "AIC",significance.level = 0.025,
  Delta, distr = "weibull", censoring.rate = NULL,
  sigma.estimator = NULL, n.cores, seed,n.sim)
}
\arguments{
\item{doses}{a numeric vector indicating the doses that will be considered in the clinical trial.}

\item{parm}{a named list of true values for the simulation. See more details in \code{\link{data_generator}}.}

\item{sample.size}{a numeric vector indicating the sample sizes per dose in the clinical trial to be evaluated in the simulation study.}

\item{model.true}{a character value indicating the true functional form of dose-response curve. See more details in \code{\link{model_response}}.}

\item{models.candidate}{an object of class 'Mods'. See more details in \code{\link[DoseFinding]{Mods}}.}

\item{selModel}{a character value indicating the model selection criterion for dose estimation. See more details in \code{\link[DoseFinding]{MCPMod}}.}

\item{significance.level}{a numeric value indicating the significance level when evaluating proof-of-concept based on an one-sided Wald test.}

\item{Delta}{a numerical value indicating the target effect size used for the target dose. See \code{\link[DoseFinding]{TD}}.}

\item{distr}{a character value indicating the distribution of the response variable. Currently, the only option available is 'weibull'.}

\item{censoring.rate}{a numeric value between 0 and 1 indicating the censoring rate when generated data. It is required when \code{distr = "weibull"}.}

\item{sigma.estimator}{a character value indicating whether the estimator for sigma should be a maximum likelihood or 
jackknife estimator. It is required when \code{distr = "weibull"}. Options are "mle" and "jackknife".}

\item{n.cores}{a numeric value indicating the number of cores to be used in the 'simulation performed in parallel. 
Use parallel::detectCores() to check the number of cores available.}

\item{seed}{an integer value, containing the random number generator (RNG) state for random number generation.}

\item{n.sim}{a numerical value indicating the number of simulated trials.}
}
\value{
An object of class mcpmod_simulation with the following components:

\code{mle}: a matrix of dimension n.sim \eqn{\times} 4 with results when using the MCP-Mod approach with MLE;

\code{mle2}: a matrix of dimension n.sim \eqn{\times} 4 with results when using the MCP-Mod approach with MLE2;

\code{bce}: a matrix of dimension n.sim \eqn{\times} 4 with results when using the MCP-Mod approach with BCE;

\code{bce2}: a matrix of dimension n.sim \eqn{\times} 4 with results when using the MCP-Mod approach with BCE2;

\code{firth}: a matrix of dimension n.sim \eqn{\times} 4 with results when using the MCP-Mod approach with Firth's estimator;

All matrices contain the following columns:
(1) the first column indicates whether proof-of-concept (1 = "yes", 0 = "no"), in other words, the p-value of Wald test was statistically significant;
(2) the second column indicates whether the true model was selected to estimate the dose-response curve (1 = "yes", 0 = "no") when proof-of-concept is demonstrated;
(3) the third column contains the estimated target dose;
(4) the fourth column contains the sample size considered in the trial.

\code{conditions}: a list containing the conditions of the simulation.

}
\description{
It simulates dose-finding trials using MCP-Mod design with Maximum Likelihood Estimator and Fisher Information (MLE), 
Maximum Likelihood Estimator and second-order Fisher Information (MLE2), Cox and Snell's Bias-Corrected Estimator and 
Fisher Information (BCE), Cox and Snell's Bias-Corrected Estimator and second-order Fisher Information (BCE2), and 
Firth Bias-Corrected estimators (Firth) as discussed in Diniz, Magalhães and Gallardo.
}
\examples{
\donttest{
library(DoseFinding)
library(MCPModBC)

## doses scenarios 
doses <- c(0, 5, 25, 50, 100)
nd <- length(doses)
sample.size <- 25

# shape parameter
sigma.true <- 0.5

# median survival time for placebo dose
mst.control <- 4 

# maximum hazard ratio between active dose and placebo dose 
hr.ratio <- 4  
# minimum hazard ratio between active dose and placebo dose
hr.Delta <- 2 

# hazard rate for placebo dose
placEff <- log(mst.control/(log(2)^sigma.true)) 

# maximum hazard rate for active dose
maxEff <- log((mst.control*(hr.ratio^sigma.true))/(log(2)^sigma.true))

# minimum hazard rate for active dose
minEff.Delta <- log((mst.control*(hr.Delta^sigma.true))/(log(2)^sigma.true))
Delta <- (minEff.Delta - placEff)
	
## MCP Parameters 
significance.level <- 0.05
selModel <- "AIC"

emax <- guesst(d = doses[4], p = 0.5, model="emax")
exp <- guesst(d = doses[4], p = 0.1, model="exponential", Maxd = doses[nd])
logit <- guesst(d = c(doses[3], doses[4]), p = c(0.1,0.8), "logistic",  
	Maxd= doses[nd])
betam <- guesst(d = doses[2], p = 0.3, "betaMod", scal=120, dMax=50, 
	Maxd= doses[nd])

models.candidate <- Mods(emax = emax, linear = NULL,
                         exponential = exp, logistic = logit,
                         betaMod = betam, doses = doses,
                         placEff = placEff, maxEff = (maxEff- placEff))
plot(models.candidate)

## Simulation Parameters
n.sim <- 10
seed <- 1234 
n.cores <- 1

## True Model
model.true <- "emax"
response <- model_response(doses = doses,
                           distr = "weibull", 
                           model.true = model.true, 
                           models.candidate = models.candidate) 
lambda.true <- response$lambda
parm <- list(lambda = lambda.true, sigma = sigma.true)

## Scenario: Censoring 10\%
censoring.rate <- 0.1

test <- mcpmod_simulation(doses = doses,
           parm = parm, sample.size = sample.size,
           model.true = model.true,
           models.candidate = models.candidate,
           selModel = selModel,
           significance.level = significance.level,
           Delta = Delta, distr = "weibull",
           censoring.rate = censoring.rate,
           sigma.estimator = "jackknife",
           n.cores = n.cores, seed = seed, n.sim = n.sim)

test
summary(test)
}

}
\references{
Bretz F, Pinheiro JC, Branson M. Combining multiple comparisons and modeling techniques in dose‐response studies. Biometrics. 2005 Sep;61(3):738-48.

Bornkamp B, Pinheiro J, Bretz F. MCPMod: An R package for the design and analysis of dose-finding studies. Journal of Statistical Software. 2009 Feb 20;29:1-23.

Diniz, Márcio A. and Gallardo, Diego I. and Magalhães, Tiago M. (2023). Improved inference  for MCP-Mod approach for time-to-event endpoints with small sample sizes. arXiv 
<doi.org/10.48550/arXiv.2301.00325>

Pinheiro J, Bornkamp B, Glimm E, Bretz F. Model‐based dose finding under model uncertainty using general parametric models. Statistics in medicine. 2014 May 10;33(10):1646-61.

}

\author{
Diniz, M.A., Gallardo D.I., Magalhaes, T.M.
}
