% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/goft_IVK.R
\name{goft}
\alias{goft}
\title{Goodness-of-fit tests for the cure rate in a mixture cure model}
\usage{
goft(
  x,
  time,
  delta,
  model = c("logit", "probit", "cloglog"),
  theta0 = NULL,
  nsimb = 499,
  h = NULL
)
}
\arguments{
\item{x}{A numeric vector representing the covariate of interest.}

\item{time}{A numeric vector of observed survival times.}

\item{delta}{A numeric vector indicating censoring status (1 = event occurred, 0 = censored).}

\item{model}{A character string specifying the parametric model for the incidence part. Can be \code{"logit"}, \code{"probit"}, or \code{"cloglog"}.}

\item{theta0}{Optional numeric vector with initial values for the model parameters. Default is \code{NULL}.}

\item{nsimb}{An integer indicating the number of bootstrap replicates.Default is \code{499}.}

\item{h}{Optional bandwidth value used for nonparametric estimation of the cure rate. Default is \code{NULL}.}
}
\value{
A list with the following components:
\describe{
\item{statistic}{Numeric value of the test statistic.}
\item{p.value}{Numeric value of the bootstrap p-value for testing the null hypothesis.}
\item{bandwidth}{The bandwidth used.}
}
}
\description{
The aim of this function is to test whether the cure rate \eqn{p}, as a function of the covariates, satisfies a certain parametric model.
}
\details{
We want to test wether the cure rate \eqn{p}, as a function of covariates, satisfies a certain parametric model, such as, logistic, probit or cloglog model.
The hypothesis are:
\deqn{
  \mathcal{H}_0 : p = p_{\theta} \quad \text{for some} \quad \theta \in \Theta
  \quad \text{vs} \quad
  \mathcal{H}_1 : p \neq p_{\theta} \quad \text{for all} \quad \theta \in \Theta,
}
where \eqn{\Theta} is a finite-dimensional parameter space and \eqn{p_{\theta}} is a known function up to the parameter vector \eqn{\theta}.

The test statistic is based on a weighted \eqn{L_2} distance between a nonparametric estimator \eqn{\hat{p}(x)} and a parametric estimator \eqn{p_{\hat{\theta}}(x)} under \eqn{\mathcal{H}_0},
as proposed by Müller and Van Keilegom (2019):
\deqn{
  \mathcal{T}_n = n h^{1/2} \int \left(\hat{p}(x) - p_{\hat{\theta}}(x)\right)^2 \pi(x) dx,
}
where \eqn{\pi(x)} is a known weighting function, often chosen as the covariate density \eqn{f(x)}.

A practical empirical version of the statistic is given by:
\deqn{
  \tilde{\mathcal{T}}_n = n h^{1/2} \frac{1}{n} \sum_{i = 1}^n \left(\hat{p}(x_i) - p_{\hat{\theta}}(x_i)\right)^2,
}
where the integral is replaced by a sample average.
}
\examples{

## Some artificial data
set.seed(123)
n <- 50
x <- runif(n, -2, 2) ## Covariate values
y <- rweibull(n, shape = .5*(x + 4)) ## True lifetimes
c <- rexp(n) ## Censoring values
p <- exp(2*x)/(1 + exp(2*x)) ## Probability of being susceptible
u <- runif(n)
t <- ifelse(u < p, pmin(y, c), c) ## Observed times
d <- ifelse(u < p, ifelse(y < c, 1, 0), 0) ## Uncensoring indicator
data <- data.frame(x = x, t = t, d = d)

goft(x, t, d, model = "logit")
}
\references{
Müller, U.U, & Van Keilegom, I. (2019).
Goodness-of-fit tests for the cure rate in a mixture cure model.
\emph{Biometrika}, 106, 211-227. \doi{10.1093/biomet/asy058}
}
