% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PCOselect.R
\name{PCOselect}
\alias{PCOselect}
\alias{plot.regions_pco_select}
\alias{summary.regions_pco_select}
\title{Select PCO scores}
\usage{
PCOselect(
  pco,
  method = "manual",
  scores = NULL,
  cutoff = 0.05,
  nreps = 500,
  results = NULL,
  criterion = "aic",
  verbose = TRUE
)

\method{plot}{regions_pco_select}(x, ...)

\method{summary}{regions_pco_select}(object, ...)
}
\arguments{
\item{pco}{a \code{regions_pco} object; the output of a call to \code{\link[=svdPCO]{svdPCO()}}.}

\item{method}{string; the method used to select the number of PCOs. Allowable options include \code{"manual"}, \code{"boot"}, \code{"variance"}, and \code{"max"}. Default is \code{"manual"}. Abbreviations allowed. See Details.}

\item{scores}{when \code{method = "manual"}, the number of PCO scores to use.}

\item{cutoff}{when \code{method = "variance"}, the cutoff for the variance explained by each PCO score.}

\item{nreps}{when \code{method = "boot"}, the number of bootstrap replications to use.}

\item{results}{when \code{method = "max"}, a \code{regions_results} object, the output of a call to \code{\link[=calcregions]{calcregions()}}.}

\item{criterion}{when \code{method = "max"}, which criterion should be used to select the number of scores. Allowable options include \code{"aic"} and \code{"bic"}. Abbreviations allowed.}

\item{verbose}{when \code{method = "boot"}, whether to display a progress bar. Default is \code{TRUE}.}

\item{x}{for \code{plot.regions_pco_select()}, a \code{regions_pco_select} object, the output of a call to \code{PCOselect()} with \code{method = "boot"} or \code{"max"}.}

\item{\dots}{ignored.}

\item{object}{a \code{regions_pco_select} object, the output of a call to \code{PCOselect()} with \code{method = "max"}.}
}
\value{
For \code{PCOselect()}, a \code{regions_pco_select} object, which is a numeric vector containing the indices of the chosen PCOs, with attributes containing information about the PCO scores chosen by the specified method. When \code{method = "boot"}, the bootstrap results are stored in the \code{"boot"} attribute. When \code{method = "max"}, the \code{regions_results} object passed to \code{regions} and other information about the quality of fit for each number of PCOs are stored in the \code{"pcomax"} attribute.

The \code{plot()} methods each return a \code{ggplot} object that can manipulated using \pkg{ggplot2} syntax. The \code{summary()} method returns a data.frame of results.
}
\description{
\code{PCOselect()} provides several methods to select the number of principal coordinates (PCOs) analysis scores to be used in subsequent analyses.
}
\details{
Each method provides an alternate way to select the number of scores. These are described below.
\subsection{\code{method = "manual"}:}{

This simply returns the number supplied to \code{scores} after running some checks to ensure it is valid.
}

\subsection{\code{method = "boot"}}{

Bootstrapping works by comparing the eigenvalue distributions of PCOs to those with randomized data in order to extract PCO axes with significant signal, which are defined as those with eigenvalues greater than those from randomized data. The returned PCO cutoff is the largest PCO axis whose eigenvalues fall below the mean eigenvalue for that axis from the randomized data. Data are randomly sampled by row. Bootstrapping is sensitive to unequal variances of columns, so \code{scale = TRUE} should be set in the call to \code{\link[=svdPCO]{svdPCO()}}, which is the default; the data are scaled in the same way prior to bootstrapping. The \code{plot()} method displays the eigenvalues of the true PCOs and boxplots summarizing the distribution of the bootstrapped eigenvalues for each PCO.
}

\subsection{\code{method = "variance"}}{

This method works by computing the ratio of each eigenvalue to the sum of the eigenvalues (i.e., to compute the proportion of variance explained by each PCO score) and select the number of scores with ratios greater than the cutoff value supplied to \code{cutoff}.
}

\subsection{\code{method = "max"}}{

This method works by selecting the smallest number of PCOs that gives a region score within .001 of the maximum possible region score for the segmented models fit in the object supplied to \code{results}. Which criterion is maximized (AIC or BIC) is determined by the value supplied to \code{criterion}. The \code{summary()} method displays the region score (estimated number of regions) for each PCO (\code{RSind}) and for PCOs cumulatively (\code{RScum}) selected using the AICc or BIC as well as the cumulative proportion of variance explained by the PCOs. The \code{plot()} method displays this information graphically, with the left y-axis displaying the region score for the PCOs individually (pale blue triangles) and cumulatively (orange circles) using each of the two criteria, and the right y-axis displaying the cumulative percentage of variance explained by the PCOs.
}
}
\examples{
data("alligator")

alligator_data <- process_measurements(alligator,
                                       pos = "Vertebra")

# Compute PCOs
alligator_PCO <- svdPCO(alligator_data)

# Select which PCOs to use
## Manually (first 4 PCOs)
(PCOs <- PCOselect(alligator_PCO, "manual", scores = 4))

## Using variance cutoff: PCOs that explain 5\% or more
## of total PCO variance
(PCOs <- PCOselect(alligator_PCO, "variance", cutoff = .05))

## Using bootstrapping with 50 reps (more reps should
## be used in practice; default is fine)
(PCOs <- PCOselect(alligator_PCO, "boot", nreps = 50))

plot(PCOs) #plot true eigenvalues against bootstrapped

## Using PCOs that optimize region score:
regionresults <- calcregions(alligator_PCO, scores = 1:21, noregions = 7,
                             minvert = 3, cont = TRUE, exhaus = TRUE,
                             verbose = FALSE)

(PCOs <- PCOselect(alligator_PCO, "max",
                   results = regionresults,
                   criterion = "bic"))

plot(PCOs)

summary(PCOs)
}
