% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DGM_fit.R
\name{DoubleGap}
\alias{DoubleGap}
\title{Fit the Double-Gap Life Expectancy Forecasting Model}
\usage{
DoubleGap(
  DF,
  DM,
  age,
  country,
  years,
  arima.order = NULL,
  drift = NULL,
  tau = NULL,
  A = NULL
)
}
\arguments{
\item{DF}{data.frame containing life expectancy records for females. 
The table must contain the following 4 columns: country, year, age, ex.}

\item{DM}{data.frame containing life expectancy records for males. 
The table must have the same format and dimensions as \code{DF}.}

\item{age}{Indicate the age for which the model to be fitted. 
Assuming \code{DF} and \code{DM} contain records for different ages, this 
argument it is used to subset the data. If you want to fit the model for age 0, 
add age = 0. Type: scalar.}

\item{country}{Indicate for which country you want to fit the model. The country
name or code must exist in \code{DF} and \code{DM}.
Type: character.}

\item{years}{Period of time to be used. Type: numeric vector.}

\item{arima.order}{A specification of the the ARIMA model to be used in 
fitting the best-practice gap. The ARIMA order is country specific.
The three integer components (p, d, q) are the AR order, 
the degree of differencing, and the MA order. Format: numerical vector of length 3.
If \code{arima.order = NULL} the function conducts a search over possible models 
according to AIC. See \code{\link[forecast]{auto.arima}} for details.}

\item{drift}{Indicate whether the ARIMA model should include a linear drift 
term or not. Type: logical value. If \code{drift = NULL}, it will be estimate 
automatically.}

\item{tau}{The level of female life expectancy at which the sex-gap is 
expected to stop widening and to start narrowing. If \code{NULL} 
then the model will run an algorithm to find it.}

\item{A}{The level of female life expectancy where we assume no further 
change in the sex-gap. If \code{NULL} the model will estimate it.}
}
\value{
The output is of class \code{DoubleGap} with the components:
 \item{input}{List with arguments provided in input. Saved for convenience.}
 \item{call}{An unevaluated function call, that is, an unevaluated 
expression which consists of the named function applied to the given arguments.}
 \item{coefficients}{Estimated coefficients.}
 \item{fitted.values}{Fitted values of the selected model.}
 \item{observed.values}{Country specific observed values obtained from input data. 
 This is a \code{data.frame} containing information about female, male and 
 best-practice life expectancy measures over time, together with associated gaps 
 in life expectancy.}
 \item{model.parts}{Object containing detailed results of the fitted model.} 
 \item{residuals}{Deviance residuals.}
}
\description{
Fit a Double-Gap model for forecasting life expectancy.
The method combines separate forecasts to obtain joint male and female 
life expectancies that are coherent with a best-practice trend. See the entire 
description and mathematical formulation of the model in 
\insertCite{pascariu2018;textual}{MortalityGaps}
}
\examples{
# Input data ------------------------------------
# Collection of life expectancies for female populations
exF <- MortalityGaps.data$exF
# Life expectancy for male populations
exM <- MortalityGaps.data$exM

# Example 1 ----------------------------------------------
# Fit DG model at age 0 for Australia using data from 1950 to 2014
M0 <- DoubleGap(DF = exF,
                DM = exM,
                age = 0,
                country = "AUS",
                years = 1950:2014)
M0
summary(M0)
ls(M0)

# Forecast life expectancy in Australia until 2030
P0 <- predict(M0, h = 16)
P0
# Plot the results
plot(P0)

\dontrun{
# Example 2 ----------------------------------------------
# Fit DG model at age 0 for Sweden. Provide details about models.
# Reproduce published results in the article.
M1 <- DoubleGap(DF = exF, 
                DM = exM, 
                age = 0, 
                country = "SWE", 
                years = 1950:2014, 
                arima.order = c(2, 1, 1), 
                drift = TRUE, 
                tau = 75, 
                A = 86)
summary(M1)
# Predict model 
P1 <- predict(M1, h = 36)
plot(P1)

# Example 3 ----------------------------------------------
# Fit DG model for USA at age 65.
M2 <- DoubleGap(DF = exF, 
                DM = exM, 
                age = 65, 
                country = "USA", 
                years = 1950:2014, 
                arima.order = c(0, 1, 0), 
                drift = FALSE, 
                tau = 15, 
                A = 24)
summary(M2)
# Predict model 
P2 <- predict(M2, h = 36)
plot(P2)
}
}
\references{
\insertAllCited{}
}
\seealso{
\code{\link{predict.DoubleGap}}
}
\author{
Marius D. Pascariu
}
