% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MNM.R
\name{MNM}
\alias{MNM}
\title{Fit a Multi-Species N-Mixture Model (MNM) using Nimble}
\usage{
MNM(
  Y = NULL,
  iterations = 60000,
  burnin = 20000,
  thin = 10,
  Xp = NULL,
  Xn = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{Y}{An array of observed count data of dimension (R, T, S), where:
\itemize{
\item \code{R}: Number of sites
\item \code{T}: Number of replicates
\item \code{S}: Number of species
}

This array is typically produced by the \code{simulateData} function.}

\item{iterations}{Integer. Total number of MCMC iterations to run. Default is 60,000.}

\item{burnin}{Integer. Number of initial MCMC iterations to discard as burn-in. Default is 20,000.}

\item{thin}{Integer. Thinning interval for MCMC samples to reduce autocorrelation. Default is 10.}

\item{Xp}{An array of covariates affecting detection probability, with dimensions (R, S, P1), where:
\itemize{
\item \code{R}: Number of sites
\item \code{S}: Number of species
\item \code{P1}: Number of detection-related covariates
}
See examples for implementation details.}

\item{Xn}{An array of covariates affecting abundance, with dimensions (R, S, P2), where:
\itemize{
\item \code{R}: Number of sites
\item \code{S}: Number of species
\item \code{P1}: Number of abundance-related covariates
}
See examples for implementation details.}

\item{verbose}{Control the level of output displayed during function execution. Default is TRUE.}

\item{...}{Additional arguments passed for prior distribution specification. Supported distributions include dnorm, dexp, dgamma, dbeta, dunif, dlnorm, dbern, dpois, dbinom, dcat, dmnorm, dwish, dchisq, dinvgamma, dt, dweib, ddirch, dmulti, dmvt. Default prior distributions are:
\itemize{
\item prior_detection_probability: prior distribution for the detection probability intercept (\code{gamma}). Default is \code{'dnorm(0, 0.001)'}.
\item prior_precision: prior distribution for the precision matrix for the species-level random effect. Default is \code{'dwish(Omega[1:S,1:S], df)'}.
\item prior_mean: prior distribution for the mean of the species-level random effect (\code{mu}). Default is \code{'dnorm(0,0.001)'}.
\item prior_hurdle: prior distribution for \code{theta}, the probability of structural zero in hurdle models. Default is \code{'dbeta(1,1)'}.
\item prior_mean_AR: prior distribution for the mean of the autoregressive random effect (\code{phi}). Default is \code{'dnorm(0,0.001)'}.
\item prior_sd_AR: prior distribution for the standard deviation of the autoregressive random effect (\code{phi}). Default is \code{'dexp(1)'}.
}
See Nimble (r-nimble.org) documentation for distribution details.}
}
\value{
An MNM object that contains the following components:
\itemize{
\item summary: Nimble model summary (mean, standard deviation, standard error, quantiles, effective sample size and Rhat value for all monitored values)
\item n_parameters: Number of parameters in the model (for use in calculating information criteria)
\item data: Observed abundances
\item fitted_Y: Predicted values for Y. Posterior predictive checks can be performed by comparing fitted_Y with the observed data.
\item logLik: Log-likelihood of the observed data (Y) given the model parameters.
\item n_converged: Number of parameters with successful convergence (Rhat < 1.1).
\item plot: traceplots and density plots for all monitored variables.
}
}
\description{
Fits a multi-species N-mixture (MNM) model to observed count data, leveraging Nimble for Bayesian inference. This model accounts for variation in detection probability and abundance across multiple species and sites.
}
\details{
This function takes observed count data and covariates, then fits an MNM model using Nimble. The model estimates species-specific detection probabilities and abundances, allowing for covariate effects. The function also supports posterior predictive checks by comparing observed counts with predicted values.
}
\note{
Ensure that the dimensions of \code{Y}, \code{Xp}, and \code{Xn} match the requirements specified above. Mismatched dimensions will result in errors during model fitting.
}
\examples{
# Example 1:
# Covariates must be of dimension (R, S, P1/P2). If covariates of an alternative dimension are used,
#  they must first be coerced into the right format.
# If we have two abundance-covariates, one site-level covariate and one species-level
# covariate, they may be combined as follows:
R <- 10  # Number of sites
S <- 5   # Number of species
T<-5
Y <- array(sample(0:10, 100, replace = TRUE), dim = c(R, T, S))
covariate_1 <- runif(R)  # Site-level covariate
covariate_2 <- runif(S)  # Species-level covariate

# Expand covariate_1 to have S columns
expanded_covariate_1 <- matrix(rep(covariate_1, S), nrow = R, ncol = S)
# Expand covariate_2 to have R rows
expanded_covariate_2 <- t(matrix(rep(covariate_2, R), nrow = S, ncol = R))
# Combine into an array of dimensions (R, S, 2)
Xn <- array(c(expanded_covariate_1, expanded_covariate_2), dim = c(R, S, 2))
dim(Xn) # this is now in the correct format and can be used.
\donttest{result <- MNM(Y, Xn = Xn)}
# nimble creates auxiliary functions that may be removed after model
# run is complete using rm(list = ls(pattern = "^str"))
\donttest{print(result@summary)}
 data(birds)
# Example 2: North American Breeding Bird Data
# Data must first be reformatted to an array of dimension (R,T,S,K)
R <- 15
T <- 10
S <- 10
K <- 4
# Ensure data is ordered consistently
birds <- birds[order(birds$Route, birds$Year, birds$English_Common_Name), ]
# Create a 4D array with proper dimension
Y <- array(NA, dim = c(R, T, S, K))
# Map route, species, and year to indices
route_idx <- as.numeric(factor(birds$Route))
species_idx <- as.numeric(factor(birds$English_Common_Name))
year_idx <- as.numeric(factor(birds$Year))
# Populate the array
stop_data <- as.matrix(birds[, grep("^Stop", colnames(birds))])
for (i in seq_len(nrow(birds))) {
  Y[route_idx[i], , species_idx[i], year_idx[i]] <- stop_data[i, ]
  }
  # Assign dimnames
  dimnames(Y) <- list(
    Route = sort(unique(birds$Route)),
      Stop = paste0("Stop", 1:T),
        Species = sort(unique(birds$English_Common_Name)),
          Year = sort(unique(birds$Year))
          )
# Selecting only 5 bird species and 1 year for analysis:
Y<-Y[,,1:5,1]
\donttest{model<-MNM_fit(Y=Y, AR=FALSE, Hurdle=FALSE, iterations=5000, burnin=1000)}
}
\references{
\itemize{
\item Royle, J. A. (2004). N-mixture models for estimating population size from spatially replicated counts. Biometrics, 60(1), 108-115.
\item Mimnagh, N., Parnell, A., Prado, E., & Moral, R. D. A. (2022). Bayesian multi-species N-mixture models for unmarked animal communities. Environmental and Ecological Statistics, 29(4), 755-778.
}
}
\seealso{
\code{simulateData}: For generating example datasets compatible with this function.
}
