% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/est_quant_ipwe.R
\name{est_quant_ipwe}
\alias{est_quant_ipwe}
\title{Estimate the marginal quantile response of a linear static treatment regime}
\usage{
est_quant_ipwe(beta, sign_beta1, x, censor_y, delta, epsi, a, tau,
  check_complete = TRUE, Penalty.level = 0)
}
\arguments{
\item{beta}{Numerical vector. Exclude the coefficient for the first nontrivial covariate. So if
there are \code{k} covariates, the length of \code{beta} should equal \code{k+1-1=k} because
the intercept needs one coefficient as well.}

\item{sign_beta1}{logical. FALSE if the coefficient for the first continuous variable 
is fixed to be negative one; TRUE if positive one.}

\item{x}{Numeric Matrix. The baseline covariates from all observed data.}

\item{censor_y}{Numeric vector. The censored survival times from all observed data, i.e. \code{censor_y = min(Y, C)}}

\item{delta}{Numeric vector. The censoring indicators from all observed data. We use 1 for uncensored, 0 for censored.}

\item{epsi}{the product of (1) the probability of being assigned the 
observed treatment level through the original treatment assignment mechanism 
and (2) the conditional survival probability of the censoring variable at \code{censor_y}.}

\item{a}{Numeric vector. The vector of observed treatment level for all observed data. Treatment levels
should be coded as 0/1.}

\item{tau}{a value between 0 and 1. This is the quantile of interest.}

\item{check_complete}{logical. Since this value estimation method is purely
nonparametric, we need at least one unit in collected data such that the observed
treatment assignment is the same what the regime parameter suggests. If \code{check_complete}
is \code{TRUE}. It will check if any observation satisfies this criterion. 
When none observation satisfies, a message is printed to console to raise users
awareness that the input regime parameter \code{beta} does not agree with any observed treatment level assignment.
Then a sufficiently small number is returned from this function, to keep
the genetic algorithm running smoothly.}

\item{Penalty.level}{the level that determines which objective function to use. 
\code{Penalty.level = 0} indicates no regularization;
\code{Penalty.level = 1} indicates the value function estimation minus the means absolute average coefficient 
is the output, which is useful trick to achieve uniqueness of estimated optimal TR
when resolution of input response is low.}
}
\description{
Assume we have binary treatment options for each subject in the target population.
This function evaluates a given treatment regime by the estimated 
marginal mean response. 
We assume the space of treatment regimes are linear
decision functions indexed by parametric coefficients.

This R function is an empirical \emph{value function} in the 
literature of optimal treatment regime estimation. Since the goal here
is to maximize population's \strong{marginal quantile}, this function, which estimates the perforamce
of a set of parameters in terms of \strong{marginal quantile},
is the objective function in a nonparametric policy-search method.

The user facing application which utilizes this function is \code{\link{IPWE_Qopt_IndCen}}.
}
\examples{
GenerateData <- function(n)
{
  x1 <- runif(n, min=-0.5,max=0.5)
  x2 <- runif(n, min=-0.5,max=0.5)
  error <- rnorm(n, sd= 1)
  ph <- rep(0.5,n)
  a <- rbinom(n = n, size = 1, prob=ph)
  c <- 1.5 +  + runif(n = n, min=0, max=2)
  cmplt_y <-  pmin(2+x1+x2 +  a*(1 - x1 - x2) +  (0.2 + a*(1+x1+x2)) * error, 4.4)
  censor_y <- pmin(cmplt_y, c)
  delta <- as.numeric(c > cmplt_y)
  return(data.frame(x1=x1,x2=x2,a=a, censor_y = censor_y, delta=delta))
}
n <- 100
data <- GenerateData(n)

# here the value for argument epsi uses 0.5 vector for brevity in notation.
quant_hat <- est_quant_ipwe(beta=c(-1,2), sign_beta1=TRUE, x=cbind(1, data$x1, data$x2), 
                            censor_y = data$censor_y, delta = data$delta, tau=0.5,
                            epsi = rep(0.5,n), a = data$a)  

}
