% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pf_optimize.R
\name{rsdc_minvar}
\alias{rsdc_minvar}
\title{Minimum-Variance Portfolio (Rolling Weights)}
\usage{
rsdc_minvar(sigma_matrix, value_cols, predicted_corr, y, long_only = TRUE)
}
\arguments{
\item{sigma_matrix}{Numeric matrix \eqn{T \times K} of forecasted \emph{volatilities}
(standard deviations), one column per asset.}

\item{value_cols}{Character or integer vector giving the columns in \code{sigma_matrix}
to use as assets (order defines the asset order).}

\item{predicted_corr}{Numeric matrix \eqn{T \times P} of pairwise correlations, where
\eqn{P = \binom{K}{2}} and the columns correspond to \code{combn(K, 2)} order.}

\item{y}{Numeric matrix \eqn{T \times K} of asset returns aligned with \code{sigma_matrix}.
Used only to compute the realized portfolio volatility.}

\item{long_only}{Logical. If \code{TRUE} (default), imposes long-only MV with the full-investment
constraint \eqn{\sum_i w_i = 1} and \eqn{w_i \ge 0}. If \code{FALSE}, solves unconstrained MV
with only \eqn{\sum_i w_i = 1}.}
}
\value{
An object of class \code{"minvar_portfolio"}:
\describe{
\item{weights}{\eqn{T \times K} matrix of MV weights (one row per time).}
\item{cov_matrices}{List of length \eqn{T} with the per-period \eqn{K \times K} covariance matrices.}
\item{volatility}{Realized standard deviation of portfolio returns (see Note on units).}
\item{y}{The input \code{y} matrix (coerced to \eqn{T \times K}).}
\item{K}{Number of assets.}
}
}
\description{
Computes rolling minimum-variance (MV) portfolio weights from a sequence of
per-period covariance matrices implied by forecasted volatilities and
pairwise correlations. Supports long-only or unconstrained MV. If the QP
solver fails at a time step, the routine falls back to equal weights.
}
\details{
\itemize{
\item \strong{Covariance build:} For each \eqn{t}, a correlation matrix \eqn{R_t}
is reconstructed ... Let \eqn{D_t = \mathrm{diag}(\sigma_{t,1},\dots,\sigma_{t,K})}
and \eqn{\Sigma_t = D_t R_t D_t}.
\item \strong{Optimization:} Minimize \eqn{\tfrac{1}{2} w^\top \Sigma_t w} subject to
\eqn{\mathbf{1}^\top w = 1} and, if \code{long_only}, \eqn{w \ge 0}
(solved with \code{quadprog::solve.QP}).
\item \strong{Failure handling:} If the QP fails at time \(t\), weights default to equal
allocation \(w_i = 1/K\).
}
}
\section{Note on units}{

The realized portfolio return at time \(t\) is computed as \code{sum(y[t, ] * weights[t, ]) / 100}.
This assumes \code{y} is expressed in \\% (percentage) units. If your \code{y} is already in decimals,
remove the \code{/ 100} in the implementation or convert inputs accordingly.
}

\examples{
# Toy example with K = 3
T <- 50; K <- 3
set.seed(42)
vols <- matrix(0.2 + 0.05*abs(sin(seq_len(T)/7)), T, K)
colnames(vols) <- paste0("A", 1:K)
# simple, stationary correlations
pred_corr <- cbind(rep(0.20, T), rep(0.10, T), rep(0.05, T))  # order: (2,1), (3,1), (3,2)
rets <- matrix(rnorm(T*K, sd = 0.01), T, K); colnames(rets) <- colnames(vols)

mv <- rsdc_minvar(sigma_matrix  = vols,
                  value_cols    = colnames(vols),
                  predicted_corr= pred_corr,
                  y             = rets,
                  long_only     = TRUE)
head(mv$weights)
mv$volatility

}
\seealso{
\code{\link{rsdc_maxdiv}} (maximum diversification),
\code{\link[quadprog]{solve.QP}}
}
