% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lipid_screen.R
\docType{data}
\name{lipid_screen}
\alias{lipid_screen}
\title{Lipid formulation screening data}
\format{
A data frame with N rows and the following columns:
\describe{
\item{RunID}{character. Optional identifier.}
\item{PEG}{numeric. Proportion (0–1).}
\item{Helper}{numeric. Proportion (0–1).}
\item{Ionizable}{numeric. Proportion (0–1).}
\item{Cholesterol}{numeric. Proportion (0–1).}
\item{Ionizable_Lipid_Type}{factor.}
\item{N_P_ratio}{numeric.}
\item{flow_rate}{numeric.}
\item{Potency}{numeric. Response.}
\item{Size}{numeric. Response (e.g., nm).}
\item{PDI}{numeric. Response (polydispersity index).}
\item{Notes}{character. Optional free-text notes.}
}
}
\source{
Simulated screening table supplied by the package author.
}
\usage{
data(lipid_screen)
}
\description{
An example dataset for modeling Potency, Size, and PDI as functions of
formulation and process settings. Percent composition columns are stored as
proportions in \code{[0, 1]} (e.g., 4.19\\% is 0.0419). This table is intended
for demonstration of SVEMnet multi-response modeling and desirability-based
random-search optimization.
}
\details{
This dataset accompanies examples showing:
\itemize{
\item fitting three SVEM models (Potency, Size, PDI) on a shared expanded
factor space via \code{bigexp_terms()} and \code{bigexp_formula()},
\item random design generation using SVEM random-table helpers (for use
with multi-response optimization),
\item multi-response optimization with \code{svem_optimize_random()} using
Derringer–Suich desirabilities and weighted combining
(\code{combine = "geom"} or \code{combine = "mean"}),
\item returning both high-score \emph{optimal} candidates and
high-uncertainty \emph{exploration} candidates,
\item optional whole-model reweighting (WMT) of response weights via
p-values, and exporting per-row scores to the original data with
\code{original_data_scored}.
}
}
\section{Acknowledgments}{

OpenAI's GPT models (o1-preview and GPT-5 Thinking via ChatGPT) were
used to assist with coding and roxygen documentation; all
content was reviewed and finalized by the author.
}

\examples{
\donttest{
# 1) Load the bundled dataset
data(lipid_screen)
str(lipid_screen)

# 2) Build a deterministic expansion using bigexp_terms()
#    Provide main effects only on the RHS; expansion width is controlled via arguments.
spec <- bigexp_terms(
  Potency ~ PEG + Helper + Ionizable + Cholesterol +
    Ionizable_Lipid_Type + N_P_ratio + flow_rate,
  data             = lipid_screen,
  factorial_order  = 3,   # up to 3-way interactions
  polynomial_order = 3,   # include up to cubic terms I(X^2), I(X^3),
  include_pc_2way  = TRUE, # include partial cubic terms
  include_pc_3way  = FALSE
)

# 3) Reuse the same locked expansion for other responses
form_pot <- bigexp_formula(spec, "Potency")
form_siz <- bigexp_formula(spec, "Size")
form_pdi <- bigexp_formula(spec, "PDI")

# 4) Fit SVEM models with the shared factor space/expansion
set.seed(1)
fit_pot <- SVEMnet(form_pot, lipid_screen)
fit_siz <- SVEMnet(form_siz, lipid_screen)
fit_pdi <- SVEMnet(form_pdi, lipid_screen)

# 5) Collect models in a named list by response
objs <- list(Potency = fit_pot, Size = fit_siz, PDI = fit_pdi)

# 6) Define multi-response goals and weights (DS desirabilities under the hood)
#    Maximize Potency (0.6), minimize Size (0.3), minimize PDI (0.1)
goals <- list(
  Potency = list(goal = "max", weight = 0.6),
  Size    = list(goal = "min", weight = 0.3),
  PDI     = list(goal = "min", weight = 0.1)
)

# Mixture constraints: components sum to 1, with bounds
mix <- list(list(
  vars  = c("PEG", "Helper", "Ionizable", "Cholesterol"),
  lower = c(0.01, 0.10, 0.10, 0.10),
  upper = c(0.05, 0.60, 0.60, 0.60),
  total = 1.0
))

# 7) Run random-search optimization (DS + optimal & exploration candidates)
set.seed(2)
opt_out <- svem_optimize_random(
  objects                  = objs,
  goals                    = goals,
  n                        = 25000,
  mixture_groups           = mix,
  level                    = 0.95,
  k_candidates             = 5,
  top_frac                 = 0.02,
  k_exploration_candidates = 5,
  exploration_top_frac     = 0.05,
  numeric_sampler          = "random",
  verbose                  = TRUE
)

# Inspect optimal solution and candidates (scores and uncertainty included)
opt_out$best
opt_out$best_pred
opt_out$best_ci
head(opt_out$candidates)

# Inspect exploration target and candidates
opt_out$exploration_best
head(opt_out$exploration_candidates)

# 8) Repeat with WMT reweighting using the original data (requires 'data')
#    Choose either "neglog10" (aggressive) or "one_minus_p" (conservative).
set.seed(3)
opt_wmt <- svem_optimize_random(
  objects                  = objs,
  goals                    = goals,
  data                     = lipid_screen,  # used for WMT and original_data_scored
  n                        = 25000,
  mixture_groups           = mix,
  level                    = 0.95,
  k_candidates             = 5,
  top_frac                 = 0.02,
  k_exploration_candidates = 5,
  exploration_top_frac     = 0.05,
  numeric_sampler          = "random",
  reweight_by_wmt          = TRUE,
  wmt_transform            = "neglog10",
  verbose                  = TRUE
)

# Compare weights and look at candidates under WMT
opt_wmt$weights_original
opt_wmt$weights_final
opt_wmt$wmt_p_values
head(opt_wmt$candidates)
head(opt_wmt$exploration_candidates)

# Scored original data (predictions, desirabilities, score, uncertainty)
head(opt_wmt$original_data_scored)
}
}
\references{
Gotwalt, C., & Ramsey, P. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849873/redirect_from_archived_page/true}

Karl, A. T. (2024). A randomized permutation whole-model test heuristic for Self-Validated Ensemble Models (SVEM). \emph{Chemometrics and Intelligent Laboratory Systems}, \emph{249}, 105122. \doi{10.1016/j.chemolab.2024.105122}

Karl, A., Wisnowski, J., & Rushing, H. (2022). JMP Pro 17 Remedies for Practical Struggles with Mixture Experiments. JMP Discovery Conference. \doi{10.13140/RG.2.2.34598.40003/1}

Lemkus, T., Gotwalt, C., Ramsey, P., & Weese, M. L. (2021). Self-Validated Ensemble Models for Design of Experiments. \emph{Chemometrics and Intelligent Laboratory Systems}, 219, 104439. \doi{10.1016/j.chemolab.2021.104439}

Xu, L., Gotwalt, C., Hong, Y., King, C. B., & Meeker, W. Q. (2020). Applications of the Fractional-Random-Weight Bootstrap. \emph{The American Statistician}, 74(4), 345–358. \doi{10.1080/00031305.2020.1731599}

Ramsey, P., Gaudard, M., & Levin, W. (2021). Accelerating Innovation with Space Filling Mixture Designs, Neural Networks and SVEM. \emph{JMP Discovery Conference}. \url{https://community.jmp.com/t5/Abstracts/Accelerating-Innovation-with-Space-Filling-Mixture-Designs/ev-p/756841}

Ramsey, P., & Gotwalt, C. (2018). Model Validation Strategies for Designed Experiments Using Bootstrapping Techniques With Applications to Biopharmaceuticals. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/Model-Validation-Strategies-for-Designed-Experiments-Using/ev-p/849647/redirect_from_archived_page/true}

Ramsey, P., Levin, W., Lemkus, T., & Gotwalt, C. (2021). SVEM: A Paradigm Shift in Design and Analysis of Experiments. \emph{JMP Discovery Conference - Europe}. \url{https://community.jmp.com/t5/Abstracts/SVEM-A-Paradigm-Shift-in-Design-and-Analysis-of-Experiments-2021/ev-p/756634}

Ramsey, P., & McNeill, P. (2023). CMC, SVEM, Neural Networks, DOE, and Complexity: It's All About Prediction. \emph{JMP Discovery Conference}.

Friedman, J. H., Hastie, T., and Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1-22.

Meinshausen, N. (2007).
Relaxed Lasso. Computational Statistics & Data Analysis, 52(1), 374-393.
}
\keyword{datasets}
