#' @title Additive check
#' @description This function checks if the given game is additive.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game defined by \code{v} is additive, \code{FALSE} otherwise. If \code{instance=TRUE} and the game is not additive, the position (binary order position if \code{binary=TRUE}; lexicographic order position otherwise) of a coalition for which additivity is violated is also returned.
#' @details A game \eqn{v\in G^N} is additive if \eqn{v(S)=\sum_{i\in S}v(i)} for all \eqn{S\in 2^N}.
#' @examples
#' v <- c(1, 5, 40, 100, 6, 41, 101, 45, 105, 140, 46, 106, 141, 145, 146)
#' additivecheck(v)
#' additivecheck(v, binary = TRUE, instance = TRUE)
#' @seealso \link{additivegame}, \link{superadditivecheck}
#' @export

additivecheck <- function(v, binary = FALSE, instance = FALSE) {

  ################################
  ### Comprobación datos entrada###
  ################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
  }

  ################################
  ##### Creación de variables######
  ################################

  checkR <- TRUE # Por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # Por defecto, no hay contraejemplo
  S <- 1 # Primera coalición
  tol <- 100*.Machine$double.eps

  ################################
  ###### Cuerpo de la función######
  ################################

  while (checkR & S <= 2^n - 1) # Mientras no haya encontrado un contraejemplo y no haya superado el número máximo de coaliciones, hago la comprobación.
  {
    # Calculamos los jugadores pertenecientes a cada coalición y su valor en v.
    num <- S
    J <- floor(log2(num)) + 1 # Jugador más grande
    Sjug <- c()
    for (ii in 1:J) # Algoritmo de paso de un número en decimal a binario.
    {
      if (num %% 2 == 1) {
        Sjug <- c(Sjug, ii)
      }
      num <- num %/% 2
    }
    vi <- sum(v[2.^(Sjug - 1)]) # Suma de los v(i).
    if (abs(v[S] - vi) > tol) {
      checkR <- FALSE # Si v(S) es distinto que la suma de los v(i), el juego no es aditivo.
      exampleR <- S # Código numérico de una coalición donde falla la condición de aditividad.
    }
    S <- S + 1
  }

  ################################
  ###### Salidas de la función#####
  ################################

  if (instance == FALSE) { # Si el usuario sólo pide una salida, es decir, instance=FALSE.
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también la posición de la coalición S.
    if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, devolvemos la posición lexicográfica de una coalición que falla.
      return(list(check = checkR, example = codebin2lex(n, exampleR)))
    } else { # Si el juego se introdujo en binario, devolvemos la posición binaria de una coalición que falla.
      return(list(check = checkR, example = exampleR))
    }
  }
} # Fin de la función
