#' @importFrom rcdd makeH
#' @importFrom rcdd scdd

#' @title Core vertices
#' @description Given a game, this function computes its core vertices.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @return If the core of \code{v} is non-empty, the core vertices are returned, as a matrix in which each row is a vertex.
#' @details The core of a game \eqn{v\in G^N} is the set of all its stable imputations:
#' \deqn{C(v)=\{x\in\mathbb{R}^n : x(N)=v(N), x(S)\ge v(S)\ \forall S \in 2^N\},}
#' where \eqn{x(S)=\sum_{i\in S} x_i}.
#' @note Function \code{corevertices234} can also compute the core vertices of games with less than five players, but takes a different approach.
#' @examples
#' v=c(0,0,0,0,0,0,0,0,1,4,1,3,6,8,10)
#' corevertices(v)
#'
#' # What if the game is a cost game?
#' cost.v <- c(2,2,2,3,4,4,5) # cost game
#' -corevertices(-cost.v) # core vertices of the cost game
#' @references Edgeworth, F. Y. (1881). \emph{Mathematical psychics: An essay on the application of mathematics to the moral sciences}. CK Paul.
#' @references Gillies, D. (1953). \emph{Some theorems on n-person games}. PhD thesis, Princeton, University Press Princeton, New Jersey.
#' @seealso \link{balancedcheck}, \link{corevertices234}, \link{plotcoreset}, \link{plotcoresets}
#' @export

corevertices <- function(v, binary = FALSE) {

  ################################
  ###Comprobación datos entrada###
  ################################

  nC <- length(v)
  n <- log(nC+1)/log(2)

  if(n>floor(n))
  {
    stop('In corevertices(v), vector v must have length 2^n-1 for some n.',call.=F)
  }

  # Antes de nada, comprobamos si el juego es equilibrado:
  if (!balancedcheck(v,binary=binary)) {
    message("The core of 'v' is empty.")
    return(NULL)
  }

  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    a <- lex2bin(v) # lo llamamos 'a' por coherencia con lo que viene después
  } else {
    a <- v
  }

  ################################
  #####Creación de variables######
  ################################

  #Creamos la matriz de base binaria para resolver el simplex
  Base=matrix(0,2^n-2,n)
  for (ii in 1:2^n-2){
    Base[ii,] <- -as.numeric(intToBits(ii)[1:n])
  }
  #Vector de la gran coalición:
  Base2 <- t(as.matrix(rep(-1,n)))

  ################################
  ######Cuerpo de la función######
  ################################

  #Vamos a utilizar el paquete rcdd.
  #Montamos el problema de programación lineal como lo piden en el paquete:
  #Primero la matriz con la parte izquierda de las desigualdades.
  #Después, la parte derecha de las desigualdades.
  #Por último, la igualdad, primero la parte izquierda y después la derecha.
  ### ÁdPS(26/05/2025): Comento la próxima línea para que no se ejecute.
  # a <-lex2bin(v) # tenemos que pasar el juego a binario (faltaba!!!!)
  ### ÁdPS(26/05/2025): Puse el lex2bin más arriba, no siempre hace falta.
  hRepresentation <- makeH(Base,-a[1:nC-1],Base2,-v[nC])
  #La función scdd devuelve las direcciones y los vértices que definen
  #la envoltura convexa del problema de programación lineal que definimos antes.
  vRepresentation <- scdd(hRepresentation)

  #Transformamos la salida en una matriz. La salida anterior devuelve las
  #direcciones y la envoltura convexa. Nos interesa solo la envoltura convexa.
  vertices <- vRepresentation[[1]][,3:(n+2)]
  # ÁdPS (26/05/2025): Hasta ahora, si solo hay un vértice, se devolvía
  # ese vértice como vector. Por coherencia ahora se convierte ese vector
  # a matriz, para que la salida siempre sea del mismo tipo.
  if (length(vertices) == n) { # si solo hay un vértice...
    vertices <- matrix(vertices,nrow=1) # ponemos el vector como matriz
  }
  vertices <- unique(vertices) # borramos vértices repetidos

  ################################
  ######Salidas de la función#####
  ################################

  return(vertices)
}
