#' @title Marginal game
#' @description Given a game and a coalition, this function returns the characteristic function of the corresponding marginal game.
#' @param v Characteristic function, as a vector.
#' @param S The position of a coalition, as an integer.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} and \code{S} are introduced according to binary order instead of lexicographic order.
#' @return The characteristic function of the \code{S}-marginal game, as a vector in binary order if \code{binary=TRUE} and in lexicographic order otherwise.
#' @details Given a game \eqn{v\in G^N} and a coalition \eqn{S\in 2^N}, the S-marginal game, \eqn{v^S\in G^N},
#' is defined by \deqn{v^S(R)=v(R\cup (N\backslash S))-v(N\backslash S)+v(R\cap (N\backslash S))\text{ for all }R\in 2^N.}
#' @examples
#' v <- c(0, 0, 0, 2, 3, 10, 20)
#' marginalgame(v, 5, binary = TRUE) # coalition {1,3}
#' n <- 3
#' for (i in 1:(2^n - 1)) {
#'   cat("[", i, "]", paste(marginalgame(lex2bin(v),codebin2lex(n,i),binary=TRUE)),"\n")
#' }
#' for (i in 1:(2^n - 1)) {
#'   cat("[", i, "]", paste(marginalgame(v,i)),"\n")
#' }
#' @references Sánchez Rodríguez, E., Mirás Calvo, M.A., Quinteiro Sandomingo, C., & Núñez Lugilde, I. (2024). Coalition-weighted Shapley values. International Journal of Game Theory 53, 547-577.
#' @export

marginalgame <- function(v, S, binary = FALSE) { # Principio de la función

  ##################################
  ### Comprobación datos entrada ###
  ##################################

  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  if (S > nC) {
    stop("'S' must not be greater than the length of 'v'.")
  }
  if (binary == FALSE) { # Si el juego se introdujo en lexicográfico, lo pasamos a binario.
    v <- lex2bin(v)
    S <- codelex2bin(n, S)
  }

  ################################
  ##### Creación de variables######
  ################################

  vS <- c() # Marginal game

  ################################
  ###### Cuerpo de la función######
  ################################
  if (S == nC) { # Si S=N, el juego marginal coincide con el original.
    vS <- v
  } else {
    v <- c(0, v) # Añadimos el valor de la coalición vacia.
    for (R in 1:nC)
    {
      vS[R] <- v[bitwOr(R, nC - S) + 1] - v[nC - S + 1] + v[bitwAnd(R, nC - S) + 1] # vS(R)=v(R U (N\S))-v(N\S)+v(R^(N\S))
    }
  }


  ################################
  ###### Salidas de la función#####
  ################################

  if (binary == FALSE) { # Devolvemos el juego aditivo en orden lexicográfico.
    return(bin2lex(vS))
  } else { # Devolvemos el juego aditivo en orden binario.
    return(vS)
  }
} # Fin de la función
