#' @title 0-normalized check
#' @description This function checks if the given game is 0-normalized.
#' @param v A characteristic function, as a vector.
#' @param binary A logical value. By default, \code{binary=FALSE}. Should be set to \code{TRUE} if \code{v} is introduced in binary order instead of lexicographic order.
#' @param instance A logical value. By default, \code{instance=FALSE}.
#' @return \code{TRUE} if the game is 0-normalized, \code{FALSE} otherwise. If \code{instance=TRUE} and the game is not 0-normalized, the function also returns a player for whose value is not zero.
#' @details A game \eqn{v\in G^N} is 0-normalized if \eqn{v(i)=0} for all \eqn{i\in N}.
#' @examples
#' v <- c(rep(0, 4), 1, rep(30, 20), rep(3, 5), 50) # v(5)=1
#' zeronormalizedcheck(v, binary = FALSE, instance = TRUE)
#' @seealso \link{normalizedgame}, \link{strategicallyequivalentcheck}, \link{zeromonotoniccheck}, \link{zeronormalizedgame}
#' @export

zeronormalizedcheck <- function(v, binary = FALSE, instance = FALSE) { # Principio de la función

  # Comprobaciones iniciales.
  nC <- length(v) # Número de coaliciones.
  n <- log(nC + 1) / log(2) # Número de jugadores.
  if (n > floor(n)) {
    stop("'v' must have length 2^n-1 for some n.")
  }

  # Definimos variables que necesitamos.
  checkR <- TRUE # por defecto, mientras no encuentre un contraejemplo, checkR=TRUE
  exampleR <- NULL # por defecto, no hay contraejemplo
  jug <- 1 # primer jugador
  tol <- 100*.Machine$double.eps # tolerancia

  # Ahora vienen los cálculos.
  # Si el juego se introduce en lexicográfico, comprobamos directamente en lexicográfico.
  # Si el juego se introduce en binario, comprobamos directamente en binario.

  if (binary == FALSE) {
    while (checkR & jug <= n) {
      if (abs(v[jug]) > tol) # comprobamos para cada jugador si v(i)=0.
      {
        checkR <- FALSE # si v(i) es distinto de 0, el juego no es 0-normalizado
        exampleR <- jug # código numérico del jugador donde falla la condición
      }
      jug <- jug + 1
    }
  } else {
    while (checkR & jug <= n) {
      if (abs(v[2^(jug - 1)]) > tol) # comprobamos para cada jugador si v(i)=0
      {
        checkR <- FALSE # si v(i) es distinto de 0, el juego no es 0-normalizado
        exampleR <- jug # código numérico del jugador donde falla la condición
      }
      jug <- jug + 1
    }
  }

  # Salidas de la función.
  if (instance == FALSE) { # Si el usuario sólo pide el check (instance=FALSE)
    return(check = checkR)
  } else { # Si instance=TRUE, la función devuelve también el número de un jugador con v(i)!=0.
    return(list(check = checkR, example = exampleR))
  }

} # Fin de la función
