% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/input_f.R
\name{shared_input}
\alias{shared_input}
\title{Shared input object}
\usage{
shared_input(expr, constrained = NULL)
}
\arguments{
\item{expr}{A value or expression to initialize the shared input with.
The expression is evaluated immediately.}

\item{constrained}{Logical. If \code{TRUE}, creates a shared environment-backed
object. If \code{FALSE}, creates an immutable copy-on-modify object.
If \code{NULL} (default), the function looks up \code{constrained} in the calling
environment; only an explicit \code{TRUE} enables shared mode.}
}
\value{
An object of class \code{shared_input_val} (immutable mode) or
\code{shared_input_env} (shared mode), both inheriting from class
\code{"shared_input"}. Each instance exposes the following user methods:

\describe{
\item{$value()}{Returns the current stored value.}
\item{$modify(new_v)}{
In immutable mode: returns a new independent wrapper with updated value.
In shared mode: updates the shared value by reference and returns a new
wrapper pointing to the same shared state.
}
\item{$clone()}{Returns a deep copy (independent wrapper and independent
internal state). Subsequent modifications on clones do not affect the
original object or its aliases.}
\item{$reset()}{Returns a new wrapper whose value is restored to the
original initialization value. In both modes this creates an independent
fresh state.}
\item{$fork(n)}{Creates \code{n} independent deep clones as a list. Useful for
generating multiple isolated copies quickly.}
}
}
\description{
Constructor for a lightweight "shared or immutable" value holder.
}
\details{
\code{shared_input()} produces a simple object that wraps a value with
controlled mutability semantics. It can operate in two distinct modes:
\itemize{
\item \strong{Immutable (non-shared)}: every modification produces a fresh, independent
copy of the object (safe for parallel or functional code).
\item \strong{Shared (constrained)}: the object’s value is stored in a common
environment shared across all aliases (by-reference semantics). This
allows coordinated updates across multiple handles.
}

The mode is determined either by the explicit argument \code{constrained}, or
by inheriting the value of a \code{constrained} variable in the parent frame.
\itemize{
\item In \strong{immutable mode}, each wrapper stores its value in closures
(\code{make_val()}) and is fully copy-on-modify. No references are shared.
\item In \strong{shared mode}, all wrappers produced by \verb{$modify()} or direct aliasing
point to the same underlying environment (\code{state}). This means updating one
updates all aliases until a \verb{$clone()} or \verb{$reset()} breaks the link.
}

The underlying \code{state} environments are internal. Users should rely only on
the public methods above.

Note: if the stored value itself is a reference type (e.g., environment,
external pointer, R6 object), those internal references remain shared
regardless of mode, following normal R semantics.
}
\examples{
# --- Immutable (default) mode ---
a <- shared_input(5)
a$value()                 # 5
a2 <- a$modify(a$value() + 7)
a$value()                 # 5
a2$value()                # 12

# Cloning and resetting
a3 <- a2$clone()
a4 <- a2$reset()
a3$value(); a4$value()    # 12, 5

# Forking
forks <- a$fork(3)
vapply(forks, function(x) x$value(), numeric(1))

# --- Shared (constrained) mode ---
constrained <- TRUE
b1 <- shared_input(10)
b2 <- b1        # alias (same state)
b1$modify(11)
b1$value(); b2$value()  # both 11

b3 <- b1$clone()
b1$modify(99)
b1$value(); b3$value()  # 99, 11

# Reset breaks sharing
b4 <- b1$reset()
b4$value()              # 10

}
