% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detect-cp.R
\name{detect_cp}
\alias{detect_cp}
\title{Detect change point}
\usage{
detect_cp(
  method,
  cpi,
  psi,
  xi = NULL,
  xi_c = NULL,
  xi_s = NULL,
  x = NULL,
  y = NULL,
  interval = c(-4, 4)
)
}
\arguments{
\item{method}{The change point analysis statistic(s) to compute. Options for
score-based statistics are:
\itemize{
\item \code{"L_S_*"} for the likelihood ratio test-based statistic (Shao et al.,
2016; Sinharay, 2016; Tu et al., 2023).
\item \code{"S_S_*"} for the score test-based statistic (Sinharay, 2016; Tu et al.,
2023).
\item \code{"W_S_*"} for the Wald test-based statistic (Sinharay, 2016; Tu et al.,
2023).
}

Options for response time-based statistics are:
\itemize{
\item \code{"L_T_*"} for the likelihood ratio test-based statistic (Cheng & Shao,
2022).
\item \code{"W_T_*"} for the Wald test-based statistic (Cheng & Shao, 2022).
}

Statistics ending in \code{"*"} can be computed in different ways. Options are:
\itemize{
\item \code{"*"} for all possible ways.
\item \code{"MAX2"} for the maximum of all two-sided statistics in the change point
interval.
\item \code{"MAX1"} for the maximum of all one-sided statistics in the change point
interval.
\item \code{"MIN1"} for the minimum of all one-sided statistics in the change point
interval.
}}

\item{cpi}{The interval to search for the change point. The lower endpoint
must be greater than or equal to 1 and the upper endpoint must be less than
the number of items in the test.}

\item{psi}{A matrix of item parameters. Rows correspond to items and columns
to parameters.}

\item{xi}{A matrix of person parameters. Rows correspond to persons and
columns to parameters. If \code{NULL} (default), person parameters are estimated
using maximum likelihood estimation.}

\item{xi_c, xi_s}{Arrays of person parameters. The first dimension corresponds
to persons, the second dimension to parameters, and the third dimension to
change point locations. \code{xi_c} is based on the items before or at the
change point and \code{xi_s} is based on the items after the change point. If
\code{NULL} (default), person parameters are estimated using maximum likelihood
estimation.}

\item{x, y}{Matrices of raw data. Rows correspond to persons and columns to
items. \code{x} is for the item scores and \code{y} the item log response times.}

\item{interval}{The interval to search for the person parameters. Default is
\code{c(-4, 4)}.}
}
\value{
A list is returned with the following elements:
\item{stat}{A matrix of change point analysis statistics.}
\item{cp}{A matrix of estimated change points.}
}
\description{
Detect a single change point.
}
\examples{
# Setup for Example ---------------------------------------------------------

# Settings
set.seed(0)     # seed for reproducibility
N <- 50         # number of persons
n <- 40         # number of items

# Randomly select 10\% speeded examinees
cv <- sample(1:N, size = N * 0.10)

# Assign change point corresponding to 10\% speeded items
cp <- n * 0.90
ci <- (cp + 1):n

# Create vector of indicators (1 = speeded, 0 = non-speeded)
ind <- ifelse(1:N \%in\% cv, 1, 0)

# Example: Item Scores and Response Times -----------------------------------

# Generate person parameters for the 3PL model and lognormal model
xi <- MASS::mvrnorm(
  N,
  mu = c(theta = 0.00, tau = 0.00),
  Sigma = matrix(c(1.00, 0.25, 0.25, 0.25), ncol = 2)
)

# Generate item parameters for the 3PL model and lognormal model
psi <- cbind(
  a = rlnorm(n, meanlog = 0.00, sdlog = 0.25),
  b = NA,
  c = runif(n, min = 0.05, max = 0.30),
  alpha = runif(n, min = 1.50, max = 2.50),
  beta = NA
)

# Generate positively correlated difficulty and time intensity parameters
psi[, c("b", "beta")] <- MASS::mvrnorm(
  n,
  mu = c(b = 0.00, beta = 3.50),
  Sigma = matrix(c(1.00, 0.20, 0.20, 0.15), ncol = 2)
)

# Simulate uncontaminated data
dat <- sim(psi, xi)
x <- dat$x
y <- dat$y

# Modify contaminated data by changing the item scores and response times
x[cv, ci] <- rbinom(length(cv) * length(ci), size = 1, prob = 0.25)
y[cv, ci] <- runif(length(cv) * length(ci), min = log(1), max = log(10))

# Detect change point
out <- detect_cp(
  method = c("L_S_MAX1", "L_T_MIN1"),
  cpi = c(1, n - 1),
  psi = psi,
  x = x,
  y = y
)
}
\references{
Cheng, Y., & Shao, C. (2022). Application of change point analysis of
response time data to detect test speededness. \emph{Educational and Psychological
Measurement}, \emph{82}(5), 1031--1062.

Shao, C., Li, J., & Cheng, Y. (2016). Detection of test speededness using
change-point analysis. \emph{Psychometrika}, \emph{81}(4), 1118--1141.

Sinharay, S. (2016). Person fit analysis in computerized adaptive testing
using tests for a change point. \emph{Journal of Educational and Behavioral
Statistics}, \emph{41}(5), 521--549.

Tu, D., Li, Y., & Cai, Y. (2023). A new perspective on detecting performance
decline: A change-point analysis based on Jensen-Shannon divergence.
\emph{Behavior Research Methods}, \emph{55}(3), 963--980.
}
