#' @title Models statistics
#'
#' @description
#' The \code{stats} function calculates models statistics. Read more \link[Metrics]{auc}
#'
#' @aliases stats
#'
#' @param modelA Model generated by \link[ai]{model} function
#' @param modelB Model generated by \link[ai]{model} function
#'
#' @return modified model list contains additional statistics
#'
#' @examples
#' \donttest{
#'
#' model_data <- data.frame(a = c(1,2,3,4,5,6),
#'                          b = c(1,2,3,4,5,6),
#'                          s = c(1,2,3,4,5,6))
#'
#' model_data <- prodata(model_data, status_colname = "s")
#'
#' config <- config(formula = "a ~ b + s")
#'
#' model <- model(model_data, config)
#'
#' stats(model)
#'
#'}
#'
#' @export

stats <- function(modelA, modelB =  NULL){

  return()

}

#' stats_compare_models()
#'
#' @param modelA modelA
#' @param modelB modelB
#'
#' @return data.frame contains comparison of both models statistics
#'
#' @export

stats_compare_models <- function(modelA, modelB){

  # Collect model types
  models_namesA <- names(modelA)
  models_namesB <- names(modelB)

  # Are models the same in both ?
  if (unique(models_namesA %in% models_namesB) == FALSE) {stop("Model A and B contain uncomparable types of models. Check the same types of models are in both , A and B models.")}

  # results
  res <- data.frame(matrix(nrow = 0, ncol = 5))

  for (i in 1:length(models_namesA)){

      # Collect stats names
      stats <- modelA[[models_namesA[i]]]$stats
      stats_namses <- names(stats)

      for (a in 1:length(stats_namses)){

         # Stat names
         stat_name <- stats_namses[a]

         # Stat values
         statA <- modelA[[models_namesA[i]]]$stats[[a]]
         statB <- modelB[[models_namesB[i]]]$stats[[a]]

         # Who is better?
         if (statA > statB){

            # Higher is better
            if (stat_name %in% ('auc')){ better <- "A"}

            # Lower is better
            if (stat_name %in% ('rmse')){ better <- "B"}
         }
         # When Equal
         if (statA == statB){ better <- 'eq' }

         res <- rbind(res, data.frame(
                                       modelType =models_namesA[i],
                                       statName = stat_name,
                                       statA = statA,
                                       statB = statB,
                                       better = better
                                      ))
         }
    }

  return(res)

}


#' stats_model()
#'
#' @param model model
#'
#' @return list contains model statistics
#'
#' @export

stats_model <- function(model){

    # Collect model types
    models_names <- names(model)

    for (i in 1:length(models_names)){

      # Predicted
      predicted <- as.numeric(model[[models_names[i]]]$predicted)

      # Observed
      expected <- as.numeric(model[[models_names[i]]]$expected)

      # Calc stats

                stats_temp <- list(
                                    # accuracy = Metrics::accuracy(expected, predicted),
                                    # ae = Metrics::ae(expected, predicted),
                                    # ape = Metrics::ape(expected, predicted),
                                    # apk1 = Metrics::apk(k = 1, expected, predicted),
                                    auc = Metrics::auc(expected, predicted),
                                    # bias = Metrics::bias(expected, predicted),
                                    # ce = Metrics::ce(expected, predicted),
                                    # f1 = Metrics::f1(expected, predicted),
                                    # fbeta1_score = Metrics::fbeta_score(expected, predicted, beta = 1),
                                    # ll = Metrics::ll(expected, predicted),
                                    # logLoss = Metrics::logLoss(expected, predicted),
                                    # mae = Metrics::mae(expected, predicted),
                                    # mape = Metrics::mape(expected, predicted)
                                    # mapk1 = Metrics::mapk(k=1, expected, predicted),
                                    # mase = Metrics::mase(expected, predicted),
                                    # mdae = Metrics::mdae(expected, predicted),
                                    # mse = Metrics::mse(expected, predicted),
                                    # msle = Metrics::msle(expected, predicted),
                                    rmse = Metrics::rmse(expected, predicted)
                                    # rmsle = Metrics::rmsle(expected, predicted),
                                    # rrse = Metrics::rrse(expected, predicted)
                                    # rse = Metrics::rse(expected, predicted),
                                    # se = Metrics::se(expected, predicted)
                                    # sle = Metrics::sle(expected, predicted),
                                    # smape = Metrics::smape(expected, predicted),
                                    # see = Metrics::sse(expected, predicted)
                                  )

        model[[models_names[i]]] <- append( model[[models_names[i]]],
                                          list(stats = stats_temp)
                                          )
    }

  return(model)

}
