#' Portal Item Types
#'
#' Every portal item has an associated item type. Each of those item types have keywords which cna be used to help narrow down search further.
#'
#' @references [REST API Documentation](https://developers.arcgis.com/rest/users-groups-and-items/items-and-item-types)
#' @export
#' @name portal_types
#' @param item_type a scalar character of the item type. See [`portal_item_types()`] for valid item types.
item_type <- S7::new_class(
  "ItemType",
  package = "arcgisutils",
  properties = list(item_type = S7::class_character),
  validator = function(self) {
    check_string(self@item_type, allow_empty = FALSE)

    if (!tolower(self@item_type) %in% tolower(portal_item_types())) {
      cli::cli_abort(
        "{.arg item_type} is not a valid item type. See {.fn portal_item_types}"
      )
    }
  }
)

#' @export
#' @name portal_types
#' @param keyword a scalar character of the item type keyword. See [`portal_item_keywords()`].
item_keyword <- S7::new_class(
  "ItemKeyword",
  package = "arcgisutils",
  properties = list(keyword = S7::class_character),
  validator = function(self) {
    check_string(self@keyword, allow_empty = FALSE)

    if (!tolower(self@keyword) %in% tolower(portal_item_keywords())) {
      cli::cli_abort(
        c(
          "{.arg keyword} is not a valid keyword.",
          "i" = "See {.url https://developers.arcgis.com/rest/users-groups-and-items/items-and-item-types}"
        )
      )
    }
  }
)

#' @export
#' @name portal_types
portal_item_keywords <- function() {
  c(
    ".NET-Windows Desktop",
    "2D",
    "360",
    "3D",
    "3dd",
    "3DObject",
    "3DTiles",
    "3DTiles Package",
    "3DTiles Service",
    "Add In",
    "Add-in",
    "Administrative Report",
    "Analysis",
    "App",
    "App Builder",
    "AppBuilder",
    "Application",
    "ArcGIS",
    "ArcGIS Desktop",
    "ArcGIS Earth",
    "ArcGIS Explorer",
    "ArcGIS Image Server",
    "ArcGIS Knowledge",
    "ArcGIS Mission Management",
    "ArcGIS Online",
    "ArcGIS Pro",
    "ArcGIS Pro Add In",
    "ArcGIS Professional",
    "ArcGIS Project Template",
    "ArcGIS Server",
    "ArcGIS Workflow Manager",
    "ArcGlobe",
    "ArcMap",
    "ArcPad",
    "ArcPad Project",
    "ArcPad Template",
    "ArcReader",
    "ArcScene",
    "Basemap",
    "BIM",
    "bpk",
    "Building",
    "Business Analyst",
    "C",
    "C#",
    "C++",
    "CAD",
    "City",
    "Classification",
    "Classifier",
    "Classify Feature",
    "Classify Pixel",
    "Code",
    "Code Sample",
    "Collector",
    "Compact Tile Package",
    "Compare",
    "Configurable",
    "Configuration",
    "Connection",
    "Coordinates Type",
    "CRS",
    "CSV",
    "Cube Map",
    "Dashboard",
    "Data",
    "Data Engineering",
    "Data Engineering Model",
    "Data Engineering Workbook",
    "Data interoperability",
    "Data Pipeline",
    "Data Store",
    "Database",
    "Deep Learning",
    "Deep Learning Package",
    "Deep Learning Studio",
    "Deep Learning Studio Project",
    "Desktop",
    "Desktop Add In",
    "Detect Objects",
    "dlpk",
    "Document",
    "Earth",
    "Earth Configuration",
    "eaz",
    "ecd",
    "epk",
    "Esri Classification Schema",
    "Esri Classifier Definition",
    "esriaddin",
    "esriaddinx",
    "esriTasks",
    "Experience",
    "Experience Builder",
    "Experience Builder Widget",
    "Explorer",
    "Explorer Add In",
    "Explorer Document",
    "Explorer Layer",
    "Explorer Map",
    "Explorer Web Map",
    "Export Package",
    "Feature",
    "Feature Access",
    "Feature Collection",
    "Feature Service",
    "Feature Service Template",
    "FeatureCollection",
    "File",
    "Flex",
    "Form",
    "Function Template",
    "Functions",
    "gcpk",
    "GeoBIM",
    "Geocoding",
    "Geodata Service",
    "Geodatabase",
    "Geoenrichment",
    "Geometry",
    "GeometryCollection",
    "GeoPackage",
    "Geoprocessing",
    "Geoprocessing Package",
    "Globe Document",
    "gpk",
    "gpkg",
    "gpkx",
    "Graph",
    "Hosted Service",
    "HTML3D/HTML",
    "Hub",
    "Hub Event",
    "Hub Initiative Template",
    "Hub Project",
    "Image",
    "Image Collection",
    "Image Service",
    "Imagery",
    "Inference",
    "Insights",
    "Insights Data Engineering",
    "Insights Data Engineering Model",
    "Insights Data Engineering Workbook",
    "Insights Model",
    "Insights Page",
    "Insights Theme",
    "Insights Workbook",
    "Insights Workbook Package",
    "Installer",
    "IntegratedMesh",
    "Investigation",
    "iPhone",
    "Java",
    "JavaScript",
    "KML",
    "Knowledge",
    "Knowledge Studio",
    "Knowledge Studio Project",
    "Label",
    "Layer",
    "Layer File",
    "Layer Package",
    "Layout",
    "LayoutFile",
    "Library",
    "Link Chart",
    "Locator",
    "Locator Package",
    "lpk",
    "lyr",
    "lyrx",
    "Mac",
    "map",
    "Map",
    "Map Area",
    "Map Document",
    "Map File",
    "Map Notes Template",
    "Map Package",
    "Map Service",
    "Mapping Site",
    "mapx",
    "Microsoft Excel",
    "Microsoft Powerpoint",
    "Microsoft Word",
    "Mission Command",
    "mmpk",
    "Mobile",
    "Mobile Base Package",
    "Mobile Map Package",
    "Mobile Scene Package",
    "model",
    "Model",
    "mpk",
    "msd",
    "mspk",
    "mxd",
    "Native",
    "nmc",
    "nmf",
    "Notebook",
    "Object Detection",
    "Offline",
    "OGC",
    "OGC Feature Service",
    "OIC",
    "Online Map",
    "OpenData",
    "Operations Dashboard",
    "Oriented Imagery Catalog",
    "Ortho",
    "Ortho Maker",
    "Ortho Mapping Project",
    "Ortho Mapping Template",
    "Orthomosaic",
    "Orthophoto",
    "OS X",
    "Package Widget",
    "pagx",
    "Panorama",
    "Parquet",
    "PDF",
    "Photo",
    "Pixel Classification",
    "Pixel Segmentation",
    "Platform",
    "pmf",
    "Point",
    "PointCloud",
    "Presentation",
    "Presentation File",
    "Pro Presentation File",
    "Pro Report",
    "Pro Report Template",
    "Processing",
    "proConfigX",
    "Project",
    "Project Package",
    "Project Template",
    "Proposals",
    "prsx",
    "Published Map",
    "Python",
    "Qt",
    "Raster",
    "Ready to Use",
    "Ready To Use",
    "Regression",
    "Relational",
    "Report",
    "Report Template",
    "Reporting",
    "Result",
    "rft",
    "Route Layer",
    "rpk",
    "rptt",
    "rptx",
    "Rule Package",
    "Sample",
    "Scenarios",
    "Scene",
    "Scene Document",
    "Scene Package",
    "Scene Service",
    "script",
    "Script",
    "Service",
    "Service Definition",
    "Shapefile",
    "Sharing",
    "Silverlight",
    "slpk",
    "Smart Cities",
    "Snippet",
    "Spherical",
    "spk",
    "Statistical Data Collection",
    "StatusActive",
    "StatusComplete",
    "StatusDraft",
    "StoryMap",
    "StoryMap Theme",
    "Stream Service",
    "Streaming",
    "Street View",
    "Style",
    "Survey123",
    "Survey123 Add In",
    "sxd",
    "Symbology",
    "Symbols",
    "Symbols (either 2d or 3d)",
    "Task Assistant",
    "template",
    "Template",
    "Templates",
    "Text",
    "Tile Package",
    "Tool",
    "Toolbox",
    "tpk",
    "tpkx",
    "Training Sample",
    "Transportation Network",
    "Urban",
    "Urban Project",
    "Vector Tile Package",
    "Vector Tile Service",
    "Virtual Reality",
    "Visio Document",
    "Voxel",
    "VR",
    "VR360",
    "vtpk",
    "Web",
    "Web ADF",
    "Web AppBuilder",
    "Web Application",
    "Web Coverage Service",
    "Web Experience",
    "Web Feature Service",
    "Web Map",
    "Web Map Service",
    "Web Map Tile Service",
    "Web Mapping Application",
    "Web Page",
    "Web Scene",
    "Web Site",
    "Widget",
    "Workbook",
    "WPF",
    "wpk",
    "xForm"
  )
}

#' @export
#' @name portal_types
portal_item_types <- function() {
  sort(c(
    "360 VR Experience",
    "CityEngine Web Scene",
    "Map Area",
    "Pro Map",
    "Web Map",
    "Web Scene",
    "Feature Collection",
    "Feature Collection Template",
    "Feature Service",
    "Geodata Service",
    "Group Layer",
    "Media Layer",
    "Image Service",
    "KML",
    "KML Collection",
    "Map Service",
    "OGCFeatureServer",
    "Oriented Imagery Catalog",
    "Relational Database Connection",
    "3DTilesService",
    "Scene Service",
    "Stream Service",
    "Vector Tile Service",
    "WCS",
    "WFS",
    "WMS",
    "WMTS",
    "Geometry Service",
    "Geocoding Service",
    "Geoprocessing Service",
    "Network Analysis Service",
    "Workflow Manager Service",
    "AppBuilder Extension",
    "AppBuilder Widget Package",
    "Analysis Model",
    "Code Attachment",
    "Dashboard",
    "Data Pipeline",
    "Deep Learning Studio Project",
    "Esri Classification Schema",
    "Excalibur Imagery Project",
    "Experience Builder Widget",
    "Experience Builder Widget Package",
    "Form",
    "GeoBIM Application",
    "GeoBIM Project",
    "Hub Event",
    "Hub Initiative",
    "Hub Initiative Template",
    "Hub Page",
    "Hub Project",
    "Hub Site Application",
    "Insights Workbook",
    "Insights Workbook Package",
    "Insights Model",
    "Insights Page",
    "Insights Theme",
    "Insights Data Engineering Workbook",
    "Insights Data Engineering Model",
    "Investigation",
    "Knowledge Studio Project",
    "Mission",
    "Mobile Application",
    "Notebook",
    "Notebook Code Snippet Library",
    "Native Application",
    "Native Application Installer",
    "Ortho Mapping Project",
    "Ortho Mapping Template",
    "Pro Presentation",
    "Solution",
    "StoryMap",
    "Urban Project",
    "Web AppBuilder Widget",
    "Web Experience",
    "Web Experience Template",
    "Web Mapping Application",
    "Workforce Project",
    "Administrative Report",
    "Apache Parquet",
    "CAD Drawing",
    "Color Set",
    "Content Category Set",
    "CSV",
    "CSV Collection",
    "Document Link",
    "Earth configuration",
    "Esri Classifier Definition",
    "Export Package",
    "File Geodatabase",
    "GeoJson",
    "GeoPackage",
    "GML",
    "Image",
    "iWork Keynote",
    "iWork Numbers",
    "iWork Pages",
    "Microsoft Excel",
    "Microsoft Powerpoint",
    "Microsoft Word",
    "PDF",
    "Report Template",
    "Service Definition",
    "Shapefile",
    "SQLite Geodatabase",
    "Statistical Data Collection",
    "StoryMap Theme",
    "Style",
    "Symbol Set",
    "Visio Document",
    "ArcPad Package",
    "Compact Tile Package",
    "Explorer Map",
    "Globe Document",
    "Layout",
    "Map Document",
    "Map Package",
    "Map Template",
    "Mobile Basemap Package",
    "Mobile Map Package",
    "Mobile Scene Package",
    "Pro Map",
    "Project Package",
    "Project Template",
    "Published Map",
    "Scene Document",
    "Task File",
    "Tile Package",
    "Vector Tile Package",
    "Explorer Layer",
    "Image Collection",
    "Layer",
    "Layer",
    "Layer Package",
    "Pro Report",
    "Scene Package",
    "3DTilesPackage",
    "Desktop Style",
    "ArcGIS Pro Configuration",
    "Deep Learning Package",
    "Geoprocessing Package",
    "Geoprocessing Package (Pro version)",
    "Geoprocessing Sample",
    "Locator Package",
    "Raster function template",
    "Rule Package",
    "Pro Report Template",
    "ArcGIS Pro Add In",
    "Code Sample",
    "Desktop Add In",
    "Desktop Application",
    "Desktop Application Template",
    "Explorer Add In",
    "Survey123 Add In",
    "Workflow Manager Package"
  ))
}

# Item types were fetched from the documentation automatically using:
# library(rvest)
# info <- read_html("https://developers.arcgis.com/rest/users-groups-and-items/items-and-item-types/#web-content") |>
#   html_nodes("table") |>
#   html_table() |>
#   dplyr::bind_rows()
#   dplyr::pull(Type) |>
#   dput()
# type_keywords <- read_html("https://developers.arcgis.com/rest/users-groups-and-items/items-and-item-types/#web-content") |>
#   html_nodes("table tbody tr td:nth-child(3) li") |>
#   html_text() |>
#   unique() |>
#   sort()
