#' 
#' PROBIT statistics function for bootstrap estimation
#'
#' @param x A data frame with **primary sampling unit (PSU)** in column named
#'   `psu` and with data column/s containing the continuous variable/s of
#'   interest with column names corresponding to `params` values
#' @param params A vector of column names corresponding to the continuous
#'   variables of interest contained in `x`
#' @param threshold cut-off value for continuous variable to differentiate
#'   case and non-case
#'
#' @return A numeric vector of the PROBIT estimate of each continuous variable
#'   of interest with length equal to `length(params)`
#'
#' @examples
#' # Example call to bootBW function:
#' sampled_clusters <- boot_bw_sample_clusters(
#'   x = indicatorsCH1, w = boot_bw_weight(villageData)
#' )
#' 
#' boot <- boot_bw_sample_within_clusters(sampled_clusters)
#' 
#' bootPROBIT(x = boot,
#'            params = "muac1",
#'            threshold = 115)
#'
#' @export
#'

bootPROBIT <- function(x, params, threshold = THRESHOLD) {
  ## Check params ----
  params <- check_params(x = x, params = params)

  ## Get data
  d <- x[[params[1]]]

  ## Shift data to the left to avoid "commutation instability" when : ----
  ##   max(x) / min(x)
  ## is small (i.e. close to unity).
  shift <- min(min(d, na.rm = TRUE), threshold) - 1
  d <- d - shift
  threshold <- threshold - shift

  ## Box-cox transformation ----
  lambda <- car::powerTransform(d)$lambda
  d <- car::bcPower(d, lambda)
  threshold <- car::bcPower(threshold, lambda)
  m <- mean(d, na.rm = TRUE)
  s <- stats::sd(d, na.rm = TRUE)

  ## PROBIT estimate ----
  x <- stats::pnorm(q = threshold, mean = m, sd = s)
  names(x) <- NULL

  ## Return x ----
  x
}
