% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Dorfman.Functions.R
\name{opt.info.dorf}
\alias{opt.info.dorf}
\title{Find the characteristics of an informative two-stage 
hierarchical (Dorfman) algorithm}
\usage{
opt.info.dorf(
  prob,
  se = 1,
  sp = 1,
  method = "OD",
  max.pool = 15,
  thresh.pool = 8,
  threshold = NULL
)
}
\arguments{
\item{prob}{a vector of all subjects' infection probabilities.}

\item{se}{the sensitivity of the diagnostic test.}

\item{sp}{the specificity of the diagnostic test.}

\item{method}{character string defining the specific screening
procedure for implementation of Dorfman retesting in a 
heterogeneous population. Options include Optimal Dorfman 
("\kbd{OD}"), Thresholded Optimal Dorfman ("\kbd{TOD}"), and
Pool-Specific Optimal Dorfman ("\kbd{PSOD}"). Further details 
are given under 'Details'.}

\item{max.pool}{the maximum allowable pool size. Further details 
are given under 'Details'.}

\item{thresh.pool}{the initial pool size used for TOD, if 
\kbd{threshold} is not specified. Further details are given 
under 'Details'.}

\item{threshold}{the threshold value for TOD. If a threshold
value is not specified, one is found algorithmically. Further 
details are given under 'Details'.}
}
\value{
A list containing:
\item{tv}{the threshold value used for TOD, if applicable.}
\item{e}{the expected expenditure of the decoding process.}
\item{v}{the variance of the expenditure of the decoding 
process.}
\item{summary}{a matrix of summary measures that includes
each individual's infection probability, pool (pool to which
they belong), pooling sensitivity, pooling specificity, 
pooling positive predictive value, and pooling negative
predictive value.}
}
\description{
Find the characteristics of an informative 
two-stage hierarchical (Dorfman) decoding process using Optimal 
Dorfman (OD), Thresholded Optimal Dorfman (TOD), or Pool-Specific 
Optimal Dorfman (PSOD) algorithms.
}
\details{
This function finds the characteristics of an informative
two-stage hierarchical (Dorfman) decoding process. Characteristics
found include the expected expenditure of the decoding process, 
the variance of the expenditure of the decoding process, and the 
pooling sensitivity, pooling specificity, pooling positive predictive
value, and pooling negative predictive value for each individual.
Calculations of these characteristics are done using equations
presented in McMahan et al. (2012). 

Optimal Dorfman (OD) is an informative Dorfman algorithm in 
which the common pool size \eqn{c=c_{opt}} minimizes 
\eqn{E(T^(c))}, the expected number of tests needed to decode 
all \eqn{N} individuals when pools of size \eqn{c} are used. 

Thresholded Optimal Dorfman (TOD) is an informative Dorfman 
algorithm in which all \eqn{N} individuals are partitioned 
into two classes, low-risk and high-risk individuals, based 
on whether their risk probability falls below or above a 
particular threshold value. The threshold can be specified 
using the \kbd{threshold} argument or the TOD algorithm can 
identify the optimal threshold value. The low-risk individuals
are tested using a optimal common pool size,  and high-risk 
individuals are tested individually.

Pool-Specific Optimal Dorfman (PSOD) is an informative Dorfman
algorithm in which optimal sizes are determined for each pool. 
A total of \eqn{N} individuals are tested in pools that minimize
the expected number of tests per individual, on a pool-by-pool 
basis. If desired, the user can add the constraint of a maximum 
allowable pool size, so that each pool will contain no more 
than the maximum allowable number of individuals.

All three informative Dorfman procedures described above require
individuals to be ordered from smallest to largest probability 
of infection. See McMahan et al. (2012) for additional details
on the implementation of informative two-stage hierarchical
(Dorfman) testing algorithms.
}
\examples{
# Find the characteristics of an informative
#   Dorfman algorithm, using the OD procedure.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
opt.info.dorf(prob=rbeta(1000,1,10), se=1, sp=1, 
method ="OD", max.pool=15, thresh.pool=8, threshold=NULL)

# Find the characteristics of an informative 
#   Dorfman algorithm, using the TOD procedure.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
set.seed(1002)
p.vec <- p.vec.func(p=0.01, alpha=2, grp.sz=20)
opt.info.dorf(prob=p.vec, se=0.95, sp=0.95, 
method="TOD", max.pool=5, threshold=0.015)
}
\references{
\insertRef{Dorfman1943}{binGroup}

\insertRef{McMahan2012a}{binGroup}
}
\seealso{
\url{http://chrisbilder.com/grouptesting/}

Other Informative Dorfman functions: 
\code{\link{accuracy.dorf}()},
\code{\link{characteristics.pool}()},
\code{\link{inf.dorf.measures}()},
\code{\link{opt.pool.size}()},
\code{\link{pool.specific.dorf}()},
\code{\link{thresh.val.dorf}()}
}
\author{
This function was originally written by Christopher S. 
McMahan for McMahan et al. (2012). The function was obtained 
from \url{http://chrisbilder.com/grouptesting/}.
}
\concept{Informative Dorfman functions}
