% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/univariate_mean_online.R
\name{online.univar.multi}
\alias{online.univar.multi}
\title{Online change point detection with potentially multiple change points.}
\usage{
online.univar.multi(
  y_vec,
  b_vec = NULL,
  train_vec = NULL,
  alpha = NULL,
  gamma = NULL,
  permu_num = NULL
)
}
\arguments{
\item{y_vec}{A \code{numeric} vector of observations.}

\item{b_vec}{A \code{numeric} vector of thresholds b_t with t >= 2.}

\item{train_vec}{A \code{numeric} vector of training data from a pre-change distribution (no change point), which is only needed to when b_vec is NULL in order to calibrate b_t.}

\item{alpha}{A \code{numeric} scalar of desired false alarm rate.}

\item{gamma}{An \code{integer} scalar of desired average run length.}

\item{permu_num}{An \code{integer} scalar of number of random permutation for calibration.}
}
\value{
An \code{integer} vector of estimated change points.
}
\description{
Perform Online change point detection with potentially multiple change points.
}
\details{
#' @title Online change point detection with controlled average run length.
#' @description  Perform online change point detection via CUSUM (single change point, type 2).
#' @param y_vec       A \code{numeric} vector of observations.
#' @param gamma       A \code{integer} scalar of interval length (>= 2).
#' @param tau_gamma   A \code{numeric} scalar of threshold.
#' @param ...         Additional arguments.
#' @return  An \code{integer} scalar of estimated change points location.
#' @export
#' @author Haotian Xu
#' @examples
#' TO DO
online.univar.one2 = function(y_vec, gamma, tau_gamma, ...){
  t = 1
  FLAG = 0
  while(FLAG == 0 & t <= length(y_vec)){
    t = t + 1
    e = max(t-gamma, 0)
    cusum_vec = sapply((e+1):(t-1), function(s) sqrt((t-s)*(s-e)/(t-e)) * abs(mean(y_vec[(e+1):s]) - mean(y_vec[(s+1):t])))
    FLAG = 1 - prod(cusum_vec <= tau_gamma)
  }
  return(t)
}



#' @title Online change point detection via CUSUM (single change point, type 3).
#' @description Perform online change point detection via CUSUM (single change point, type 3).
#' @param y_vec       A \code{numeric} vector of observations.
#' @param tau_vec     A \code{numeric} vector of thresholds at time t>= 1.
#' @param ...         Additional arguments.
#' @return  An \code{integer} scalar of estimated change point location.
#' @export
#' @author Haotian Xu
#' @examples
#' TO DO
online.univar.one3 = function(y_vec, tau_vec, ...){
  if(length(y_vec) != length(tau_vec)){
    stop("y_vec and tau_vec should have the same length.")
  }
  t = 1
  FLAG = 0
  while(FLAG == 0 & t <= length(y_vec)){
    t = t + 1
    J = floor(log2(t))
    j = 0
    while(j < J & FLAG == 0){
      j = j + 1
      s_j = t - 2^(j-1)
      cusum = sqrt((t-s_j)*s_j/t) * abs(mean(y_vec[1:s_j]) - mean(y_vec[(s_j+1):t]))
      FLAG = (cusum > tau_vec[t])
    }
  }
  return(t)
}
}
\examples{
y_vec = rnorm(200) + c(rep(0, 50), rep(1, 100), rep(0, 50))
train_vec = rnorm(150)
# control the false alarm rate
temp1 = online.univar.multi(y_vec = y_vec, train_vec = train_vec, alpha = 0.05, permu_num = 20)
temp1
}
\references{
Yu, Padilla, Wang and Rinaldo (2020) <arxiv:2006.03283>
}
\author{
Haotian Xu
}
